/*
  ______                              _
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2017 Semtech
 
Description: Main program
 
*/
 
#include "mbed.h"
#include "radio.h"
#include "sx1280-hal.h"
 
 
/*!
 * \brief Defines the nominal frequency
 */
#define RF_FREQUENCY                                2400000000UL // Hz
 
/*!
 * \brief Defines the output power in dBm
 *
 * \remark The range of the output power is [-18..+13] dBm
 */
#define TX_OUTPUT_POWER                             13
 
/*!
 * \brief Defines the buffer size, i.e. the payload size
 */
#define BUFFER_SIZE                                 255
 
/*!
 * \brief The size of the buffer
 */
//uint8_t BufferSize = 253;
uint8_t BufferSize = 12;
 
/*!
 * \brief The buffer
 */
uint8_t Buffer[BUFFER_SIZE];
 
int8_t RssiValue = 0;
int8_t SnrValue = 0;
 
/*!
 * \brief Function to be executed on Radio Tx Done event
 */
void OnTxDone( void );
 
/*!
 * \brief Function to be executed on Radio Rx Done event
 */
void OnRxDone( void );
 
/*!
 * \brief Function executed on Radio Tx Timeout event
 */
void OnTxTimeout( void );
 
/*!
 * \brief Function executed on Radio Rx Timeout event
 */
void OnRxTimeout( void );
 
/*!
 * \brief Function executed on Radio Rx Error event
 */
void OnRxError( IrqErrorCode_t );
 
/*!
 * \brief All the callbacks are stored in a structure
 */
RadioCallbacks_t callbacks =
{
    &OnTxDone,        // txDone
    &OnRxDone,        // rxDone
    NULL,             // syncWordDone
    NULL,             // headerDone
    &OnTxTimeout,     // txTimeout
    &OnRxTimeout,     // rxTimeout
    &OnRxError,       // rxError
    NULL,             // rangingDone
    NULL,             // cadDone
};
 
// mosi, miso, sclk, nss, busy, dio1, dio2, dio3, rst, callbacks...
SX1280Hal Radio( D11, D12, D13, D7, D3, D5, NC, NC, A0, &callbacks );
 
DigitalOut ANT_SW( A3 );
DigitalOut TxLed( A4 );
DigitalOut RxLed( A5 );
 
/*!
 * \brief Define IO for Unused Pin
 */
DigitalOut F_CS( D6 );      // MBED description of pin
DigitalOut SD_CS( D8 );     // MBED description of pin
 
/*!
 * \brief Number of tick size steps for tx timeout
 */
#define TX_TIMEOUT_VALUE                            100 // ms
 
/*!
 * \brief Number of tick size steps for rx timeout
 */
#define RX_TIMEOUT_VALUE                            100 // ms
 
/*!
 * \brief Size of ticks (used for Tx and Rx timeout)
 */
#define RX_TIMEOUT_TICK_SIZE                        RADIO_TICK_SIZE_1000_US
 
/*!
 * \brief Mask of IRQs to listen to in rx mode
 */
uint16_t RxIrqMask = IRQ_RX_DONE | IRQ_RX_TX_TIMEOUT;
 
/*!
 * \brief Mask of IRQs to listen to in tx mode
 */
uint16_t TxIrqMask = IRQ_TX_DONE | IRQ_RX_TX_TIMEOUT;
 
/*!
 * \brief Locals parameters and status for radio API
 * NEED TO BE OPTIMIZED, COPY OF STUCTURE ALREADY EXISTING
 */
PacketParams_t PacketParams;
PacketStatus_t PacketStatus;

#if 1
    const uint8_t nb_freq = 4;
    uint32_t tx_freq[nb_freq] = {
        2400000000U,
        2406000000U,
        2412000000U,
        2418000000U
    };
#endif
#if 0
    const uint8_t nb_freq = 1;
    uint32_t tx_freq[nb_freq] = {
        2400000000U
    };
#endif
#if 0
    const uint8_t nb_freq = 2;
    uint32_t tx_freq[nb_freq] = {
        2400000000U,
        2412000000U
    };
#endif

/*!
 * \brief Specify serial datarate for UART debug output
 */
void baud( int baudrate )
{
    Serial s( USBTX, USBRX );
 
    s.baud( baudrate );
}
 
int main( )
{
    uint32_t cnt;
    ModulationParams_t modulationParams;
 
    baud( 115200 );
 
    F_CS   = 1;
    SD_CS  = 1;
    RxLed  = 1;
    TxLed  = 1;
    ANT_SW = 1;
 
    wait_ms( 500 ); // wait for on board DC/DC start-up time
 
    Radio.Init( );
    Radio.SetRegulatorMode( USE_DCDC ); // Can also be set in LDO mode but consume more power
 
    memset( &Buffer, 0x00, BufferSize );
 
    printf( "\nsx1280 TX test (fw:0x%x)\n\r", Radio.GetFirmwareVersion( ) );
    modulationParams.PacketType                  = PACKET_TYPE_LORA;
    modulationParams.Params.LoRa.SpreadingFactor = LORA_SF10;
    modulationParams.Params.LoRa.Bandwidth       = LORA_BW_1600;
    modulationParams.Params.LoRa.CodingRate      = LORA_CR_4_5;
 
    PacketParams.PacketType                 = PACKET_TYPE_LORA;
    PacketParams.Params.LoRa.PreambleLength = 0x08;
    PacketParams.Params.LoRa.HeaderType     = LORA_PACKET_VARIABLE_LENGTH;
    PacketParams.Params.LoRa.PayloadLength  = BufferSize;
    PacketParams.Params.LoRa.Crc            = LORA_CRC_ON;
    PacketParams.Params.LoRa.InvertIQ       = LORA_IQ_NORMAL;
 
    Radio.SetStandby( STDBY_RC );
    Radio.SetPacketType( modulationParams.PacketType );
    Radio.SetModulationParams( &modulationParams );
    Radio.SetPacketParams( &PacketParams );
 
    Radio.SetRfFrequency( RF_FREQUENCY );
    Radio.SetBufferBaseAddresses( 0x00, 0x00 );
    Radio.SetTxParams( TX_OUTPUT_POWER, RADIO_RAMP_20_US );
 
    RxLed = 0;
    TxLed = 0;
 
    cnt = 0;
    //while( cnt < 1000 )
    //while( cnt < 10 )
    while( 1 )
    {
        Buffer[0] = 0; /* ID */
        Buffer[1] = (uint8_t)(cnt >> 24);
        Buffer[2] = (uint8_t)(cnt >> 16);
        Buffer[3] = (uint8_t)(cnt >> 8);
        Buffer[4] = (uint8_t)(cnt >> 0);
        Radio.SetRfFrequency( tx_freq[cnt % nb_freq] );
        Radio.SetDioIrqParams( TxIrqMask, TxIrqMask, IRQ_RADIO_NONE, IRQ_RADIO_NONE );
        TxLed = 1;
        Radio.SendPayload( Buffer, BufferSize, ( TickTime_t ){ RX_TIMEOUT_TICK_SIZE, TX_TIMEOUT_VALUE } );
#if 0
        wait_us( 2500 );
#else
        wait_ms( 1000 );
#endif
        TxLed = 0;
        cnt += 1;
    }
}
 
void OnTxDone( void )
{
}
 
void OnRxDone( void )
{
    RssiValue = PacketStatus.LoRa.RssiPkt;
    SnrValue = PacketStatus.LoRa.SnrPkt;
    printf("rssi: %d; snr: %d\n\r", RssiValue, SnrValue );
}
 
void OnTxTimeout( void )
{
    printf( "<>>>>>>>>TXE\r\n" );
}
 
void OnRxTimeout( void )
{
}
 
void OnRxError( IrqErrorCode_t errorCode )
{
    printf( "RXE<>>>>>>>>\r\n" );
}
 
void OnRangingDone( IrqRangingCode_t val )
{
}
 
void OnCadDone( bool channelActivityDetected )
{
}
