#include "mbed.h"

#include "sx1272-hal.h"
#include "radio.h"
#include "main.h"

#define TX_LOOP 1
#define PAYLOAD_SIZE_LONG 112 //97
#define PAYLOAD_SIZE_SHORT 12

uint32_t Channels[10] = {
    867100000,
    867300000,
    867500000,
    867700000,
    867900000,
    868100000,
    868300000,
    868500000,
    868300000,
    868800000
};

typedef enum Modulation
{
    MOD_LORA,
    MOD_FSK
} Modulation_t;

typedef enum LoRaBandwidth
{
    LORA_BW_125K,
    LORA_BW_250K,
    LORA_BW_500K,
    LORA_BW_NA
} LoRaBandwidth_t;

/*!
 * Serial communication for debug logs
 */
Serial pc(USBTX, USBRX); // tx, rx

/*!
 * Interrupt handler for nucleo user button
 */
InterruptIn DatarateButton( USER_BUTTON );

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;
 
/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

static uint8_t LoRaWANBuffer[255];
static uint16_t FCnt = 0;

/*
 *  ISR variables declarations
 */
static volatile bool SingleDatarate = true;
static volatile uint8_t CurrentDatarate = 7;
static volatile bool ButtonPressed = false;

/* -------------- */

void UserButtonPressed( void )
{
    if( ButtonPressed == true )
    {
        return;
    }

    if( SingleDatarate == false )
    {
        // Increase datarate by 1
        CurrentDatarate = (CurrentDatarate == 7) ? 12 : CurrentDatarate-1;
    }
   
    ButtonPressed = true;
}

/* -------------- */

uint32_t SendPacket( uint32_t freq_hz, RadioModems_t modulation, LoRaBandwidth_t bw, uint8_t pkt_size )
{
    uint32_t TimeOnAir;

    Radio.SetChannel( freq_hz );

    switch( modulation )
    {
        case MODEM_LORA:
            Radio.SetTxConfig( MODEM_LORA, 0, 0, bw, CurrentDatarate, 1, 8, false, true, 0, 0, false, 10e3 );
            TimeOnAir = Radio.TimeOnAir( MODEM_LORA, pkt_size );
            break;
        case MODEM_FSK:
            Radio.SetTxConfig( MODEM_FSK, 0, 25e3, 0, 50e3, 0, 5, false, true, 0, 0, false, 3e3 );
            TimeOnAir = Radio.TimeOnAir( MODEM_FSK, pkt_size );
            break;
        default:
            break;
    }
    
    Radio.Send( LoRaWANBuffer, pkt_size );
    
    return TimeOnAir;
}

/* -------------- */

int main() {
    Timer t;
    double PktToA;
    int i;

    pc.printf( "\n > Initializing... < \n" );
    
    // Get USER button pressed
    DatarateButton.fall( &UserButtonPressed );
    
    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );
 
    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  )
    {
        pc.printf( "Radio could not be detected!\n", NULL );
        wait( 1 );
    }
 
    pc.printf( "\n > Board Type: SX1272MB2xAS < \n" );
    
    Radio.SetPublicNetwork( true );
    
    while( 1 )
    {
        // Check if there is a packet to be sent
        if( ButtonPressed == true )
        {
            // Set packet payload
            LoRaWANBuffer[0] = 0x40; // Unconfirmed Data Up
            //LoRaWANBuffer[0] = 0x80; // Confirmed Data Up
            LoRaWANBuffer[1] = 0x03; // DevAddr
            LoRaWANBuffer[2] = 0x00;
            LoRaWANBuffer[3] = 0xFE;
            LoRaWANBuffer[4] = 0xCA;
            LoRaWANBuffer[5] = 0x0; // FCtrl
            LoRaWANBuffer[6] = 0; // FCnt, filled later
            LoRaWANBuffer[7] = 0; // FCnt, filled later
            LoRaWANBuffer[8] = 1; // FPort
            LoRaWANBuffer[9] = 0x0; // FRMPayload

#if 1
            FCnt = 0;
            while( 1 )
            {
                LoRaWANBuffer[6] = FCnt;
                LoRaWANBuffer[7] = FCnt >> 8;
             
                // Send LoRa packet
                PktToA = SendPacket( Channels[FCnt % 8], MODEM_LORA, LORA_BW_125K, PAYLOAD_SIZE_SHORT );
                //pc.printf( "(%d) Sending LoRa packet: Freq=%u, SF%u (%.1lfms), FCnt=%u...\n", i, TxFreq, CurrentDatarate, PktToA, FCnt );
                wait_ms( PktToA );
                Radio.Sleep( );
                
                FCnt += 1;   
            }
#endif

#if 0            
            FCnt = 0;
            for( i = 0; i < (int)TX_LOOP; i++ )
            {
                LoRaWANBuffer[6] = FCnt;
                LoRaWANBuffer[7] = FCnt >> 8;
                
                // Send LoRa packet
                PktToA = SendPacket( Channels[0], MODEM_LORA, LORA_BW_125K, PAYLOAD_SIZE_LONG );
                //pc.printf( "(%d) Sending LoRa packet: Freq=%u, SF%u (%.1lfms), FCnt=%u...\n", i, TxFreq, CurrentDatarate, PktToA, FCnt );
                wait_ms( PktToA );
                Radio.Sleep( );
                
                FCnt += 1;
                LoRaWANBuffer[6] = FCnt;
                LoRaWANBuffer[7] = FCnt >> 8;
                
                // Send FSK packet
                PktToA = SendPacket( Channels[0], MODEM_FSK, LORA_BW_NA, PAYLOAD_SIZE_SHORT );
                //pc.printf( "(%d) Sending FSK packet: Freq=%u, FCnt=%u...\n", i, TxFreq, PktToA, FCnt );
                wait_ms( PktToA );
                Radio.Sleep( );

                FCnt += 1;
                LoRaWANBuffer[6] = FCnt;
                LoRaWANBuffer[7] = FCnt >> 8;
                
                // Send LoRa packet
                PktToA = SendPacket( Channels[8], MODEM_LORA, LORA_BW_500K, PAYLOAD_SIZE_SHORT );
                //pc.printf( "(%d) Sending LoRa packet: Freq=%u, SF%u (%.1lfms), FCnt=%u...\n", i, TxFreq, CurrentDatarate, PktToA, FCnt );
                wait_ms( PktToA );
                Radio.Sleep( );
                
                FCnt += 1;
            }
#endif

            // Stop sending
            ButtonPressed = false;
        }
        
        // Receive packets
        // TODO
    }
}

void OnTxDone( void )
{
    //Radio.Sleep( );
    //pc.printf( "> OnTxDone\n\r" );
}
 
void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr )
{
    Radio.Sleep( );
    pc.printf( "> OnRxDone\n\r" );
}
 
void OnTxTimeout( void )
{
    Radio.Sleep( );
    pc.printf( "> OnTxTimeout\n\r" );
}
 
void OnRxTimeout( void )
{
    Radio.Sleep( );
    pc.printf( "> OnRxTimeout\n\r" );
}
 
void OnRxError( void )
{
    Radio.Sleep( );
    pc.printf( "> OnRxError\n\r" );
}