/**************************************************************************/
/*!

    @file          main.cpp
    @author        Adafruit Industries
    @modified_by   Ben Zhang <benzh@eecs.berkeley.edu>
    @license       BSD (see license.txt)
    
    This example will wait for any ISO14443A card or tag, and
    depending on the size of the UID will attempt to read from it.
   
    If the card has a 4-byte UID it is probably a Mifare
    Classic card, and the following steps are taken:
   
    - Authenticate block 4 (the first block of Sector 1) using
      the default KEYA of 0XFF 0XFF 0XFF 0XFF 0XFF 0XFF
    - If authentication succeeds, we can then read any of the
      4 blocks in that sector (though only block 4 is read here)
     
    If the card has a 7-byte UID it is probably a Mifare
    Ultralight card, and the 4 byte pages can be read directly.
    Page 4 is read by default since this is the first 'general-
    purpose' page on the tags.


This is an example sketch for the Adafruit PN532 NFC/RFID breakout boards
This library works with the Adafruit NFC breakout 
  ----> https://www.adafruit.com/products/364
 
Check out the links above for our tutorials and wiring diagrams 
These chips use SPI to communicate, 4 required to interface

Adafruit invests time and resources providing this open source code, 
please support Adafruit and open-source hardware by purchasing 
products from Adafruit!

*/
/**************************************************************************/

#include "mbed.h"
#include "Adafruit_PN532.h"
#include "TSISensor.h"
 


#define SS   PTD0
// PTD1 is also LED_BLUE, it will blink during SPI communication.
#define SCK  PTD1
#define MOSI PTD2
#define MISO PTD3

DigitalOut red(LED_RED);
DigitalOut green(LED_GREEN);
DigitalOut blue(LED_BLUE);

Serial pc(USBTX, USBRX);
Adafruit_PN532 nfc(SCK, MISO, MOSI, SS);
float readVal = 0;
float lastTouch = 0;
void loop(void);
TSISensor tsi;
PwmOut led(LED_GREEN);

int main() {
  pc.printf("Hello!\n");  
  // By default, no color
  green = 1; red = 1, blue = 1;
 
  
  
    
//  while (true) {
//        float test=  tsi.readPercentage();
//        pc.printf("%f",test);
//        wait(0.1);
//    }
  nfc.begin();

  uint32_t versiondata = nfc.getFirmwareVersion();
  if (! versiondata) {
    pc.printf("Didn't find PN53x board");
    while (1); // halt
  }
  // Got ok data, print it out!
  pc.printf("Found chip PN5%2X with Firmware ver. %d.%d\n",
            versiondata >> 24,
            (versiondata >> 16) & 0xFF,
            (versiondata >> 8) & 0xFF);
  
  // configure board to read RFID tags
  nfc.SAMConfig();
  
  pc.printf("Waiting for an ISO14443A Card ...\n");
  
  while(1) { loop(); }
}


void loop(void) {
  // Turn back to no color 
  green = 1; red = 1, blue = 1;
  float tsiVal = tsi.readPercentage();
  if (tsiVal > 0) {
    lastTouch = tsiVal;
  }
  uint8_t success;
  uint8_t uid[] = { 0, 0, 0, 0, 0, 0, 0 };  // Buffer to store the returned UID
  uint8_t uidLength;                        // Length of the UID (4 or 7 bytes depending on ISO14443A card type)
  
  // Wait for an ISO14443A type cards (Mifare, etc.).  When one is found
  // 'uid' will be populated with the UID, and uidLength will indicate
  // if the uid is 4 bytes (Mifare Classic) or 7 bytes (Mifare Ultralight)
  success = nfc.readPassiveTargetID(PN532_MIFARE_ISO14443A, uid, &uidLength);
  
  if (success) {
    // Turn off to indicate Card swipe
    green = 1; red = 1, blue = 1;
    // Display some basic information about the card
    pc.printf("\n\nFound an ISO14443A card\n");
    pc.printf("  UID Length: %d bytes", uidLength);
    pc.printf("  UID Value: ");
    nfc.PrintHex(uid, uidLength);
    pc.printf("\n");
    // We are not actually writing to the card. We just want to know the ID.
    // Read value associated with ID to a database once we have an SD card...
    led = 1.0 - readVal;
    // Would write to database with ID card and lastTouch
    readVal = lastTouch;
    
    }
    
}