#include "mbed.h"
#include "max2769ssp0.h"


extern "C" void MYSSP0_IRQHandler(void) __irq;

// CONSTRUCTOR
MAX2769SSP0::MAX2769SSP0(PinName _mosi=p11, PinName _sck=p13, PinName _ssel=p14) {
    /* 
     * DEFAULT PORT IS SSP0, others are not supported
     */    
    if ( (_mosi  == p11) && (_sck == p13) && (_ssel == p14) ) {
        sppx = 0;
        // Turn on SSP0 periferal
        LPC_SC->PCONP |= (0x01UL << 21);
        // Turn on the SSP0 clock
        LPC_SC->PCLKSEL1 |= (0x01U << 10);  // Turn on peripheral clock
        // DIP11 is P0.18
        LPC_PINCON->PINSEL1  &= ~(3UL <<  4); // Reset P0.18
        LPC_PINCON->PINSEL1  |=  (2UL <<  4); // Set P0.18 to MOSI0
        LPC_PINCON->PINMODE1 &= ~(3UL <<  4); // 
        LPC_PINCON->PINMODE1 |=  (2UL <<  4); // Set P0.18 with no pull-up or pull-down
        // DIP12 is P0.17
        LPC_PINCON->PINSEL1  &= ~(3UL <<  2); // Reset P0.17 
        LPC_PINCON->PINSEL1  |=  (2UL <<  2); // Set P0.17 to MISO0
        LPC_PINCON->PINMODE1 &= ~(3UL <<  2); // 
        LPC_PINCON->PINMODE1 |=  (2UL <<  2); // Set P0.17 with no pull-up or pull-down
        // DIP13 is P0.15
        LPC_PINCON->PINSEL0  &= ~(3UL << 30); // Reset P0.15
        LPC_PINCON->PINSEL0  |=  (2UL << 30); // Set P0.15 to SCK0
        LPC_PINCON->PINMODE0 &= ~(3UL << 30); // 
        LPC_PINCON->PINMODE0 |=  (2UL << 30); // Set P0.15 with no pull-up or pull-down
        // DIP14 is P0.16
        LPC_PINCON->PINSEL1  &= ~(3UL <<  0); // Reset P0.16
        LPC_PINCON->PINSEL1  |=  (2UL <<  0); // Set P0.16 to SSEL0
        LPC_PINCON->PINMODE1 &= ~(3UL <<  0); // 
        LPC_PINCON->PINMODE1 |=  (2UL <<  0); // Set P0.16 with no pull-up or pull-down
        
        // Configure interrupts for SSP0
        LPC_SSP0->IMSC = 
            (1UL << 0) | // Receive overrun
            (1UL << 1) | // Receive timeout
            (0UL << 2) | // RX FIFO half full
            (0UL << 3);  // TX FIFO half empty
        NVIC_SetVector(SSP0_IRQn, (uint32_t) MYSSP0_IRQHandler);
        NVIC_EnableIRQ(SSP0_IRQn);
        
        // Configure SSP0
        LPC_SSP0->CR0 = 
            ( 0xFUL << 0) |   // 16 bits transfers
            (   1UL << 4) |   // Texas mode
            (0x00UL << 8);    // Serial clock rate
        // Not really relevant since we are in slave mode
        LPC_SSP0->CPSR = 0;    
        
        LPC_SSP0->CR1 = 
            ( 0UL << 0) |   // Loopback mode disabled
            ( 0UL << 1) |   // SSP controller disabled
            ( 1UL << 2) |   // Slave mode!
            ( 1UL << 3);    // Slave output is disabled

        LPC_SSP0->DMACR = 
            ( 0UL << 0) |   // DMA disabled on RX path
            ( 0UL << 1);    // DMA disabled on TX path
            
    }  else {
        sppx = -1;
    }
}


// DISTRUCTOR
MAX2769SSP0::~MAX2769SSP0() {
    // do nothing
}

//
void MAX2769SSP0::Read(uint16_t *_puiBuff, uint32_t _iLocations) {
    int k;
    
    // Sanity check, port is not valid
    if (sppx < 0) {
        error("%s; this SSP port is not valid\n", __FUNCTION__);
    }
    
    // Sanity check, buffer pointer is not valid
    if (_puiBuff == NULL) {
        error("%s; buffer pointer %p is not valid\n", __FUNCTION__, _puiBuff);
    }
    
    //LPC_SSP0->DR = 0x5555;
    if (LPC_SSP0->SR & 0x01) {
        fprintf(stderr, "SSP0: TX FIFO empty\n");
    } else {
        if (LPC_SSP0->SR & 0x02) {
            fprintf(stderr, "SSP0: TX FIFO not full\n");
        } else {
            fprintf(stderr, "SSP0: TX FIFO full\n");
        }
    }
    if (LPC_SSP0->SR & 0x04) {
        if (LPC_SSP0->SR & 0x08) {
            fprintf(stderr, "SSP0: RX FIFO full\n");
        } else {
            fprintf(stderr, "SSP0: RX FIFO not full\n");
        }
    } else {
        fprintf(stderr, "SSP0: RX FIFO empty\n");
    }
    if (LPC_SSP0->SR & 0x10) {
        fprintf(stderr, "SSP0: BUSY\n");
    } else {
        fprintf(stderr, "SSP0: IDLE\n");
    }
    
    k=0;
    while (k < _iLocations) {
        if (LPC_SSP0->SR & 0x4) { // if the RX FIFO is not empty
            _puiBuff[k++] = (uint16_t) (LPC_SSP0->DR & 0xFFFF);
        }
    }  
}


// MYSSP0_IRQHandler
extern "C" void MYSSP0_IRQHandler(void) __irq {
    if (LPC_SSP0->MIS & 0x1UL) {
        myled1 = 1;
        LPC_SSP0->ICR &= ~0x1UL;
    }
    if (LPC_SSP0->MIS & 0x2UL) {
        myled2 = 1;
        LPC_SSP0->ICR &= ~0x2UL;
    }
}
