/*
 * driverAtmelRFInterface.c
 *
 *  Created on: 14 July 2014
 *      Author: mBed Team
 */

#include "arm_hal_interrupt.h"
#include "arm_hal_phy.h"
#include "driverRFPhy.h"
#include "driverAtmelRFInterface.h"
#include "low_level_RF.h"

#include <stdio.h>

void (*app_rf_settings_cb)(void) = 0;
static uint8_t rf_part_num = 0;
static uint8_t rf_rx_lqi;
static int8_t rf_rx_rssi;
/*TODO: RSSI Base value setting*/
static int8_t rf_rssi_base_val = -91;
static uint8_t phy_timers_enabled = 0;

/*
 * \brief Delay function for RF interface.
 *
 * \param ticks Number of delay ticks
 *
 * \return none
 */
void rf_if_delay_function(uint16_t ticks)
{
    while(ticks--);
}

/*
 * \brief Function initialises the RF timer for ACK wait and calibration.
 *
 * \param none
 *
 * \return none
 */
void rf_if_timer_init(void)
{
    /*  TODO   */
}

/*
 * \brief Function starts the ACK wait timeout.
 *
 * \param slots Given slots, resolution 50us
 *
 * \return none
 */
void rf_if_ack_wait_timer_start(uint16_t slots)
{
    /*  TODO   */
}

/*
 * \brief Function starts the calibration interval.
 *
 * \param slots Given slots, resolution 50us
 *
 * \return none
 */
void rf_if_calibration_timer_start(uint32_t slots)
{
    /*  TODO   */
}

/*
 * \brief Function stops the ACK wait timeout.
 *
 * \param none
 *
 * \return none
 */
void rf_if_ack_wait_timer_stop(void)
{
    phy_timers_enabled &= ~PHY_ACK_WAIT_TIMER;
}

/*
 * \brief Function is a call back for ACK wait timeout.
 *
 * \param none
 *
 * \return none
 */
void rf_if_ack_wait_timer_interrupt(void)
{
    /*  TODO   */
}

/*
 * \brief Function is a call back for calibration interval timer.
 *
 * \param none
 *
 * \return none
 */
void rf_if_calibration_timer_interrupt(void)
{
    /*  TODO   */
}

/*
 * \brief Function sets SLP_TR pin high in RF interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_slp_tr_pin_high(void)
{
      RF_SLP_TR_Set(1);
}

/*
 * \brief Function sets SLP_TR pin low in RF interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_slp_tr_pin_low(void)
{
    RF_SLP_TR_Set(0);   
}

/*
 * \brief Function returns peripheral ID for SPI interface.
 *
 * \param none
 *
 * \return peripheral ID
 */
uint8_t rf_if_get_rf_spi_periph(uint8_t spi_interface)
{
    uint8_t ret_val = 19 + spi_interface;
    return ret_val;
}

/*
 * \brief Function returns interrupt number for I/O port.
 *
 * \param none
 *
 * \return interrupt number
 */
uint8_t rf_if_get_rf_irq_number(uint8_t port)
{
    /*  not needed in mBed  */
    return 0x00;
}


/*
 * \brief Function enables RF irq pin interrupts in RF interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_irq(void)
{
    /*  not needed in mBed  */
}


/*
 * \brief Function initialises RF interrupt pin in RF interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_init_irq_delete(void)
{
    RF_IRQ_Init();  
}

/*
 * \brief Function initialises the SPI interface for RF.
 *
 * \param none
 *
 * \return none
 */
void rf_if_spi_init(void)
{
    /*  not needed in mBed  */
}

/*
 * \brief Function initialises SLP_TR pin in RF interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_init_slp_tr_pin(void)
{
    /*Chip select*/
      //RF_SLP_TR_Set(0);       // moved to reset function
}

/*
 * \brief Function returns peripheral ID for I/O port.
 *
 * \param none
 *
 * \return peripheral ID
 */
uint8_t rf_if_get_port_peripheral_id(uint8_t port)
{
    uint8_t ret_val = 9 + port;
    return ret_val;
}

/*
 * \brief Function initialises I/O pins for RF interface.
 *
 * \param none
 *
 * \return peripheral ID
 */
void rf_if_init_spi_pins(void)
{
    /*  not needed in mBed  */
}

/*
 * \brief Function reads data from the given RF SRAM address.
 *
 * \param ptr Read pointer
 * \param sram_address Read address in SRAM
 * \param len Length of the read
 *
 * \return none
 */
void rf_if_read_payload(uint8_t *ptr, uint8_t sram_address, uint8_t len)
{
    uint8_t i;

      RF_CS_Set(0);
    spi_exchange(0x20);
    spi_exchange(sram_address);
    for(i=0; i<len; i++)
        *ptr++ = spi_exchange(0);

    /*Read LQI and RSSI in variable*/
    rf_rx_lqi = spi_exchange(0);
    rf_rx_rssi = (rf_rssi_base_val + spi_exchange(0));
      RF_CS_Set(1);
}

/*
 * \brief Function polls while the SPI chip select is active.
 *
 * \param none
 *
 * \return none
 */
void rf_if_spi_poll_chip_select(void)
{
    RF_CS_while_active();   
}

void rf_if_set_bit(uint8_t addr, uint8_t bit, uint8_t bit_mask)
{
    uint8_t reg = rf_if_read_register(addr);
    reg &= ~bit_mask;
    reg |= bit;
    rf_if_write_register(addr, reg);
}

/*
 * \brief Function clears bit(s) in given RF register.
 *
 * \param addr Address of the register to clear
 * \param bit Bit(s) to clear
 *
 * \return none
 */
void rf_if_clear_bit(uint8_t addr, uint8_t bit)
{
    uint8_t reg = rf_if_read_register(addr);
    reg &= ~bit;
    rf_if_write_register(addr, reg);
}

/*
 * \brief Function writes register in RF.
 *
 * \param addr Address on the RF
 * \param data Written data
 *
 * \return none
 */
void rf_if_write_register(uint8_t addr, uint8_t data)
{
    uint8_t cmd = 0xC0;
    arm_enter_critical();

      spi_write(cmd | addr, data);

    arm_exit_critical();
}

/*
 * \brief Function reads RF register.
 *
 * \param addr Address on the RF
 *
 * \return Read data
 */
uint8_t rf_if_read_register(uint8_t addr)
{
    uint8_t cmd = 0x80;
    uint8_t data;
    arm_enter_critical();

      data = spi_read(cmd | addr);
    arm_exit_critical();
    return data;
}

/*
 * \brief Function resets the RF.
 *
 * \param none
 *
 * \return none
 */
void rf_if_reset_radio(void)
{    
    /* Reset and init RF_CS - chip select  */
      RF_RST_Set(1);
    rf_if_delay_function(65000);
    RF_RST_Set(0);
    rf_if_delay_function(65000);
      /* Set default states */
      RF_CS_Set(1);
      RF_SLP_TR_Set(0);
    rf_if_delay_function(65000);
    RF_RST_Set(1);
    rf_if_delay_function(65000);    

      /*Initialise RF interrupt pin*/
    RF_IRQ_Init();
}

/*
 * \brief Function enables the Front end usage.
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_pa_ext(void)
{
    /*Set PA_EXT_EN to enable controlling of external front end*/
    rf_if_set_bit(TRX_CTRL_1, PA_EXT_EN, PA_EXT_EN);
}

/*
 * \brief Function disables the Front end usage.
 *
 * \param none
 *
 * \return none
 */
void rf_if_disable_pa_ext(void)
{
    /*Clear PA_EXT_EN to disable controlling of external front end*/
    rf_if_clear_bit(TRX_CTRL_1, PA_EXT_EN);
}

/*
 * \brief Function enables the Antenna diversity usage.
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_ant_div(void)
{
    /*Set ANT_EXT_SW_EN to enable controlling of antenna diversity*/
    rf_if_set_bit(ANT_DIV, ANT_EXT_SW_EN, ANT_EXT_SW_EN);
}

/*
 * \brief Function disables the Antenna diversity usage.
 *
 * \param none
 *
 * \return none
 */
void rf_if_disable_ant_div(void)
{
    rf_if_clear_bit(ANT_DIV, ANT_EXT_SW_EN);
}

/*
 * \brief Function sets the SLP TR pin.
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_slptr(void)
{
      RF_SLP_TR_Set(1);
}

/*
 * \brief Function clears the SLP TR pin.
 *
 * \param none
 *
 * \return none
 */
void rf_if_disable_slptr(void)
{
      RF_SLP_TR_Set(0);
}

/*
 * \brief Function writes the antenna diversity settings.
 *
 * \param none
 *
 * \return none
 */
void rf_if_write_antenna_diversity_settings(void)
{
    /*Recommended setting of PDT_THRES is 3 when antenna diversity is used*/
    rf_if_set_bit(RX_CTRL, 0x03, 0x0f);
    rf_if_write_register(ANT_DIV, ANT_DIV_EN | ANT_EXT_SW_EN | ANT_CTRL_DEFAULT);
}

/*
 * \brief Function writes the TX output power register.
 *
 * \param value Given register value
 *
 * \return none
 */
void rf_if_write_set_tx_power_register(uint8_t value)
{
    rf_if_write_register(PHY_TX_PWR, value);
}

/*
 * \brief Function writes the RPC register.
 *
 * \param value Given register value
 *
 * \return none
 */
void rf_if_write_set_trx_rpc_register(uint8_t value)
{
    rf_if_write_register(TRX_RPC, value);
}

/*
 * \brief Function returns the RF part number.
 *
 * \param none
 *
 * \return part number
 */
uint8_t rf_if_read_part_num_delete(void)
{
    return rf_if_read_register(PART_NUM);
}

/*
 * \brief Function writes the RF settings and initialises SPI interface.
 *
 * \param none
 *
 * \return none
 */
void rf_if_write_rf_settings(void)
{    
    /* unslotted mode - max. frame & csma retries = 0 */
    rf_if_write_register(XAH_CTRL_0,0);
    if (rf_if_read_register(XAH_CTRL_0) != 0) {
      printf("Error: XAH_CTRL_0 reg. incorrect!\r\n"); 
        //while(1);
    }    
    
    /* TX_AUTO_CRC On - IRQ_Mask_Mode = 0 - IRQ_Polarity = 0 (Pin IRQ is active high) */
    rf_if_write_register(TRX_CTRL_1, 0x20);
    if (rf_if_read_register(TRX_CTRL_1) != 0x20) {
      printf("Error: TRX_CTRL_1 reg. incorrect!\r\n"); 
        //while(1);
    }
    
  /*CCA Mode - Carrier sense OR energy above threshold. Channel list is set separately*/
    rf_if_write_register(PHY_CC_CCA, 0x05);
    if (rf_if_read_register(PHY_CC_CCA) != 0x05) {
      printf("Error: PHY_CC_CCA reg. incorrect!\r\n"); 
        //while(1);
    }  
    
  /*Read transceiver PART_NUM*/
  rf_part_num = rf_if_read_register(PART_NUM);

  /*Sub-GHz RF settings*/
  if(rf_part_num == PART_AT86RF212)
  {
      /*GC_TX_OFFS mode-dependent setting - OQPSK*/
      rf_if_write_register(RF_CTRL_0, 0x32);
    
          if(rf_if_read_register(VERSION_NUM) == VERSION_AT86RF212B)
      {
          /*TX Output Power setting - 0 dBm North American Band*/
          rf_if_write_register(PHY_TX_PWR, 0x03);
      }
      else
      {
          /*TX Output Power setting - 0 dBm North American Band*/
          rf_if_write_register(PHY_TX_PWR, 0x24);
      }

      /*PHY Mode: IEEE 802.15.4-2006/2011 - OQPSK-SIN-250*/
      rf_if_write_register(TRX_CTRL_2, OQPSK_SIN_250);
      rf_rssi_base_val = -98;
    }
    /*2.4GHz RF settings*/
    else if (rf_part_num == PART_AT86RF233)
    {
      printf("Part detected: ATMEL AT86RF233\r\n");    
       /*PHY Mode: IEEE 802.15.4 - Data Rate 250 kb/s*/
       rf_if_write_register(TRX_CTRL_2, 0);
       rf_rssi_base_val = -91;
    } else {
        // other module not yet defined
        printf("Error: RF Part Unknown!\r\n");             
    }
}


/*
 * \brief Function checks the channel availability
 *
 * \param none
 *
 * \return 1 Channel clear
 * \return 0 Channel not clear
 */
uint8_t rf_if_check_cca(void)
{
    uint8_t retval = 0;
    if(rf_if_read_register(TRX_STATUS) & CCA_STATUS)
    {
        retval = 1;
    }
    return retval;
}

/*
 * \brief Function checks if the CRC is valid in received frame
 *
 * \param none
 *
 * \return 1 CRC ok
 * \return 0 CRC failed
 */
uint8_t rf_if_check_crc(void)
{
    uint8_t retval = 0;
    if(rf_if_read_register(PHY_RSSI) & CRC_VALID)
    {
        retval = 1;
    }
    return retval;
}

/*
 * \brief Function returns the RF state
 *
 * \param none
 *
 * \return RF state
 */
uint8_t rf_if_read_trx_state(void)
{
    return rf_if_read_register(TRX_STATUS) & 0x1F;
}

/*
 * \brief Function reads data from RF SRAM.
 *
 * \param ptr Read pointer
 * \param len Length of the read
 *
 * \return none
 */
void rf_if_read_packet(uint8_t *ptr, uint8_t len)
{
    if(rf_part_num == PART_AT86RF231 || rf_part_num == PART_AT86RF212)
        rf_if_read_payload(ptr, 0, len);
    else if(rf_part_num == PART_AT86RF233)
        rf_if_read_payload(ptr, 1, len);
}

/*
 * \brief Function writes RF short address registers
 *
 * \param short_address Given short address
 *
 * \return none
 */
void rf_if_write_short_addr_registers(uint8_t *short_address)
{
    rf_if_write_register(SHORT_ADDR_1, *short_address++);
    rf_if_write_register(SHORT_ADDR_0, *short_address);
}

/*
 * \brief Function sets the frame pending in ACK message
 *
 * \param state Given frame pending state
 *
 * \return none
 */
void rf_if_ack_pending_ctrl(uint8_t state)
{
    arm_enter_critical();
    if(state)
    {
        rf_if_set_bit(CSMA_SEED_1, (1 << AACK_SET_PD), (1 << AACK_SET_PD));
    }
    else
    {
        rf_if_clear_bit(CSMA_SEED_1, (1 << AACK_SET_PD));
    }
    arm_exit_critical();
}

/*
 * \brief Function returns the state of frame pending control
 *
 * \param none
 *
 * \return Frame pending state
 */
uint8_t rf_if_last_acked_pending(void)
{
    uint8_t last_acked_data_pending;

    if(rf_if_read_register(CSMA_SEED_1) & 0x20)
        last_acked_data_pending = 1;
    else
        last_acked_data_pending = 0;

    return last_acked_data_pending;
}

/*
 * \brief Function calibrates the RF part.
 *
 * \param none
 *
 * \return none
 */
void rf_if_calibration(void)
{
    rf_if_set_bit(FTN_CTRL, FTN_START, FTN_START);
    /*Wait while calibration is running*/
    while(rf_if_read_register(FTN_CTRL) & FTN_START);
}

/*
 * \brief Function writes RF PAN Id registers
 *
 * \param pan_id Given PAN Id
 *
 * \return none
 */
void rf_if_write_pan_id_registers(uint8_t *pan_id)
{
    rf_if_write_register(PAN_ID_1, *pan_id++);
    rf_if_write_register(PAN_ID_0, *pan_id);
}

/*
 * \brief Function writes RF IEEE Address registers
 *
 * \param address Given IEEE Address
 *
 * \return none
 */
void rf_if_write_ieee_addr_registers(uint8_t *address)
{
    uint8_t i;
    uint8_t temp = IEEE_ADDR_0;

    for(i=0; i<8; i++)
        rf_if_write_register(temp++, address[7-i]);
}

/*
 * \brief Function writes data in RF frame buffer.
 *
 * \param ptr Pointer to data
 * \param length Pointer to length
 *
 * \return none
 */
void rf_if_write_frame_buffer(uint8_t *ptr, uint8_t length)
{
    uint8_t i;
    uint8_t cmd = 0x60;

      RF_CS_Set(0);
    spi_exchange(cmd);
    spi_exchange(length + 2);
    for(i=0; i<length; i++)
        spi_exchange(*ptr++);

    rf_if_delay_function(10);   
      RF_CS_Set(1);
}

/*
 * \brief Function returns 8-bit random value.
 *
 * \param none
 *
 * \return random value
 */
uint8_t rf_if_read_rnd(void)
{
    uint8_t temp;

    temp = ((rf_if_read_register(PHY_RSSI)>>5) << 6);
    temp |= ((rf_if_read_register(PHY_RSSI)>>5) << 4);
    temp |= ((rf_if_read_register(PHY_RSSI)>>5) << 2);
    temp |= ((rf_if_read_register(PHY_RSSI)>>5));
    return temp;
}

/*
 * \brief Function changes the state of the RF.
 *
 * \param trx_state Given RF state
 *
 * \return none
 */
void rf_if_change_trx_state(rf_trx_states_t trx_state)
{
    arm_enter_critical();
    rf_if_write_register(TRX_STATE, trx_state);
    /*Wait while not in desired state*/
    rf_poll_trx_state_change(trx_state);
    arm_exit_critical();
}

/*
 * \brief Function enables the TX END interrupt
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_tx_end_interrupt(void)
{
    rf_if_set_bit(IRQ_MASK, TRX_END, 0x08);
}

/*
 * \brief Function enables the RX END interrupt
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_rx_end_interrupt(void)
{
    rf_if_set_bit(IRQ_MASK, TRX_END, 0x08);
}

/*
 * \brief Function enables the RX START interrupt
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_rx_start_interrupt(void)
{
    rf_if_set_bit(IRQ_MASK, RX_START, 0x04);
}

/*
 * \brief Function enables the CCA ED interrupt
 *
 * \param none
 *
 * \return none
 */
void rf_if_enable_cca_ed_done_interrupt(void)
{
    rf_if_set_bit(IRQ_MASK, CCA_ED_DONE, 0x10);
}

/*
 * \brief Function starts the CCA process
 *
 * \param none
 *
 * \return none
 */
void rf_if_start_cca_process(void)
{
    rf_if_set_bit(PHY_CC_CCA, CCA_REQUEST, 0x80);
}

/*
 * \brief Function returns the length of the received packet
 *
 * \param none
 *
 * \return packet length
 */
uint8_t rf_if_read_received_frame_length(void)
{
    uint8_t length;

      RF_CS_Set(0);
    spi_exchange(0x20);
    length = spi_exchange(0);
      RF_CS_Set(1);
    return length;
}

/*
 * \brief Function returns the LQI of the received packet
 *
 * \param none
 *
 * \return packet LQI
 */
uint8_t rf_if_read_lqi(void)
{
    return rf_rx_lqi;
}

/*
 * \brief Function returns the RSSI of the received packet
 *
 * \param none
 *
 * \return packet RSSI
 */
int8_t rf_if_read_rssi(void)
{
    return rf_rx_rssi;
}

/*
 * \brief Function sets the RF channel field
 *
 * \param Given channel
 *
 * \return none
 */
void rf_if_set_channel_register(uint8_t channel)
{
    rf_if_set_bit(PHY_CC_CCA, channel, 0x1f);
}

/*
 * \brief Function returns the pointer to RF interrupt handler
 *
 * \param none
 *
 * \return RF interrupt handler function
 */
void (*rf_if_get_rf_interrupt_function())(void)
{
    return rf_if_interrupt_handler;
}

/*
 * \brief Function is a RF interrupt vector. End of frame in RX and TX are handled here as well as CCA process interrupt.
 *
 * \param none
 *
 * \return none
 */
void rf_if_interrupt_handler(void)
{
    uint8_t irq_status;

    /*Read interrupt flag*/
    irq_status = rf_if_read_register(IRQ_STATUS);

    /*Disable interrupt on RF*/
    rf_if_clear_bit(IRQ_MASK, irq_status);
    /*RX start interrupt*/
    if(irq_status & RX_START)
    {
    }
    /*Address matching interrupt*/
    if(irq_status & AMI)
    {
    }
    if(irq_status & TRX_UR)
    {
    }
    /*Frame end interrupt (RX and TX)*/
    if(irq_status & TRX_END)
    {
        /*TX done interrupt*/
        if(rf_if_read_trx_state() == PLL_ON || rf_if_read_trx_state() == TX_ARET_ON)
        {
            rf_handle_tx_end();
        }
        /*Frame received interrupt*/
        else
        {
            rf_handle_rx_end();
        }
    }
    if(irq_status & CCA_ED_DONE)
    {
        rf_handle_cca_ed_done();
    }
}

