#include "configuration.h"
#include "mbedConnectorInterface.h"
#include "ns_6lowpan_support.h"
#include "nsdl_support.h"
#include "nsdl_utils.h"
#include "debug.h"

// we have to redefine DBG as its used differently here...
#ifdef DBG
    #undef DBG
    #define DBG  printf
#endif

// externally defined configurable...
extern uint8_t app_MAC_address[8];
extern uint32_t channel_list;

// Globals
static int8_t rf_phy_device_register_id = -1;
static int8_t net_6lowpan_id = -1;
int8_t coap_udp_socket = -1;
static int access_point_status = 0;
uint8_t app_defined_stack_heap[APP_DEV_HEAP_SIZE];

/** Used for Receive Data source Address store*/
static  ns_address_t app_src;
static sn_nsdl_addr_s sn_addr_s;

extern void app_heap_error_handler(heap_fail_t event);

link_layer_setups_s app_link_info;
link_layer_address_s app_link_address_info;
network_layer_address_s app_nd_address_info;

// initialize the 6LowPAN network
void init_network(bool canActAsRouterNode) {
    /*
     * Currently based on non-Thread 6LowPAN stack
     */
    DBG("\r\ninit_network: Initializing network stack heap and MAC address...\r\n");
    ns_dyn_mem_init(app_defined_stack_heap, APP_DEV_HEAP_SIZE, app_heap_error_handler,0);
    rf_set_mac_address(app_MAC_address);
      
    // init RF interface
    DBG("init_network: Initializing the RF Interface...\r\n");
    rf_phy_device_register_id = rf_device_register();
    randLIB_seed_random();
    
    //Init network core
    DBG("init_network: Initializing 6LowPAN network core stack...\r\n");
    net_init_core();
}

void NSDL_receive_socket(void * cb)
{
  socket_callback_t * cb_res =0;
  int16_t length;
  cb_res = (socket_callback_t *) cb;
  uint8_t *payload;

  if(cb_res->event_type == SOCKET_DATA)
  {
    DBG("LINK LQI:");
    debug_hex(cb_res->LINK_LQI);
    create_lqi_resource(cb_res->LINK_LQI);
    DBG("\r\n");

    if ( cb_res->d_len > 0)
    {
      payload = (uint8_t *) own_alloc(cb_res->d_len);
      if(payload)
      {
        //Read data to the RX buffer
        length = socket_read(cb_res->socket_id, &app_src, payload, cb_res->d_len);  //replace rx_buffer payload
        if(length)
        {
          if(cb_res->socket_id == coap_udp_socket)
          {
            // Handles data received in UDP socket                      
            // Call application protocol parser.
            sn_addr_s.type = SN_NSDL_ADDRESS_TYPE_IPV6;
            sn_addr_s.addr_len = 16;
            sn_addr_s.port = app_src.identifier;
            sn_addr_s.addr_ptr = app_src.address;
            printf("Data 1\r\n");   
            if(sn_nsdl_process_coap(payload, length, &sn_addr_s))  // 0= ok, -1=failure
            {
              DBG("Error processing CoAP packet\r\n");
            }
            printf("Data 4\r\n");   
          }
        }
        own_free(payload);
      }
    }
  }
#if 1       // enabled for debug
  else if(cb_res->event_type == SOCKET_TX_DONE)
  {
    //DBG("*");
  }
  else if(cb_res->event_type == SOCKET_NO_ROUTE)
  {
    DBG("SOCKET_NO_ROUTE\r\n");
  }
  else if(cb_res->event_type == SOCKET_TX_FAIL)
  {
    DBG("SOCKET_TX_FAIL\r\n");
  }
#endif   
}


/**
  * \brief Network state event handler.
  * \param event show network start response or current network state.
  *
  * - NET_READY: Save NVK peristant data to NVM and Net role
  * - NET_NO_BEACON: Link Layer Active Scan Fail, Stack is Already at Idle state
  * - NET_NO_ND_ROUTER: No ND Router at current Channel Stack is Already at Idle state
  * - NET_BORDER_ROUTER_LOST: Connection to Access point is lost wait for Scan Result
  * - NET_PARENT_POLL_FAIL: Host should run net start without any PAN-id filter and all channels
  * - NET_PANA_SERVER_AUTH_FAIL: Pana Authentication fail, Stack is Already at Idle state
  */
void app_parse_network_event(arm_event_s *event )
{
  arm_nwk_interface_status_type_e status = (arm_nwk_interface_status_type_e)event->event_data;
  switch (status)
  {
    case ARM_NWK_BOOTSTRAP_READY:
      /* NEtwork is ready and node is connected to Access Point */
      if(access_point_status==0)
      {
        uint8_t temp_ipv6[16];
        DBG("Network Connection Ready\r\n");
        access_point_status=1;
        //Read Address

        if( arm_nwk_nd_address_read(net_6lowpan_id,&app_nd_address_info) != 0)
        {
          DBG("ND Address read fail\r\n");
        }
        else
        {
          DBG("ND Access Point: ");
          printf_ipv6_address(app_nd_address_info.border_router); //REVIEW

          DBG("ND Prefix 64: ");
          printf_array(app_nd_address_info.prefix, 8);            //REVIEW

          if(arm_net_address_get(net_6lowpan_id,ADDR_IPV6_GP,temp_ipv6) == 0)
          {
            DBG("GP IPv6: ");
            printf_ipv6_address(temp_ipv6);         //REVIEW
          }
        }

        if( arm_nwk_mac_address_read(net_6lowpan_id,&app_link_address_info) != 0)
        {
          DBG("MAC Address read fail\r\n");
        }
        else
        {
          uint8_t temp[2];
          common_write_16_bit(app_link_address_info.mac_short,temp);
          DBG("MAC 16-bit: ");
          printf_array(temp, 2);  //REVIEW
          common_write_16_bit(app_link_address_info.PANId,temp);
          DBG("PAN-ID: ");
          printf_array(temp, 2);  //REVIEW
          DBG("MAC 64-bit: ");
          printf_array(app_link_address_info.long_euid64, 8); //REVIEW
          DBG("EUID64(Based on MAC 64-bit address): ");
          printf_array(app_link_address_info.euid64, 8);  //REVIEW
        }
      }
      break;
    
    case ARM_NWK_NWK_SCAN_FAIL:
      /* Link Layer Active Scan Fail, Stack is Already at Idle state */
      DBG("Link Layer Scan Fail: No Beacons\r\n");
      access_point_status=0;
      //dnssd_disable(1);
      break;
    
    case ARM_NWK_IP_ADDRESS_ALLOCATION_FAIL:
      /* No ND Router at current Channel Stack is Already at Idle state */
      DBG("ND Scan/ GP REG fail\r\n");
      access_point_status=0;
      //dnssd_disable(1);
      break;
    
    case ARM_NWK_NWK_CONNECTION_DOWN:
      /* Connection to Access point is lost wait for Scan Result */
      DBG("ND/RPL scan new network\r\n");
      access_point_status=0;
      break;

    case ARM_NWK_NWK_PARENT_POLL_FAIL:
      DBG("Network parent poll failed\r\n");
      access_point_status=0;
      break;
    
    case ARM_NWK_AUHTENTICATION_FAIL:
      DBG("Network authentication fail\r\n");
      access_point_status=0;
      break;
    
    default:
      debug_hex(status);    //REVIEW
      DBG("Unknown event");
      break;
  }
  
  if(access_point_status == 0)
  {
    //Set Timer for new Trig
    timer_sys_event(RETRY_TIMER, 10000);
  }
}


/**
  * \brief Handler for events sent to the application.
  * \param event received event.
  *
  * - EV_NETWORK event, Network Event state event handler
  * - EV_INIT, Set Certificate Chain list, init multicast, Start net start if NVM have session
  * - EV_DEBUG, Terminal handler
  */
void tasklet_main(arm_event_s *event)
{
  if(event->sender == 0)
  {
    arm_library_event_type_e event_type;
    event_type = (arm_library_event_type_e)event->event_type;

    switch(event_type)
    {
      case ARM_LIB_NWK_INTERFACE_EVENT:
        /* Network Event state event handler */
        DBG("Event: ARM_LIB_NWK_INTERFACE\r\n");
        app_parse_network_event(event);                  
        break;

      case ARM_LIB_TASKLET_INIT_EVENT:
        /*Init event from stack at start-up*/
        DBG("Event: ARM_LIB_TASKLET_INIT\r\n");
        multicast_set_parameters(10,0,20,3,75 );

        net_6lowpan_id = arm_nwk_interface_init(NET_INTERFACE_RF_6LOWPAN, rf_phy_device_register_id, "6LoWPAN_BORDER_ROUTER");
        if(net_6lowpan_id < 0) 
        {
          DBG("Interface Generate Fail\r\n");
          while(1);
        } 
        else 
        {
          //SET Bootsrap
          if(arm_nwk_interface_configure_6lowpan_bootstrap_set(net_6lowpan_id, NET_6LOWPAN_HOST, 1) != 0)  // Last parameter enables MLE protocol
          {
            //Bootsrap SET fail
            DBG("Bootstrap Fail\r\n");
            while(1);
          }
          else
          {
            int8_t retval = -1;
            arm_nwk_6lowpan_gp_address_mode(net_6lowpan_id, NET_6LOWPAN_GP16_ADDRESS, NODE_SHORT_ADDRESS, 1);  // 5 = short address for link-layer // 1 = generate automatically if duplicate address is encountered
            arm_nwk_link_layer_security_mode(net_6lowpan_id, NET_SEC_MODE_NO_LINK_SECURITY, 0, 0);
            arm_nwk_6lowpan_link_scan_paramameter_set(rf_phy_device_register_id, channel_list, 5);
            retval = arm_nwk_interface_up(net_6lowpan_id);
            if(retval != 0)
            {
              //6Lowpan Bootsrap start fail
              DBG("6LowPAN Bootstrap start Fail\r\n");
              while(1);
            }
            else
            {
              //6Lowpan Bootsrap start OK
              DBG("6LowPAN Bootstrap Start OK\r\n");
            }  
            // open sockets
            coap_udp_socket = socket_open(SOCKET_UDP, AUDP_SOCKET_PORT, NSDL_receive_socket);
          }
          timer_sys_event(NSP_REGISTRATION_TIMER, NSP_RD_UPDATE_PERIOD);
        }
        break;
                                    
      case ARM_LIB_SYSTEM_TIMER_EVENT:
        timer_sys_event_cancel(event->event_id);
        if (event->event_id == NSP_REGISTRATION_TIMER) 
        {   
          printf("Time to register...\r\n");                                    
          NSP_registration();
          timer_sys_event(NSP_REGISTRATION_TIMER, NSP_RD_UPDATE_PERIOD);                                 
        } 
        else if (event->event_id == RETRY_TIMER) 
        {
          DBG("Event: ARM_LIB_SYSTEM_TIMER (event_id = 1)\r\n");
          int8_t retval = -1;
          retval = arm_nwk_interface_up(net_6lowpan_id);
          if(retval != 0)
          {
            //6Lowpan Bootsrap start fail
            DBG("6LowPAN Bootstrap Start Failure\r\n");
            while(1);
          }
          else
          {
            //6Lowpan Bootsrap start OK
            DBG("6LowPAN Bootstrap Start OK\r\n");
          }             
        }
        break;
                                
      default:
        break;
    }
  }
}

void app_heap_error_handler(heap_fail_t event) {
    switch (event)
    {
        case NS_DYN_MEM_NULL_FREE:
            break;
        case NS_DYN_MEM_DOUBLE_FREE:
            break;
        case NS_DYN_MEM_ALLOCATE_SIZE_NOT_VALID:
            break;
        case NS_DYN_MEM_POINTER_NOT_VALID:
            break;
        case NS_DYN_MEM_HEAP_SECTOR_CORRUPTED:
            break;
        case NS_DYN_MEM_HEAP_SECTOR_UNITIALIZED:
            break;    
        default:
            break;
    }
    while(1);
}


