/**
 * @file    main.cpp
 * @brief   mbed Connected Home Endpoint main entry point
 * @author  Doug Anson
 * @version 1.0
 * @see
 *
 * Copyright (c) 2014
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// mbed Connector Interface (configuration)
#include "mbedConnectorInterface.h"

// mbed Connector Endpoint includes
#include "ConnectorEndpoint.h"
#include "OptionsBuilder.h"

// USB Serial port access for debugging/logging
RawSerial pc(USBTX,USBRX);

// Logging facility
Logger logger(&pc);

// Static Resources
#include "StaticResource.h"
StaticResource mfg(&logger,"3/0/0","ARM-mbed");
StaticResource model(&logger,"3/0/1","mbed-6LoWPAN-demo");

//
// Dynamic Resource Note:
//
//  mbedConnectorInterface supports up to IPT_MAX_ENTRIES 
//  (currently 5) independent dynamic resources.
//
//  You can increase this (at the cost of memory) 
//  in mbedConnectorinterface.h
//

// Light Resource
#include "LightResource.h"
//LightResource light(&logger,"3311/0/5706");

// LED Resource
#include "OnBoardLED.h"
LEDResource led(&logger,"3311/1/5706");

// Temperature Resource
#include "TempResource.h"
//TempResource temp(&logger,"3303/0/5700", true); /* true if observable */

// DHT11 Temperature Resource
#include "DHT11_TempResource.h"
//DHT11_TempResource dht_temp(&logger,"3303/0/5700", false); /* true if observable */

// DHT11 Humidity  Resource
#include "DHT11_HumiResource.h"
//DHT11_HumiResource dht_humi(&logger,"3304/0/5700", true); /* true if observable */

// Illuminance Resource
#include "IlluminanceResource.h"
//IlluminanceResource illum(&logger,"3301/0/5700", true); /* true if observable */

// LedBar Resource
#include "LedBarResource.h"
//LedBarResource ledbar(&logger,"3306/0/5853", false); /* true if observable */

// Button Resource
#include "ButtonResource.h"
ButtonResource button(&logger,"3200/0/5500", true); /* true if observable */

// Buzzer Resource
#include "BuzzerResource.h"
//BuzzerResource buzzer(&logger,"3201/0/5500", false); /* true if observable */

// Loudness Resource
#include "LoudnessResource.h"
LoudnessResource loud(&logger,"3324/0/5700", true); /* true if observable */

// Luminance Resource
#include "LuminanceResource.h"
LuminanceResource lumi(&logger,"3300/0/5700", true); /* true if observable */

// Presence Resource
#include "PresenceResource.h"
PresenceResource pres(&logger,"3302/0/5700", true); /* true if observable */



// Set our own unique endpoint name
#define MY_ENDPOINT_NAME                       "mbed1"

// My NSP Domain
#define MY_NSP_DOMAIN                          "domain"                               

// Customization Example: My custom NSP IPv4 or IPv6 address and NSP CoAP port 
//uint8_t my_nsp_address[NSP_IP_ADDRESS_LENGTH] = {0x20,0x02,0x0d,0xb4,0x00,0x00,0x00,0x00,0x02,0x0c,0x43,0xff,0xfe,0x87,0x81,0x7c}; 
uint8_t my_nsp_address[NSP_IP_ADDRESS_LENGTH] = {0x20,0x02,0x0d,0xb4,0x00,0x00,0x00,0x00,0xba,0x27,0xeb,0xff,0xfe,0xc6,0x35,0x38}; 
int my_nsp_coap_port                          = 5683;

// called from the Endpoint::start() below to create resources and the endpoint internals...
Connector::Options *configure_endpoint(Connector::OptionsBuilder &config)
{
    // Build the endpoint configuration parameters
    logger.log("configure_endpoint: building endpoint configuration...");
//    temp.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
    button.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
//    DHT_temp.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
//    DHT_humi.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
//    illum.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
    lumi.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
    pres.setMaxAge(0); /* MaxAge = 0 to disable caching of the value in the Device Server */
 
    return config.setEndpointNodename(MY_ENDPOINT_NAME)                   // custom endpoint name
                 .setNSPAddress(my_nsp_address)                           // custom NSP address
                 .setDomain(MY_NSP_DOMAIN)                                // custom NSP domain
                 .setNSPPortNumber(my_nsp_coap_port)                      // custom NSP CoAP port
                 
                 // add the static resource representing this endpoint
                 .addResource(&mfg)
                 .addResource(&model)
                                    
                 // Add my specific physical dynamic resources...
//                 .addResource(&light)
//                 .addResource(&temp, 1000)
                 .addResource(&led)
//                 .addResource(&illum, 1000)
                 .addResource(&lumi, 1000)
//                 .addResource(&dht_temp)
//                 .addResource(&dht_humi)
//                 .addResource(&ledbar)
                 .addResource(&button, 1000)
//                 .addResource(&buzzer)
                 .addResource(&loud, 1000)
                 .addResource(&pres, 1000)
                   
                 // finalize the configuration...
                 .build();
}

// main entry point...
int main()
{
    // Announce
    logger.log("\r\n\r\nmbed mDS Sample Endpoint v1.0 (6LoWPAN)");

    // we have to plumb our network first
    Connector::Endpoint::plumbNetwork();
     
    // starts the endpoint by finalizing its configuration (configure_endpoint() above called),creating a Thread and reading NSP events...
    logger.log("Start the endpoint to finish setup and enter the main loop...");
    Connector::Endpoint::start();
}
