#include "mbed.h"
#include "EthernetInterface.h"
#include "nsdl_support.h"
#include "dbg.h"
// Include various resources
#include "barometer.h"
#include "gas_sensor.h"
#include "humidity.h"
#include "illuminance.h"
#include "rain.h"
#include "temperature.h"
#include "wind.h"
#include "wind_gust.h"
#include "wind_direction.h"

Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "10.45.0.206"
#define MASK    "255.255.255.0"
#define GW      "10.45.0.1"

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
static const char* NSP_ADDRESS = "108.201.184.41"; /* smartobjectservice.com */ 
//static const char* NSP_ADDRESS = "192.168.1.220"; /* local NSP*/ 
//static const char* NSP_ADDRESS = "192.168.1.200"; /* local NSP*/ 
static const int NSP_PORT = 5683;
char endpoint_name[32] = "mbed-";
uint8_t ep_type[] = {"mbed_device"};
uint8_t lifetime_ptr[] = {"60"};

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;

static void ethernet_init()
{
    char mbed_uid[33]; // for creating unique name for the board
    char macbytes[6];

    /* Initialize network */
#ifdef DHCP
    NSDL_DEBUG("DHCP in use\r\n");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    NSDL_DEBUG("eth.init done\r\n");
    
    if(eth.connect(30000) == 0)
        pc.printf("Connect OK\n\r");

    mbed_mac_address(macbytes);
    sprintf(mbed_uid,"%02x%02x%02x%02x%02x%02x", macbytes[0], macbytes[1], macbytes[2], macbytes[3], macbytes[4], macbytes[5]); 
    strncat(endpoint_name, mbed_uid, 12);

    NSDL_DEBUG("IP Address:%s ", eth.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

bool UDP_blocking = false;
unsigned int UDP_timeout = 1000;

static void nsp_init()
{
    server.init();
    
    server.bind(NSP_PORT);
    
    server.set_blocking(UDP_blocking, UDP_timeout);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/0")-1, (uint8_t*) "3/0/0", sizeof("Manufacturer")-1, (uint8_t*) "Manufacturer",  (uint8_t*) "mbed_DEMO", sizeof("mbed_DEMO")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/1")-1, (uint8_t*) "3/0/1", sizeof("Model_Number")-1, (uint8_t*) "Model_Number",  (uint8_t*) "WS_1", sizeof("WS_1")-1);

    // Dynamic resources
    create_barometer_resource(resource_ptr);
    create_gas_sensor_resource(resource_ptr);
    create_humidity_resource(resource_ptr);
    create_illuminance_resource(resource_ptr);
    create_rain_resource(resource_ptr);
    create_temperature_resource(resource_ptr);
    create_wind_resource(resource_ptr);
    create_wind_gust_resource(resource_ptr);
    create_wind_direction_resource(resource_ptr);

        /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
        pc.printf("NSP registering failed\r\n");
    else
        pc.printf("NSP registering OK\r\n");
    nsdl_clean_register_endpoint(&endpoint_ptr);
    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{
    NSDL_DEBUG("mbed LWM2M Weather Station Example\n");
    
    // Initialize Ethernet interface first
    ethernet_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
