/***************************************************************************//**
 * @file main.cpp
 * @brief Demo program for the mbed asynchronous serial API.
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2015 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * DISCLAIMER OF WARRANTY/LIMITATION OF REMEDIES: Silicon Labs has no
 * obligation to support this Software. Silicon Labs is providing the
 * Software "AS IS", with no express or implied warranties of any kind,
 * including, but not limited to, any implied warranties of merchantability
 * or fitness for any particular purpose or warranties against infringement
 * of any proprietary rights of a third party.
 *
 * Silicon Labs will not be liable for any consequential, incidental, or
 * special damages, or any other relief, or for any claim by any third party,
 * arising from your use of this Software.
 *
 ******************************************************************************/
 
#include "mbed.h"

/*------------ Constant definitions --------------*/
#define TX_PIN          USBTX
#define RX_PIN          USBRX
#define BRATE           115200
#define LED_PIN         LED0
#define TOGGLE_RATE     (0.5f)
#define BUFF_LENGTH     5

/*-------- Check if platform compatible ----------*/
#if DEVICE_SERIAL_ASYNCH
Serial test_connection(USBTX, USBRX);
#else
#error "Platform not compatible with Low Power APIs for Serial"
#endif

/*------------------ Variables -------------------*/
LowPowerTicker      blinker;
bool                blinking = false;
event_callback_t    serialEventCb;
DigitalOut          LED(LED_PIN);
uint8_t             rx_buf[BUFF_LENGTH + 1];

/*------------------ Callbacks -------------------*/
void blink(void) {
    LED = !LED;
}

/**
* This is a callback! Do not call frequency-dependent operations here.
*
* For a more thorough explanation, go here: 
* https://developer.mbed.org/teams/SiliconLabs/wiki/Using-the-improved-mbed-sleep-API#mixing-sleep-with-synchronous-code
**/
void serialCb(int events) {
    /* Something triggered the callback, either buffer is full or 'S' is received */
    unsigned char i;
    if(events & SERIAL_EVENT_RX_CHARACTER_MATCH) {
        //Received 'S', check for buffer length
        for(i = 0; i < BUFF_LENGTH; i++) {
            //Found the length!
            if(rx_buf[i] == 'S') break;
        }
        
        // Toggle blinking
        if(blinking) {
            blinker.detach();
            blinking = false;
        } else {
            blinker.attach(blink, TOGGLE_RATE);
            blinking = true;
        }
    } else if (events & SERIAL_EVENT_RX_COMPLETE) {
        i = BUFF_LENGTH - 1;
    } else {
        rx_buf[0] = 'E';
        rx_buf[1] = 'R';
        rx_buf[2] = 'R';
        rx_buf[3] = '!';
        rx_buf[4] = 0;
        i = 3;
    }
    
    // Echo string, no callback
    test_connection.write(rx_buf, i+1, 0, 0);
    
    // Reset serial reception
    test_connection.read(rx_buf, BUFF_LENGTH, serialEventCb, SERIAL_EVENT_RX_ALL, 'S');
}

/*-------------------- Main ----------------------*/
int main() {
    /* Very Simple Main (tm) */
    serialEventCb.attach(serialCb);
    
    /* Setup serial connection */
    test_connection.baud(BRATE);
    test_connection.printf("Low Power API test\n\nSend 'S' to toggle blinking\n");
    test_connection.read(rx_buf, BUFF_LENGTH, serialEventCb, SERIAL_EVENT_RX_ALL, 'S');
    
    /* Let the callbacks take care of everything */
    while(1) sleep();
}