#ifndef _MQTT_H_
#define _MQTT_H_

#include "mbed.h"
#include "MQTTNetwork.h"
#include "MQTTClient.h"
#include "MQTTmbed.h"
#include <assert.h>

#define MAX_CARS_ON_ROAD 5

enum drive_state_t{
 NORMAL_STATE = 0,
 STOPPED_STATE,
 CROSSING_STATE
};

typedef struct position_msg{
    int position;
    int speed;
    int car_id;
    int road_id; 
    int counter; // for debugging and ensuring everyone is synchronized
    drive_state_t state; // for debugging
}position_msg_t;


typedef struct control_msg{
    int speed;
    int car_id;
    int road_id;
    int counter; // for debugging synchronization
}control_msg_t;


class mqtt{

private:
    // static functions for control message callbacks
    static void control_message_arrived(MQTT::MessageData& md);
    
    // functions for creatin of mqtt and wifi bring up
    WiFiInterface* setup_wifi();
    int send_position_msg(position_msg_t* msg);
    void manage_network();
    void bringup_network();
    
    // data structures needed for creation of network
    MQTT::Client<MQTTNetwork, Countdown>* client;
    MQTT::Client<MQTTNetwork, Countdown>* setup_mqtt(MQTTNetwork& network); 
    MQTTNetwork* new_network;
    
    // thread that will manage the mqtt network status
    Thread* thread;
    
    // queue for sending position msessages
    Queue<position_msg_t, 16> position_queue;
    
    // queue for control message passsing between mqtt and car threads
    Queue<control_msg_t, 1> control_queue[MAX_CARS_ON_ROAD];

    
public:
    // singleton instance for managing network
    // we can only have one network so this should prohibit accidental
    // creations of multiple objects
    static mqtt* mqtt_singleton;
    
    //
    int mqtt_id;
    
    // empty constructor
    mqtt();
    
    // setup and tear down API's
    void setup_network();
    int shutdown_network();

    
    static mqtt *instance()
    {
        if (!mqtt_singleton)
          mqtt_singleton = new mqtt;
        return mqtt_singleton;
    }
    
    // adding to a queue to be sent on mqtt network
    inline void add_to_position_queue(position_msg_t* msg)
    {
        position_queue.put(msg);
    }
    
    // adding to a queue to be sent on mqtt network
    inline void add_to_control_queue(int car_id,control_msg_t* msg)
    {
        control_queue[car_id].put(msg);
    }
    
    // adding to a queue to be sent on mqtt network
    inline control_msg_t* get_control_msg(int car_id)
    {
        osEvent evt = control_queue[car_id].get();
        assert(evt.status == osEventMessage);
        control_msg_t *message = (control_msg_t*)evt.value.p;
        return message;
    }
    
};


#endif