#include "mbed.h"
#include <cctype>
#include "Car.h"
#include "TextLCD.h"
#include "signal_wrapper.h"

Serial pc(USBTX, USBRX);    // The connection to USB
TextLCD lcd(p15, p16, p17, p18, p19, p20, TextLCD::LCD16x2 ); // rs, e, d4-d7

// define any needed global variables
// ...
// ...
#define MAX_CAR_SPEED 15
#define MIN_CAR_SPEED 5

// This function present the "prompt" string to the user before accepting
// an integer back from the Serial connection. If the user enters anything
// other than a digit (e.g. return key, letters), the currently built number
// is returned
int read_int(char* prompt) {
    int number = 0;
    
    pc.printf(prompt);
    
    char input;
    while(1) {
        input = pc.getc();
        pc.putc(input);
        
        if( std::isdigit(input) ) {
            number = (number * 10) + (input-'0');   
        } else {
            pc.putc(input);
            break;   
        } 
    }
    
    return number;
}

void handle_input(int* car_speed)
{
    int valid_reading = 0;
    while( !valid_reading )
    {
        *car_speed = read_int("car_speed:");   
        if( *car_speed >= MIN_CAR_SPEED && *car_speed <= MAX_CAR_SPEED )
        {
            valid_reading = 1;
            printf("\r\n");
        }
        else
        {
            printf("\r\n");
            printf("invalid input ... range: %d -> %d\r\n", MIN_CAR_SPEED, MAX_CAR_SPEED);
        }
            
    }
}

// The main thread of the program. This runs the main program, accepting user
// input, creating cars, displaying state of cars, and handling coordination that
// may be needed. Timing statistics are also presented.
int main()
{   
    // ------------------------------------------------------------------------------
    // The following three variables are used for timing statistics, do not modify them
    Timer stopwatch;    // A timer to keep track of how long the updates take, for statistics purposes
    int numberCycles = 0;
    int totalUpdateTime = 0;
    // ------------------------------------------------------------------------------
    // prepare screen for input
    printf("\033[2J");
    printf("\r\n");
    
    
    
    Car car(0);         // Create a car. This call will need to be modified with any additional parameters you add
    stopwatch.start();  // Start out stats timer
    
    while (true) {      // Run the simulation indefinitely
        
        int carspeed =0;
        // Prompt the user for an initial speed and ensure it is valid
        handle_input(&carspeed);
        // Start the car / let the car know its initial speed
        car.reset(carspeed);
        lcd.cls();
        // ----------------------------------------------------------------------
        // Timing statistics logic, do not modify
        stopwatch.reset();
        // ----------------------------------------------------------------------
        
        do {    // Run the simulation until the car exits the road
;
            // wait some time before sending signal
            while(stopwatch.read_ms() < TICK)
            {
                // do nothing I guess
            }
            
            //signal car it can update
            send_signal( CAR_SIGNAL );
            
            // wait for car to update
            wait_for_signal( CAR_UPDATE_MAIN_SIGNAL );

            // ------------------------------------------------------------------
            // Timing statistics logic, do not modify
            totalUpdateTime += stopwatch.read_ms();
            numberCycles++;
            stopwatch.reset();
            // ------------------------------------------------------------------
            lcd.cls();
            // Display the state of the car to the LCD
            lcd.printf( "%d -> %d \n",car.speed,car.position );
            
        } while (car.is_simulating());     // Modify this condition to stop the simulation when the car exits the road
        // ----------------------------------------------------------------------
        // Timing statistics printout, do not modify
        pc.printf("Resetting...\r\n");
        pc.printf("Average update cycle took: %fms \r\n", (totalUpdateTime*1.0)/(numberCycles*1.0));
        totalUpdateTime = 0;
        numberCycles = 0;
        // ----------------------------------------------------------------------
    }
}
