#include "MTSCellularManager.hpp"

MTSCellularManager::MTSCellularManager(const char* apn_) : apn(apn_), io(NULL), radio(NULL) {
}

MTSCellularManager::~MTSCellularManager() {
    delete radio;
    delete io;
}

bool MTSCellularManager::init() {
    logInfo("initializing cellular radio");

    io = new mts::MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
    // radio default baud rate is 115200
    io->baud(115200);
    if (! io)
        return false;
    
    radio = mts::CellularFactory::create(io);
    radio2 = mts::CellularFactory::create(io);
    if (! radio)
        return false;
    
    // Transport must be set properly before any TCPSocketConnection or UDPSocket objects are created
    Transport::setTransport(radio);

    logInfo("setting APN");
    if (radio->setApn(apn) != MTS_SUCCESS) {
        logError("failed to set APN to \"%s\"", apn);
        return false;
    }

    logInfo("bringing up the link");
    if (! radio->connect()) {
        logError("failed to bring up the link");
        return false;
    } else {
        return true;
    }
}

void MTSCellularManager::uninit() {
    logInfo("finished - bringing down link");
    radio->disconnect();
    delete radio;
    delete io;
}

void MTSCellularManager::enableGps() {

    if( !radio->GPSenabled() ) {
        logInfo("GPS: enabling");
        radio->GPSenable();
        while( !radio->GPSenabled() ) {
            logInfo("...");
            wait(5);
        }
        logInfo("GPS enabled");
    }
}

MTSCellularManager::GPSStatus MTSCellularManager::gpsPollStatus() {
    GPSStatus status;

    int i;
    char c;
    
    if( !radio->GPSenabled() ) {
        logInfo("GPS: enabling");
        radio->GPSenable();
        while( !radio->GPSenabled() ) {
            logInfo("...");
            wait(5);
        }
    }


    logInfo("GPS polling");
    mts::Cellular::gpsData gpsValues = radio->GPSgetPosition();

    // mocked data
    /*
    gpsValues.success = true;
    gpsValues.fix = 3;
    gpsValues.satellites = 4;
    gpsValues.timestamp = "17/05/12, 08:59:56.483";
    gpsValues.latitude = "5925.4477N";
    gpsValues.longitude = "02444.9046E";
    gpsValues.kmhr = "40";
    */

    if (gpsValues.success && gpsValues.fix > 2) {
        logInfo("GPS polled: success %d, sats: %d, fix: %d, ts %s, lat %s, lng %s",
            gpsValues.success, gpsValues.satellites, gpsValues.fix, gpsValues.timestamp.c_str(), gpsValues.latitude.c_str(), gpsValues.longitude.c_str());
        status.success = gpsValues.success;
        status.fix = gpsValues.fix;
        status.speedVal = gpsValues.kmhr / 0.621371f;
        status.latitudeVal = .0;
        status.longitudeVal = .0;
        sscanf(gpsValues.latitude.substr(0, gpsValues.latitude.size()-1).c_str(), "%02d%lf", &i, &status.latitudeVal);
        c = gpsValues.latitude.c_str()[gpsValues.latitude.size()-1];
        //logInfo("i=%d,lat=%f,c=%c", i, latitudeVal ,c);
        status.latitudeVal /= 60.0;
        status.latitudeVal += i;
        if(c == 'S')
            status.latitudeVal = 0.0 - status.latitudeVal;
        sscanf(gpsValues.longitude.substr(0, gpsValues.longitude.size()-1).c_str(), "%03d%lf", &i, &status.longitudeVal);
        c = gpsValues.longitude.c_str()[gpsValues.longitude.size()-1];
        //logInfo("i=%d,lon=%f,c=%c", i, status.longitudeVal, c);
        status.longitudeVal /= 60.0;
        status.longitudeVal += i;
        if(c == 'W')
            status.longitudeVal = 0.0 - status.longitudeVal;
    } else {
        status.success = false;
        logInfo("GPS polled failed: success %d, sats: %d, fix: %d",
            gpsValues.success, gpsValues.satellites, gpsValues.fix);
    }

    return status;
}
