/**
 * @file main.cpp
 * @brief ros-face-driver for mbed stm32f446
 * @author m.hdk
 * @date 2021
 */
 
#include "mbed.h"
#include <ros.h>
#include <std_msgs/String.h>

 
ros::NodeHandle nh;
DigitalOut myled(LED2);
DigitalOut ch0(D2);
DigitalOut ch1(D3);
DigitalOut ch2(D4);
DigitalOut ch3(D5);
DigitalOut ch4(D6);
DigitalOut ch5(D7);
DigitalOut ch6(D8);
DigitalOut ch7(D9);

class FaceDriver
{
public:
    FaceDriver(float period, float width);
    void enable();
    void disable();
    void write(int ch, bool s);

private:
    Timeout tim;
    Ticker tic;

    float period_;
    float width_;
    bool out;
    int pin;
    int over_cut_width();
    void rise();
    void fall();
    void on_pin();
    void off_pin();
};

FaceDriver::FaceDriver(float period,
                       float width)
    : period_(period), width_(width)
{

    out = false;
    period_ = period;
    width_ = width;
}

void FaceDriver::enable()
{
    over_cut_width();
    tic.attach(this, &FaceDriver::rise, period_);
}

void FaceDriver::disable()
{
    tic.detach();
    off_pin();
    out = false;
}

void FaceDriver::on_pin()
{
    switch (pin)
    {
    case 0:
        ch0 = 1;
        break;
    case 1:
        ch1 = 1;
        break;
    case 2:
        ch2 = 1;
        break;
    case 3:
        ch3 = 1;
        break;
    case 4:
        ch4 = 1;
        break;
    case 5:
        ch5 = 1;
        break;
    case 6:
        ch6 = 1;
        break;
    case 7:
        ch7 = 1;
        break;
    default:
        break;
    }
}

void FaceDriver::off_pin()
{
    switch (pin)
    {
    case 0:
        ch0 = 0;
        break;
    case 1:
        ch1 = 0;
        break;
    case 2:
        ch2 = 0;
        break;
    case 3:
        ch3 = 0;
        break;
    case 4:
        ch4 = 0;
        break;
    case 5:
        ch5 = 0;
        break;
    case 6:
        ch6 = 0;
        break;
    case 7:
        ch7 = 0;
        break;
    default:
        break;
    }
}
void FaceDriver::rise()
{
    if (out)
    {
        on_pin();
        tim.attach(this, &FaceDriver::fall, width_);
    }
}

void FaceDriver::fall()
{
    off_pin();
}

int FaceDriver::over_cut_width()
{
    float max = 0.0f;
    max = (period_ / 2.0f);
    if (max < width_)
    {
        width_ = max;
        return 1;
    }
    else
    {
        return 0;
    }
}

void FaceDriver::write(int ch, bool trg)
{
    if (trg)
    {
        out = true;
    }
    else
    {
        out = false;
    }
    pin = ch;
}


FaceDriver face(0.025, 0.002);

void pulse(int i){
    float t = 0.3f;
    switch(i){
        case 0:
            face.write(0, true);
            wait(t);
            face.write(0, false);
        break;
        case 1:
            face.write(1, true);
            wait(t);
            face.write(1, false);
        break;
        case 2:
            face.write(2, true);
            wait(t);
            face.write(2, false);
        break;
        case 3:
            face.write(3, true);
            wait(t);
            face.write(3, false);
        break;
        case 4:
            face.write(4, true);
            wait(t);
            face.write(4, false);
        break;
        case 5:
            face.write(5, true);
            wait(t);
            face.write(5, false);
        break;
        case 6:
            face.write(6, true);
            wait(t);
            face.write(6, false);
        break;
        case 7:
            face.write(7, true);
            wait(t);
            face.write(7, false);
        break;
    }
}

std_msgs::String Read;


void cb(const std_msgs::String& msg){
    Read = msg;
    myled = 1;
    if(!strcmp(Read.data, "ch0")){
        pulse(0);
    }else if(!strcmp(Read.data, "ch1")){            
        pulse(1);
    }else if(!strcmp(Read.data, "ch2")){            
        pulse(2);
    }else if(!strcmp(Read.data, "ch3")){            
        pulse(3);
    }else if(!strcmp(Read.data, "ch4")){            
        pulse(4);
    }else if(!strcmp(Read.data, "ch5")){            
        pulse(5);
    }else if(!strcmp(Read.data, "ch6")){            
        pulse(6);
    }else if(!strcmp(Read.data, "ch7")){            
        pulse(7);
    }else if(!strcmp(Read.data, "debag_on")){
         face.write(0, true);
    }else if(!strcmp(Read.data, "debag_off")){
         face.write(0, false);
    }else{
        wait(0.2);
        myled = 0;
        wait(0.2);
        myled = 1;
        wait(0.2);
    }
    myled = 0;
}


ros::Subscriber<std_msgs::String> sub("face_cmd", &cb);


int main() {
    nh.initNode();
    nh.subscribe(sub);
    face.enable();
 
    while (1) {
        nh.spinOnce();
        wait_ms(1);
    }
}