#include "mbed.h"
#include "x_nucleo_iks01a1.h"
#include "x_nucleo_53l0a1.h"
#include "senet_packet.h"
#include "mDot.h"
#include "dot_util.h"

//////////////////////////////////////////////////////////////////////////
// * these options must match the the Senet credentials for your device //
// * edit their values to match your configuration                      //
/////////////////////////////////////////////////////////////////////////
// Network Id for Senet public network
static uint8_t app_eui[] = { 0x00, 0x25, 0x0C, 0x00, 0x00, 0x01, 0x00, 0x01 };
// Register at or Sign in to http://portal.senetco.com/ and register your NodeId to receive your ApKey
// replace this default key with yours
static uint8_t app_key[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

mDot* dot = NULL;

DevI2C i2c(D14, D15);

/* Instantiate the expansion board */
static X_NUCLEO_IKS01A1 *mems_expansion_board = X_NUCLEO_IKS01A1::Instance(&i2c);
// MAKE SURE YOU JUMPER PIN D7 TO PIN D8!
static X_NUCLEO_53L0A1 *range_board = X_NUCLEO_53L0A1::Instance(&i2c, A2, D7, D2);

/* Retrieve the composing elements of the expansion board */
static GyroSensor *gyroscope = mems_expansion_board->GetGyroscope();
static MotionSensor *accelerometer = mems_expansion_board->GetAccelerometer();
static MagneticSensor *magnetometer = mems_expansion_board->magnetometer;
static HumiditySensor *humidity_sensor = mems_expansion_board->ht_sensor;
static PressureSensor *pressure_sensor = mems_expansion_board->pt_sensor;
static TempSensor *temp_sensor1 = mems_expansion_board->ht_sensor;
static TempSensor *temp_sensor2 = mems_expansion_board->pt_sensor;

/* Helper function for printing floats & doubles */
static char *printDouble(char* str, double v, int decimalDigits=2) {
    int i = 1;
    int intPart, fractPart;
    int len;
    char *ptr;

    /* prepare decimal digits multiplicator */
    for (;decimalDigits!=0; i*=10, decimalDigits--);

    /* calculate integer & fractinal parts */
    intPart = (int)v;
    fractPart = (int)((v-(double)(int)v)*i);

    /* fill in integer part */
    sprintf(str, "%i.", intPart);

    /* prepare fill in of fractional part */
    len = strlen(str);
    ptr = &str[len];

    /* fill in leading fractional zeros */
    for (i/=10;i>1; i/=10, ptr++) {
      if(fractPart >= i) break;
      *ptr = '0';
    }

    /* fill in (rest of) fractional part */
    sprintf(ptr, "%i", fractPart);

    return str;
}

void start_mems_board() {
    uint8_t id;

    humidity_sensor->read_id(&id);
    printf("HTS221  humidity & temperature    = 0x%X\r\n", id);

    pressure_sensor->read_id(&id);
    printf("LPS25H  pressure & temperature    = 0x%X\r\n", id);

    magnetometer->read_id(&id);
    printf("LIS3MDL magnetometer              = 0x%X\r\n", id);

    gyroscope->read_id(&id);
    printf("LSM6DS0 accelerometer & gyroscope = 0x%X\r\n", id);

    wait(3);
}

void start_range_board() {
    range_board->InitBoard();

    wait(1);
}

int main() {
    float value1, value2;
    char buffer1[32], buffer2[32];
    int32_t axes[3];
    int ret;
    int status;
    uint32_t distance;    
    std::vector<uint8_t> tx_data;
    uint8_t buffer[64];
    float value;
        
    SensorPacket packet(buffer, sizeof(buffer));

    printf("\r\n--- Application starting up ---\r\n");

    printf("\r\n--- Initializing mems board ---\r\n");
    start_mems_board();

    printf("\r\n--- Initializing range board ---\r\n");
    start_range_board();

    printf("\r\n--- Initializing LoRa stack ---\r\n");
    // get a handle to the stack
    dot = mDot::getInstance();
    // reset config to start from a known state
    dot->resetConfig();
    dot->resetNetworkSession();
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    if (dot->getJoinMode() != mDot::OTA) {
        logInfo("changing network join mode to OTA");
        if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
            logError("failed to set network join mode to OTA");
        }
    }
    update_ota_config_id_key(app_eui, app_key, 0, true, 0);
    // save changes to configuration
    logInfo("saving configuration");
    if (!dot->saveConfig()) {
        logError("failed to save configuration");
    }

    // display configuration
    display_config();

    // join network
    while (true) {
        if (dot->joinNetwork() == mDot::MDOT_OK) {
            break;
        }

        wait(1);
    }

    while (true) {
        // display sensor data on debug port
        ret = temp_sensor1->get_temperature(&value1);
        if (ret) {
            printf("failed to get temp C\r\n");
        }
        ret = humidity_sensor->get_humidity(&value2);
        if (ret) {
            printf("failed to get humidity\r\n");
        }
        printf("HTS221: [temp] %7s°C,   [hum] %s%%\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));

        ret = temp_sensor2->get_fahrenheit(&value1);
        if (ret) {
            printf("failed to get temp F\r\n");
        }
        ret = pressure_sensor->get_pressure(&value2);
        if (ret) {
            printf("failed to get pressure F\r\n");
        }
        printf("LPS25H: [temp] %7s°F, [press] %smbar\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));

        ret = magnetometer->get_m_axes(axes);
        if (ret) {
            printf("failed to get magnetometer\r\n");
        }
        printf("LIS3MDL [mag/mgauss]:  %7ld, %7ld, %7ld\r\n", axes[0], axes[1], axes[2]);
        
        ret = accelerometer->get_x_axes(axes);
        if (ret) {
            printf("failed to get accelerometer\r\n");
        }
        printf("LSM6DS0 [acc/mg]:      %7ld, %7ld, %7ld\r\n", axes[0], axes[1], axes[2]);

        ret = gyroscope->get_g_axes(axes);
        if (ret) {
            printf("failed to get gyroscope\r\n");
        }
        printf("LSM6DS0 [gyro/mdps]:   %7ld, %7ld, %7ld\r\n", axes[0], axes[1], axes[2]);

        status = range_board->sensor_centre->GetDistance(&distance);
        if (status == VL53L0X_ERROR_NONE) {
            printf("Distance : %ld\n", distance);
        } else {
            printf("failed to get distance - possibly out of range!\r\n");
        }

        // add temperature to packet - slot 0, sensor type 2
        packet.addSensorValue(0, 2, (int16_t)value1);

        // add distance to packet - slot 1, sensor type 3
        packet.addSensorValue(1, 3, (int16_t)distance);

        // add pressure to packet - slot 2, sensor type 4
        packet.addSensorValue(2, 4, (int16_t)value2);

        //serialize and send the data
        packet.serialize();
        tx_data.assign(packet.payload(), packet.payload() + packet.length());

        if (dot->send(tx_data) != mDot::MDOT_OK) {
            printf("failed to send data!\r\n");
        }

        wait(1);
    }
}
