#include "mbed.h"
#include <time.h>
#include <rtc_api.h>

int is_rtc_enabled(void) {
    return (RTC ->ISR & RTC_ISR_INITS ) && (RTC ->ISR & RTC_ISR_RSF );
}

int main() {
    // attach our custom RTC enabled function so time can survive across sleeps & resets
    attach_rtc(&rtc_read, &rtc_write, &rtc_init, &is_rtc_enabled);
    if (!is_rtc_enabled()) 
        set_time(0);
    RTC_HandleTypeDef RtcHandle;
    RtcHandle.Instance = RTC;

    int interval = 4;
            
    while(true) {
        
        __PWR_CLK_ENABLE();
        HAL_PWR_EnableBkUpAccess();
        
        HAL_PWREx_EnableMainRegulatorLowVoltage();
        HAL_PWREx_EnableFlashPowerDown();
        HAL_PWREx_EnableLowRegulatorLowVoltage();
    
        // make sure RTC and LSE stay on while we're asleep
        RCC ->BDCR |= RCC_BDCR_LSEON;
        RCC ->BDCR |= RCC_BDCR_RTCEN;

        printf("configuring wakeup timer for %lu seconds from now\r\n", interval);

        HAL_RTCEx_SetWakeUpTimer_IT(&RtcHandle, interval, RTC_WAKEUPCLOCK_CK_SPRE_16BITS );

        printf("entering sleep mode %08x\r\n", RTC->ISR);
        fflush(stdout);
        
        // clear any pending interrupts
        EXTI ->PR = (1 << 22);

        HAL_PWR_EnterSTOPMode(PWR_LOWPOWERREGULATOR_ON, PWR_STOPENTRY_WFI );
                
        HAL_PWREx_DisableMainRegulatorLowVoltage();
        HAL_PWREx_DisableFlashPowerDown();
        HAL_PWREx_DisableLowRegulatorLowVoltage();
        
        __HAL_RTC_WAKEUPTIMER_CLEAR_FLAG(&RtcHandle, RTC_FLAG_WUTF);

        // reconfigure PLL
        SetSysClock();
        wait(1.0);
        printf("wake\r\n");    
    }

    return 0;    
}
