/******************************************************************************
SCL3300.h
SCL3300 Arduino Library Header File
David Armstrong
Version 3.2.0 - September 3, 2021
https://github.com/DavidArmstrong/SCL3300

Resources:
Uses SPI.h for SPI operation
This code is released under the [MIT License](http://opensource.org/licenses/MIT).
Please review the LICENSE.md file included with this example.
Distributed as-is; no warranty is given.

This file prototypes the SCL3300 class, as implemented in SCL3300.cpp

******************************************************************************/

// ensure this library description is only included once
#ifndef __SCL3300_h
#define __SCL3300_h

// Uncomment the following line for debugging output
//#define debug_scl3300

#include "mbed.h"// SPI library is used for...SPI.

//Define allowed commands to SCL3300 inclinometer
#define RdAccX      0x040000f7
#define RdAccY      0x080000fd
#define RdAccZ      0x0c0000fb
#define RdSTO       0x100000e9
#define EnaAngOut   0xb0001f6f
#define RdAngX      0x240000c7
#define RdAngY      0x280000cd
#define RdAngZ      0x2c0000cb
#define RdTemp      0x140000ef
#define RdStatSum   0x180000e5
#define RdErrFlg1   0x1c0000e3
#define RdErrFlg2   0x200000c1
#define RdCMD       0x340000df
#define ChgMode1    0xb400001f
#define ChgMode2    0xb4000102
#define ChgMode3    0xb4000225
#define ChgMode4    0xb4000338
#define SetPwrDwn   0xb400046b
#define WakeUp      0xb400001f
#define SWreset     0xb4002098
#define RdWHOAMI    0x40000091
#define RdSer1      0x640000a7
#define RdSer2      0x680000AD
#define RdCurBank   0x7c0000b3
#define SwtchBnk0   0xfc000073
#define SwtchBnk1   0xfc00016e

// Structure to hold raw sensor data
// We need to populate all this every time we read a set of data
struct SCL3300data {
  public:
    int16_t AccX;
    int16_t AccY;
    int16_t AccZ;
    int16_t STO;
    int16_t TEMP;
    int16_t AngX;
    int16_t AngY;
    int16_t AngZ;
    uint16_t StatusSum;
    uint16_t WHOAMI;
};

// SCL3300 library interface description
class SCL3300 {
  // user-accessible "public" interface
  public:
    SCL3300 (PinName mosi, PinName miso, PinName sclk, PinName scl3300_csPin);//constructer
    
    SCL3300data sclData;
    bool setMode(int mode);
    bool begin(void);
    //Functions to retrieve sensor data
    bool isConnected();
    bool available(void);
    void setFastReadMode();
    void stopFastReadMode();
    double getCalculatedAngleX(void);
    double getCalculatedAngleY(void);
    double getCalculatedAngleZ(void);
    double getTiltLevelOffsetAngleX(void);
    double getTiltLevelOffsetAngleY(void);
    double getTiltLevelOffsetAngleZ(void);
    double getCalculatedAccelerometerX(void);
    double getCalculatedAccelerometerY(void);
    double getCalculatedAccelerometerZ(void);
    uint16_t getErrFlag1(void);
    uint16_t getErrFlag2(void);
    unsigned long getSerialNumber(void);
    double getCalculatedTemperatureCelsius(void);
    double getCalculatedTemperatureFarenheit(void);
    double angle(int16_t SCL3300_ANG); //two's complement value expected
    double acceleration(int16_t SCL3300_ACC);
    bool crcerr, statuserr;
    uint16_t powerDownMode(void);
    uint16_t WakeMeUp(void);
    uint16_t reset(void);
    
  // library-accessible "private" interface
  private:
    SPI _spi;
    Timer timer1;
    Timer timer2;
    DigitalOut _scl3300_csPin; //  SPI chip select pin
    uint8_t scl3300_mode ; // no Default mode
    uint8_t SCL3300_CMD, SCL3300_CRC;
    uint16_t SCL3300_DATA;
    double Temperature, X_angle, Y_angle, Z_angle;
    bool setFastRead; //chose mode
   uint32_t _transfer_sck;
    
    void initSPI();
    void beginTransmission();
    void endTransmission();
    uint8_t CalculateCRC(uint32_t Data);
    uint8_t CRC8(uint8_t BitValue, uint8_t SCL3300_CRC);
    unsigned long transfer(unsigned long value);

    union FourByte {
      unsigned long bit32;
      unsigned int bit16[2];
      unsigned char bit8[4];
    };
    unsigned long modeCMD[5];
};
#endif
