#include <math.h>
#include "TMP007.h"
 
TMP007::TMP007(PinName sda, PinName scl, int addr) 
    :_i2cPort(sda, scl), 
    _addr(addr<<1) 
{}

//--------------------------------------------------------------------------------
bool TMP007::writeWord(const uint8_t reg, const uint16_t word)
{
    const char cmd[] = { reg,  word & 0xff, (word&0xff00)>>8 };
    
    if( 0 == _i2cPort.write(_addr, cmd, 3 ))
        return true;
    else
        return false;    
}

//--------------------------------------------------------------------------------
uint16_t TMP007::readWord(const uint8_t reg)
{
    uint16_t res = 0;
    char data[]  = {0, 0};
    const char cmd = reg;
    
    _i2cPort.write(_addr, &cmd, 1);
    _i2cPort.read(_addr, data, 2);  
    res = data[1]<<8 & data[0];
    
    return res;
}

//--------------------------------------------------------------------------------
void TMP007::writeConfig(const uint16_t value)
{
    writeWord(TMP007_CONFIG, value);   
}

//--------------------------------------------------------------------------------
 uint16_t TMP007::readConfig()
{   
    uint16_t res = readWord(TMP007_CONFIG);
    return res;   
}

//--------------------------------------------------------------------------------
uint16_t TMP007::readStatus()
{
    return readWord(TMP007_STATUS);
}

//--------------------------------------------------------------------------------
void TMP007::writeStatusMask(const uint16_t value)
{
    writeWord(TMP007_STATUS_MASK, value);
}

//-------------------------------------------------------------------------------- 
/** TMP007 Config method.
 *  Used for configuring the sensor with desired samples.
 */
void TMP007::setSamples(uint16_t samples)
{
    char    data[2];
 
    data[0]   = TMP007_CONFIG;
    data[1]   = samples | TMP007_CFG_MODEON | TMP007_CFG_ALRTEN;
 
    _i2cPort.write( _addr, data, 1 ); 
}
 
//--------------------------------------------------------------------------------
/** TMP007 Read Raw Temperature method.
 *  Used for getting the raw data off the chip.
 */
uint16_t TMP007::readRawLocalTemperature()
{
  uint16_t raw = readWord(TMP007_LOCAL_TEMP);
 
  raw >>= 2;
  return raw;
}
 
//--------------------------------------------------------------------------------
/** TMP007 Read Raw Voltage method.
 *  Used for reading the raw voltage from the thermopile.
 */
uint16_t TMP007::readRawSensorVoltage() 
{  
    return readWord(TMP007_SENSOR_VOLT);
}
 
//--------------------------------------------------------------------------------
/** TMP007 Read Object Temperature(C) method.
 *  Used for calculating the object temperature in celcius.
 */
double TMP007::readObjTempC() 
{
  double Tdie = readRawLocalTemperature();
  double Vobj = readRawSensorVoltage();
  Vobj *= 156.25;     // 156.25 nV per LSB
  Vobj /= 1000000000; // nV -> V
  Tdie *= 0.03125;    // convert to celsius
  Tdie += 273.15;     // convert to kelvin
 
  // Equations for calculating temperature found in section 7.3.4 in the user guide
  double tdie_tref = Tdie - TMP007_TREF;
  double S = (1 + TMP007_A1*tdie_tref + TMP007_A2*tdie_tref*tdie_tref);
  S *= TMP007_S0;
  S /= 10000000;
  S /= 10000000;
 
  double Vos = TMP007_B0 + TMP007_B1*tdie_tref + TMP007_B2*tdie_tref*tdie_tref;
 
  double fVobj = (Vobj - Vos) + TMP007_C2*(Vobj-Vos)*(Vobj-Vos);
 
  double Tobj = sqrt(sqrt(Tdie * Tdie * Tdie * Tdie + fVobj/S));
 
  Tobj -= 273.15; // Kelvin -> *C
  return Tobj;
}

//--------------------------------------------------------------------------------
/** TMP007 Read Object Temperature(F) method.
 *  Used for calculating the object temperature in fahrenheit.
 */
double TMP007::readObjTempF() 
{
  double Tobj = readObjTempC();
  Tobj = Tobj * 9.0/5.0 + 32.0; // convert to fahrenheit
  return Tobj;
}

//--------------------------------------------------------------------------------
/** TMP007 Read Die Temperature(C) method.
 *  Used for calculating the die temperature in celcius.
 */
double TMP007::readDieTempC() 
{
  double Tdie = readRawLocalTemperature();
  Tdie *= 0.03125; // convert to celsius
  return Tdie;
}

//--------------------------------------------------------------------------------
/** TMP007 Read Die Temperature(F) method.
 *  Used for calculating the die temperature in fahrenheit.
 */
double TMP007::readDieTempF() 
{
  double Tdie = readDieTempC();
  Tdie = Tdie * 9.0/5.0 + 32.0; // convert to fahrenheit
  return Tdie;
}