/** TMP007 Temperature methods.
 *  Used for interfacing the TMP007 with the mbed.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "TMP007.h"
 
 * #define Address 0x40 // depending on pin ADR0 and ADR1 it can be either 0x40, 0x41, 0x44 or 0x45
 *
 * TMP007 sensor(p9, p10, Address); 
 *
 * int main()
 * {
 * while(1) {
 *   printf("TTemperature: %f F \r \n", sensor.readObjTempF(Address));
 *   wait(0.5);
 * }
 * }
 * @endcode
 */
#ifndef TMP007_H
#define TMP007_H
 
#include <I2C.h>
#include "PinNames.h"
#include "TMP007_Defs.h"


class TMP007
{
public:
 
    // Constructor
    TMP007(PinName sda, PinName scl, int addr);
    
    void writeConfig(const uint16_t value);
    uint16_t readConfig() ;
    
    uint16_t readStatus();
    void writeStatusMask(const uint16_t value);
    
    /** Configures sensor, use before reading from it */
    void setSamples(const uint16_t samples);
    
    /** Read raw sensor temperature */
    uint16_t readRawLocalTemperature();
    
    /** Read raw thermopile voltage */
    uint16_t readRawSensorVoltage();
    
    /** Calculate object temperature (C) based on raw sensor temp and thermopile voltage */
    double readObjTempC();
    
    /** Calculate object temperature (F) based on raw sensor temp and thermopile voltage */
    double readObjTempF();
    
    /** Caculate sensor temperature (C) based on raw reading */
    double readDieTempC();  
    
    /** Caculate sensor temperature (F) based on raw reading */
    double readDieTempF();
    
  private:
    /**
     * Writes a word value into the register.  Note: the device must have the
     * auto-increment bit set in the MODE1 register to work.
     *
     * @param reg Register location to write into
     * @param word Word to write
     * @return True if successful
     */
    bool writeWord(const uint8_t reg, const uint16_t word);

    /**
     * Read a word value from the register.  Note: the device must have the
     * auto-increment bit set in the MODE1 register to work.
     *
     * @param reg Register location to read from
     * @return Value in the specified register
     */
    uint16_t readWord(const uint8_t reg);
    
private:
    mbed::I2C        _i2cPort;
    uint8_t          _addr;
 
};
 
#endif