/* Discrete RGB color sensor
 *
 * - uses single-channel light-dependent resistor (via ADC)
 *   and a RGB LED.
 * -  compensates background light
 *
 * Copyright (c) 2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef __RGB_SENSOR_H__
#define __RGB_SENSOR_H__

#define RGB_MASK 0xFFF

#ifndef RGB_LED_ON
#define RGB_LED_ON true
#endif/*RGB_LED_ON*/

#ifndef RGB_OVERSAMPLING
#define RGB_OVERSAMPLING 32
#endif/*RGB_OVERSAMPLING*/

#ifndef RGB_SENSOR_IGNORE
#define RGB_SENSOR_IGNORE (RGB_OVERSAMPLING/2)
#endif/*RGB_SENSOR_IGNORE*/


#ifndef RGB_MAX_ADC_CLK
#define RGB_MAX_ADC_CLK 6000000UL
#endif/*RGB_MAX_ADC_CLK*/

#define RGB_CHANNELS 4

typedef struct
{
    int red,green,blue;
} TRGB_Channel;

typedef union
{
    TRGB_Channel ch;
    int data[3];
} TRGB;

typedef bool (*TRGB_Callback)(const TRGB &rgb);

class RGB_Sensor
{
    public:
      RGB_Sensor(PinName red, PinName green, PinName blue, PinName adc);
      ~RGB_Sensor(void);
      bool capture(TRGB &rgb);
      bool capture(const TRGB_Callback callback);
      bool wait(void);
    protected:
      TRGB_Callback m_callback;
      bool m_done;
      uint8_t m_adc_channel, m_rgb_channel;
      DigitalOut m_red, m_green, m_blue;
      int m_adc_filter[RGB_CHANNELS][2], m_adc_filter_pos;
      int m_adc_aggregation[RGB_CHANNELS], m_adc_count;
      void convert(TRGB &rgb);
      int filter(int sample);
    private:
      static void __adc_irq(void);
      void adc_irq(void);
      static RGB_Sensor *m_global;
};

#endif/*__RGB_SENSOR_H__*/