#include <mbed.h>
#include <thunk.h>

class CTest
{
public:
    CTest(void);
    void test(void);
    void hexdump(const void* data, int length);

    Serial pc;
    CThunk<CTest> thunk;

    void callback2(void* context);

private:
    CThunkIRQ<CTest> m_irq;
    DigitalOut m_led1, m_led2, m_led3;

    void callback1(void);
    void callback_irq(void);

    uint32_t m_counter;
};

CTest::CTest(void)
    :pc(USBTX, USBRX),
    thunk(this, &CTest::callback1),
    m_irq(this, TIMER0_IRQn, &CTest::callback_irq),
    m_led1(LED1),
    m_led2(LED2),
    m_led3(LED3)
{
    m_counter = 0;

    pc.baud(115200);
    
    /* enable timer */
    LPC_SC->PCONP |= 2;
    LPC_TIM0->TCR = 2;
    LPC_TIM0->MR0 = 2398000;
    LPC_TIM0->MCR = 3;
    LPC_TIM0->TCR = 1;
}

void CTest::callback1(void)
{
    pc.printf("callback1: called (this=0x%0X)\n", this);

    /* turn on LED1 */
    m_led1 = 1;

    /* increment member variable */
    pc.printf("callback1:   m_counter before: %i\n", m_counter);
    m_counter++;
    pc.printf("callback1:   m_counter after : %i\n", m_counter);
}

void CTest::callback2(void* context)
{
    pc.printf("callback2: called with context value 0x%08X\n", context);

    /* turn on LED2 */
    m_led2 = 1;

    /* increment member variable */
    pc.printf("callback2:   m_counter before: %i\n", m_counter);
    m_counter+=2;
    pc.printf("callback2:   m_counter after : %i\n", m_counter);
}

void CTest::callback_irq(void)
{
    uint32_t reason = LPC_TIM0->IR;

    /* verify reason */
    if(reason & 1)
        m_led3 = !m_led3;

    /* acknowledge IRQ */
    LPC_TIM0->IR = reason;
}

void CTest::hexdump(const void* data, int length)
{
    int i;

    pc.printf("Dump %u bytes from 0x%08X\n", length, data);

    for(i=0; i<length; i++)
    {
        if((i%16) == 0)
            pc.printf("\n");
        else
            if((i%8) == 0)
                pc.printf(" - ");
        pc.printf("0x%02X ", ((uint8_t*)data)[i]);
    }
    pc.printf("\n");
}

int main(void)
{
    /* allocate thunking test class */
    CTest test;
    /* allocate 32 bit pointer to thunk-entry */
    CThunkEntry entry;

    /* to make a point: get 32 bit entry point pointer from thunk */
    entry = test.thunk;

    /* TEST1: */
    /* callback function has been set in the CTest constructor */
    test.hexdump((const void*)entry, 20);
    /* call entry point */
    entry();

    /* TEST2: */
    /* assign a context ... */
    test.thunk.context(0xDEADBEEF);
    /* and switch callback to callback2 */
    test.thunk.callback(&CTest::callback2);
    /* call entry point */
    entry();

    /* sit waiting for timer interrupts */
    while(1)
        __WFI();
}
