/* General C++ Object Thunking class
 *
 * - allows direct callbacks to non-static C++ class functions
 * - keeps track for the corresponding class instance
 * - supports an optional context parameter for the called function
 * - ideally suited for class object receiving interrupts (NVIC_SetVector)
 *
 * Copyright (c) 2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef __CTHUNKIRQ_H__
#define __CTHUNKIRQ_H__

#include "CThunk.h"

template<class T>
class CThunkIRQ: CThunk<T> {

public:
    typedef void (T::*CCallbackSimple)(void);
    typedef void (T::*CCallback)(void* context);

    inline CThunkIRQ(T *instance, IRQn_Type IRQn)
        :CThunk<T>(instance)
    {
        register_irq(IRQn);
    }

    inline CThunkIRQ(T *instance, IRQn_Type IRQn, CCallback callback)
        :CThunk<T>(instance, callback)
    {
        register_irq(IRQn);
    }

    inline CThunkIRQ(T *instance, IRQn_Type IRQn, CCallbackSimple callback)
        :CThunk<T>(instance, callback)
    {
        register_irq(IRQn);
    }

    inline CThunkIRQ(T &instance, IRQn_Type IRQn, CCallback callback)
        :CThunk<T>(instance, callback)
    {
        register_irq(IRQn);
    }

    inline CThunkIRQ(T &instance, IRQn_Type IRQn, CCallbackSimple callback)
        :CThunk<T>(instance, callback)
    {
        register_irq(IRQn);
    }

    inline CThunkIRQ(T &instance, IRQn_Type IRQn, CCallback callback, void* context)
        :CThunk<T>(instance, callback, context)
    {
        register_irq(IRQn);
    }

    inline ~CThunkIRQ(void)
    {
        NVIC_DisableIRQ(m_irqn);
        NVIC_SetVector(m_irqn, m_prev_handler);        
    }

private:
    IRQn_Type m_irqn;
    uint32_t m_prev_handler;
    inline void register_irq(IRQn_Type IRQn)
    {
        /* remember IRQn for destructor */
        m_irqn = IRQn;

        /* update IRQ handler atomically */
        NVIC_DisableIRQ(IRQn);
        m_prev_handler = NVIC_GetVector(IRQn);
        NVIC_SetVector(IRQn, CThunk<T>::entry());
        NVIC_EnableIRQ(IRQn);
    }
};

#endif/*__CTHUNKIRQ_H__*/
