#include "mbed.h"
#include "LocalPinNames.h"
#include "BridgeDriver.h"
#include "FrontPanelButtons.h"
#include "TextLCD.h"

I2C i2c( P0_10, P0_11 ); // I2C bus (SDA, SCL)
BridgeDriver bridges( &i2c ); // bridge
TextLCD_I2C lcd( &i2c, MCP23008_SA0, TextLCD::LCD20x4 ); // LCD
FrontPanelButtons buttons( &i2c );    // front panel buttons

DigitalIn TDC(DIO0, PullDown);
AnalogIn signal(AI0);

Ticker ErrorWatch;
Timer timer; //initialize a timer
Timer avgCycTimer;

float totaltime = 0;

SPI spi(P0_9, P0_8, P0_7); // mosi(out), miso(in), sclk(clock)
DigitalOut cs_Current(P1_0); // cs (the chip select signal)

float stateDir = 0;
int numCycles = 200000;
int cycleCount = 1;

void errorCheck(){
    
    if (timer.read() > 3.0){
        bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
        
        lcd.cls(); //clear the display
        lcd.setAddress ( 0, 0 );
        lcd.printf( "<Error Occurred>" );
        lcd.setAddress ( 0, 1 );
        lcd.printf( "Timeout, transition greater than 3      seconds" );
        lcd.setAddress ( 0, 3 );
        lcd.printf( "<Press Sel to Resume" );
        while (!buttons.readSel());
        lcd.cls(); //clear the display
        
        lcd.setAddress( 0, 0 );
        lcd.printf( "Cycle %d/%3d", cycleCount, numCycles );
        lcd.setAddress( 0, 1 );
        lcd.printf( "Avg t(sec): %1.3f", (totaltime / cycleCount));

        timer.reset();
        timer.start();
        
        bridges.drive( BridgeDriver::MOTOR_B, stateDir );
    }   
}



float calcCurrent(int channel){
        
    // Select the device by seting chip select low
    cs_Current = 0;

    // sending the 6 bits + 1 bit to ignore the null bit
    // coming from the device, so the data that is sent is 1100000
    // get data for channel 0
    //spi.write(0x68); // 1101000
    
    // create the message to send in order to get the appropriate data transmitted from the ADC
    unsigned char channelBits = (channel-1) << 2; // shift the channel bits into there position
    unsigned char msg = 96 + channelBits;
    spi.write(msg);
    
    // now the device sends back the readings 12 bits, 7 bits at a time
    uint8_t high = spi.write(0x00);
    uint8_t low = spi.write(0x00);

    // shift out the right bits
    low = ( high << 5 ) | (low >> 2);
    high = high >> 3;
    
    
    float DigOutVal = ( high << 8 ) | low; // shift and or the result together

    cs_Current = 1; // Deselect the device
    
    //return current;
    return DigOutVal;
}

int main() {
   
    //initialize Ticker
    ErrorWatch.attach(&errorCheck, 3);
    
    // Setup the spi for 7 bit data, high steady state clock,
    // second edge capture, with a 1MHz clock rate
    spi.format(7,0);
    spi.frequency(1000000);

    i2c.frequency(1000000);
    lcd.setBacklight(TextLCD::LightOn);
    wait(.6);
    lcd.cls(); //clear the display
    lcd.setAddress(0,0); //set cursor to 0,0
       
    int count = 0;
    
    bridges.enablePwm    (  0,  1,  0,  0 );
    bridges.drive( BridgeDriver::MOTOR_B, 0.0 );

    lcd.cls(); //clear the display
    lcd.setAddress ( 0, 2 );
    lcd.printf( "<Press Sel to start>" );
    
    while (!buttons.readSel()){
        if(buttons.readUp() && numCycles < 999999 ){
            numCycles += 500;
            wait(0.2); //so that the speed of changing the numbers is more controllable, should mean you can move 20 digits per second
        }
        else if (buttons.readDown() && numCycles > 0 ){
            numCycles -= 500;
            wait(0.2); //so that the speed of changing the numbers is more controllable, should mean you can move 20 digits per second
        }
        lcd.setAddress ( 0, 0 );
        lcd.printf( "<Cycles: %d >" , numCycles );
    }
        
    bridges.drive( 2, 1 ); //turn on the hall sensor voltage
    
    int hallCount = 109;
    
    //Initialize the spoiler by rotating to stall currents and finding how many ticks the hall sensor takes through the rotation
    bridges.drive( BridgeDriver::MOTOR_B, -1.0 );
    while (calcCurrent(4) < 2200);
    bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
//    wait(1);
//    bridges.drive( BridgeDriver::MOTOR_B, 1.0 );
//    wait(1);
//    
//    int flag = 0;
//    while (calcCurrent(3) < 2200){
//        if(signal.read() >= 0.8 && flag == 0){
//                flag = 1;
//                hallCount++;
//            }
//        else if (signal.read() < 0.8 && flag == 1)
//            flag = 0;
//    }

    bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
    wait(3);
    
    
    lcd.cls(); //clear the display
    
    while(cycleCount <= numCycles){
        
        avgCycTimer.reset();
        avgCycTimer.start();
        
        lcd.setAddress( 0, 0 );
        lcd.printf( "Cycle %d/%3d", cycleCount, numCycles );
        
        stateDir = 1;
        bridges.drive( BridgeDriver::MOTOR_B, 1.0 );
        timer.reset();
        timer.start();
        
        int flag = 0;
        count = 0;
        
        while (TDC.read() == 1){ //while not at TDC
            if(signal.read() >= 0.8 && flag == 0){
                flag = 1;
                count++;
            }
            else if (signal.read() < 0.8 && flag == 1)
                flag = 0;
        }
        
        timer.stop();
    
        //Pause at TDC for 5 seconds before continuing
        stateDir = 0;
        bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
        wait(1);
        stateDir = 1;
        bridges.drive( BridgeDriver::MOTOR_B, 1.0 );
        
        timer.reset();
        timer.start();
        
        int stopCount = 0;
        while (stopCount < (hallCount - count - 20)){
            if(signal.read() >= 0.8 && flag == 0){
                flag = 1;
                stopCount++;
            }
            else if (signal.read() < 0.8 && flag == 1)
                flag = 0;
        }
        
        timer.stop();
        
        //Pause at the end state before rotating back
        stateDir = 0;
        bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
        wait(1);
        stateDir = -1;
        bridges.drive( BridgeDriver::MOTOR_B, -1.0 );
        
        timer.reset();
        timer.start();
        
        flag = 0;
        count = 0;
        while (TDC.read() == 1){ //while not at TDC
            if(signal.read() >= 0.8 && flag == 0){
                flag = 1;
                count++;
            }
            else if (signal.read() < 0.8 && flag == 1)
                flag = 0;
        }
        
        timer.stop();
        
        //Pause at TDC for 5 seconds before continuing
        stateDir = 0;
        bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
        wait(1);
        stateDir = -1;
        bridges.drive( BridgeDriver::MOTOR_B, -1.0 );
        
        timer.reset();
        timer.start();
        
        stopCount = 0;
        while (stopCount < (hallCount - count - 15)){
            if(signal.read() >= 0.8 && flag == 0){
                flag = 1;
                stopCount++;
            }
            else if (signal.read() < 0.8 && flag == 1)
                flag = 0;
        }
        
        timer.stop();
        
        //Pause at TDC for 5 seconds before continuing
        stateDir = 0;
        bridges.drive( BridgeDriver::MOTOR_B, 0.0 );
        wait(1);  
        
        avgCycTimer.stop();
        
        totaltime += avgCycTimer.read();
        lcd.setAddress( 0, 1 );
        lcd.printf( "Avg t(sec): %1.3f", (totaltime / cycleCount));
        wait(0.2);
        
        cycleCount++;
    } 
    
    lcd.cls(); //clear the display   
    lcd.setAddress(0,0);
    lcd.printf("END OF PROGRAM");
    lcd.setAddress(0,1);
    lcd.printf("Num Cycles Met");
    return 0;
    
}
