#include "mbed.h"
#include "BridgeDriver.h"
#include "FrontPanelButtons.h"
#include "SDFileSystem.h"
#include "LocalPinNames.h"
#include "TextLCD.h"
#include "max31855.h"
#include <string>
#include <stdio.h>
using std::string;

#include "DS18S20.h"
#include "DS18B20.h"
#include "OneWireDefs.h"

#define THERMOMETER DS18B20
// device( crcOn, useAddress, parasitic, mbed pin )
THERMOMETER device(true, true, false, p25);
    
Serial pc(USBTX, USBRX);

Timer timer;    // general purpose timer
I2C i2c( P0_10, P0_11 ); // I2C bus (SDA, SCL)
//TextLCD_I2C lcd( &i2c, MCP23008_SA0, TextLCD::lcd20x4 ); // lcd
FrontPanelButtons buttons( &i2c );    // front panel buttons

Timer debounceTimer;
Timer avgCycTimer;

DigitalIn signal(DIO0, PullUp);
CAN can(p9, p10);

//SPI Interfaces
SPI testSPI(P0_9, P0_8, P0_7); // mosi(out), miso(in), sclk(clock) //SPI Bus
//SPI testSPI(P0_9, DIO3, DIO1); // mosi(out), miso(in), sclk(clock) //SPI Bus


CANMessage readBuffer;
int numBuffMsg = 0;

float totaltime = 0;
int cycleCount = 1;
int numCycles = 1000;
    
float currTemp = 0; //Float value to hold temperature returned

//Function Definitions
void waitSwitch(char [], int);
void waitSwitchRelease(char [], int);
void waitLatch(char [], int);
int checkLatchStatus();
float getTemp();


/******************************************************************************/
/************                  CAN Commands                       *************/
/******************************************************************************/

void openDoorCommand(char dataBytes[]){
    dataBytes[3] = 0x03;
    can.write(CANMessage(534, dataBytes, 4)); // open the door
    wait(0.1);
    dataBytes[3] = 0x00;
    can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
}

void closeDoorCommand(char dataBytes[]){
    dataBytes[3] = 0x02;
    can.write(CANMessage(534, dataBytes, 4)); // close the door
    wait(0.1);
    dataBytes[3] = 0x00;
    can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE 
}
    
/******************************************************************************/
/************                    Receieve                         *************/
/******************************************************************************/

void receive(){     
    CANMessage msg;
    if (can.read(msg)){
        if (msg.id == 534){
            readBuffer = msg;
            numBuffMsg = 1;
        }
    }
}


/******************************************************************************/
/************                   Full Init                         *************/
/******************************************************************************/

void fullInit(){
//    i2c.frequency(1000000);
//    lcd.setBacklight(TextLCD::LightOn);
//    wait(.6);
//    lcd.cls(); //clear the display
    
    can.frequency(500000);
    can.attach(&receive, CAN::RxIrq);   


    //********* Init the Thermocouple **********//
    
    while (!device.initialize());    // keep calling until it works
    
    device.setResolution(nineBit);
}


/******************************************************************************/
/************                 waitSwitch                          *************/
/******************************************************************************/

void waitSwitch(char dataBytes[], int openTimeout){
           
    int openComplete = 0;
    while (!openComplete){
        
        openDoorCommand(dataBytes); //send CAN message to open the door
        
        timer.reset();
        timer.start();
        int flag = 0;
        while (!flag){
            if (signal.read() == 0){
                while (!flag){        
                    debounceTimer.reset();
                    debounceTimer.start();
                    while (debounceTimer.read_ms() < 40);
                    if ( signal.read() == 0){
                        flag = 1;
                        openComplete = 1;
                    }
                }
            }
            else if (timer.read() >= openTimeout)
                flag = 1;
        }
        
        timer.stop();
    
        // timeout on opening
        if (timer.read() >= openTimeout){
                        
            avgCycTimer.stop(); //pause
            
            dataBytes[3] = 0x01;
            can.write(CANMessage(534, dataBytes, 4)); // stop the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
    
            //Error Message
//            pc.setAddress ( 0, 1 );
            pc.printf( "<Press Button to Resume>\r\n" );
//            pc.setAddress( 0, 2 );
            pc.printf( "ERROR Open Timeout\r\n" );
        
            while (!buttons.readSel()); //Do nothing until the button is selected
        
            avgCycTimer.start(); //start
            
            //Remove Error Message
//            pc.cls(); //clear the display
            
//            pc.setAddress( 0, 0 );
            pc.printf( "Cycle %d of %3d\r\n", cycleCount, numCycles );
//            pc.setAddress( 0, 1 );
            pc.printf( "Avg t(sec): %1.3f\r\n", (totaltime / cycleCount));
//            pc.setAddress( 0, 2 );
            pc.printf( "STATUS: OPENING\r\n");
        } 
    }
}


/******************************************************************************/
/************          Close Door / Switch Release                *************/
/******************************************************************************/

void waitSwitchRelease(char dataBytes[], int switchReleaseTimeout){
        
    int switchReleaseComplete = 0;
    while (!switchReleaseComplete){
        
        closeDoorCommand(dataBytes); //send CAN message to close the door
        
        timer.reset();
        timer.start();
        int flag = 0;
        while (!flag){
            if (signal.read() == 1){
                while (!flag){        
                    debounceTimer.reset();
                    debounceTimer.start();
                    while (debounceTimer.read_ms() < 40);
                    if ( signal.read() == 1){
                        flag = 1;
                        switchReleaseComplete = 1;
                    }
                }
            }
            else if (timer.read() >= switchReleaseTimeout)
                flag = 1;
        }
        
        timer.stop();

        // timeout on switch release
        if (timer.read() >= switchReleaseTimeout){
                        
            avgCycTimer.stop(); //pause
            
            dataBytes[3] = 0x01;
            can.write(CANMessage(534, dataBytes, 4)); // stop the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
            
            //Error Message
//            pc.setAddress ( 0, 1 );
            pc.printf( "<Press Sel to Resume>\r\n" );
//            pc.setAddress( 0, 2 );
            pc.printf( "ERROR Release Switch Timeout\r\n" );
        
            while (!buttons.readSel()); //Do nothing until the button is selected
            
            avgCycTimer.start(); //start
            
            //Remove Error Message
//            pc.cls(); //clear the display
            
//            pc.setAddress( 0, 0 );
            pc.printf( "Cycle %d of %3d\r\n", cycleCount, numCycles );
//            pc.setAddress( 0, 1 );
            pc.printf( "Avg t(sec): %1.3f\r\n", (totaltime / cycleCount));
//            pc.setAddress( 0, 2 );
            pc.printf( "STATUS: SWITCH\r\n");
        } 
    } 
} 


/******************************************************************************/
/************                 Close Door to Latch                 *************/
/******************************************************************************/

void waitLatch(char dataBytes[], int closeTimeout){
        
    //Loop through the close sequence until the latch is closed
    int closeComplete = 0;
    while (!closeComplete){
        
        closeDoorCommand(dataBytes); //send CAN message to close the door
        
        timer.reset();
        timer.start();
        int flag = 0;
        while (!flag){
            
            if (checkLatchStatus()){
                flag = 1;
                closeComplete = 1;
            }
            
            //if the timer goes off
            else if (timer.read() >= closeTimeout)
                flag = 1;
        }
        
        timer.stop();

        // timeout on closing
        if (timer.read() >= closeTimeout){
                        
            avgCycTimer.stop(); //pause
            
            //Error Message
//            pc.setAddress ( 0, 1 );
            pc.printf( "<Press Sel to Resume>\r\n" );
//            pc.setAddress( 0, 2 );
            pc.printf( "ERROR Close Timeout\r\n" );
            
            dataBytes[3] = 0x01;
            can.write(CANMessage(534, dataBytes, 4)); // stop the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
            
            while (!buttons.readSel()); //Do nothing until the button is selected
            
            avgCycTimer.start(); //start
            
            //Remove Error Message
//            pc.cls(); //clear the display
            
//            pc.setAddress( 0, 0 );
            pc.printf( "Cycle %d of %3d\r\n", cycleCount, numCycles );
//            pc.setAddress( 0, 1 );
            pc.printf( "Avg t(sec): %1.3f\r\n", (totaltime / cycleCount));
//            pc.setAddress( 0, 2 );
            pc.printf( "STATUS: CLOSING\r\n");
        } 
    } 
}


/******************************************************************************/
/************                  Check Latch Status                 *************/
/******************************************************************************/

// Returns 1 if latch is closed, 0 if it is not closed
int checkLatchStatus(){
    
    //convert the data array into a single 64 bit value so that we can grab the range of bits we desire, no matter where they're located
    long long allData = 0; 
    for (int j = 0; j < 8; j++){
        long long tempData1 = (long long)readBuffer.data[j];
        tempData1 <<= (8 * j); //shift data bits into there proper position in the 64 bit long long
        allData |= tempData1;
    }
    
    int _startBitValue = 4, _numBitsValue = 4; //get the latch status bits
    //isolate the desired range of bits for comparison
    // (_numBitsValue - 1) makes it so the following, startBit = 13, numBites = 5 would mean you want up to bit 17, but (13 + 5) brings you to 18, therefore do (numBits - 1)
    int compareBits = (allData >> _startBitValue) & ~(~0 << ((_startBitValue+(_numBitsValue-1))-_startBitValue+1)); 

    int compareValue = 0; //value to watch (i.e. 0 for latch closed)
    
    //if the latch is closed
    if (compareBits == compareValue)
        return 1;
    
    return 0;  
}


/******************************************************************************/
/************                        getTemp                      *************/
/******************************************************************************/

float getTemp(){
    
    return device.readTemperature(); 
}

/******************************************************************************/
/************                        Main                         *************/
/******************************************************************************/

int main() {

    fullInit();
    
    int openTimeout = 10; //10sec timeout
    int switchReleaseTimeout = 2;
    int closeTimeout = 10;
    
    char dataBytes[4];
    dataBytes[0] = 0x00;
    dataBytes[1] = 0x00;
    dataBytes[2] = 0x00;

    
//    pc.setAddress ( 0, 3 );
    pc.printf( "<Press Button to start>\r\n" );
    
    
//    while (!buttons.readSel()){
//        if(buttons.readUp() && numCycles < 999999 ){
//            numCycles = numCycles + 100;
//            wait(0.2);
//        }
//        else if (buttons.readDown() && numCycles > 0 ){
//            numCycles = numCycles - 100;
//            wait(0.2);
//        }
//        pc.setAddress ( 0, 0 );
//        pc.printf( "<Num Cycles:%5d >" , numCycles );
//    }
     
//    pc.cls(); //clear the display
     
//    while(1){
//        currTemp = getTemp();
//        printf("%f",currTemp);
//        wait(1);
//    }
    
/******************************************************************************/
/************                   Cycle Loop                        *************/
/******************************************************************************/

    while (cycleCount <= numCycles){
   
//        pc.setAddress( 0, 0 );
        pc.printf( "Cycle %d of %3d\r\n", cycleCount, numCycles );
        avgCycTimer.reset();
        avgCycTimer.start();
        

/******************************************************************************/
/************                      Open                           *************/
/******************************************************************************/

        openDoorCommand(dataBytes);
    
//        pc.setAddress( 0, 2 );
        pc.printf( "STATUS: OPENING\r\n");
    
        waitSwitch(dataBytes, openTimeout);

/******************************************************************************/
/************                waitSwitchRelease                    *************/
/******************************************************************************/

        closeDoorCommand(dataBytes);
        
//        pc.setAddress( 0, 2 );
        pc.printf( "STATUS: SWITCH\r\n");
    
        waitSwitchRelease(dataBytes, switchReleaseTimeout); 

/******************************************************************************/
/************                waitSwitchRelease                    *************/
/******************************************************************************/
      
//        pc.setAddress( 0, 2 );
        pc.printf( "STATUS: CLOSING\r\n");
    
        waitLatch(dataBytes, closeTimeout);
  
/******************************************************************************/
/************                     End Cycle                       *************/
/******************************************************************************/

        totaltime += avgCycTimer.read();
//        pc.setAddress( 0, 1 );
        pc.printf( "Avg t(sec): %1.3f\r\n", (totaltime / cycleCount));
        wait(0.2);
        cycleCount++;
    }
}














    /* 
    pc.setAddress ( 0, 0 );
    pc.printf( "^ Up = Open" );
    pc.setAddress ( 0, 1 );
    pc.printf( "v Down = Close" );
    pc.setAddress ( 0, 2 );
    pc.printf( "> Right = Stop" );
    
     while(1){
        if (buttons.readUp()){
            dataBytes[3] = 0x03;
            can.write(CANMessage(534, dataBytes, 4)); // open the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
        }
        else if (buttons.readDown()){
            dataBytes[3] = 0x02;
            can.write(CANMessage(534, dataBytes, 4)); // close the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
        }
        else if (buttons.readRight()){
            dataBytes[3] = 0x01;
            can.write(CANMessage(534, dataBytes, 4)); // stop the door
            wait(0.1);
            dataBytes[3] = 0x00;
            can.write(CANMessage(534, dataBytes, 4)); // Set to IDLE
        }
    }*/
    /*
    while (1){
        //convert the data array into a single 64 bit value so that we can grab the range of bits we desire, no matter where they're located
        long long allData = 0; //((long long)*readMsg[k].data);
        for (int j = 0; j < 8; j++){
            long long tempData1 = (long long)readBuffer.data[j];
            tempData1 <<= (8 * j); //shift data bits into there proper position in the 64 bit long long
            allData |= tempData1;
        }
        
        int _startBitValue = 4, _numBitsValue = 4; //get the latch status bits
        //isolate the desired range of bits for comparison
        // (_numBitsValue - 1) makes it so the following, startBit = 13, numBites = 5 would mean you want up to bit 17, but (13 + 5) brings you to 18, therefore do (numBits - 1)
        int compareBits = (allData >> _startBitValue) & ~(~0 << ((_startBitValue+(_numBitsValue-1))-_startBitValue+1)); 
        
        int compareValue = 0; //value to watch (i.e. 0 for latch closed)
        //if the latch is closed
        if (compareBits == compareValue){
            pc.setAddress ( 0, 0 );
            pc.printf( "Latch: Closed      " );
        }
        else{
            pc.setAddress ( 0, 0 );
            pc.printf( "Latch: Not Closed " );
        }
    }*/