#include "Motor.hpp"

//Constructor
Motor::Motor(LineData lineData){
    this->errorFlag = 0;
    
    //Order of Line: Command, Local_Name, MOTOR, MotorID(A,B,C,D)
    //Since we can't return from a constructor, instead we'll flip a flag, and check it after we've added the device in interpretCommand
    if (lineData.numWords != 4)
        this->errorFlag = 1;

    string channel = lineData.word[3]; //Parameter is a single character, so dereference the point to the word

    if ((channel.compare("A") == 0) || (channel.compare("a") == 0))
            this->motor = BridgeDriver::MOTOR_A;
    else if ((channel.compare("B") == 0) || (channel.compare("b") == 0))
            this->motor = BridgeDriver::MOTOR_B;
    else if ((channel.compare("C") == 0) || (channel.compare("c") == 0))
            this->motor = BridgeDriver::MOTOR_C;
    else if ((channel.compare("D") == 0) || (channel.compare("d") == 0))
            this->motor = BridgeDriver::MOTOR_D;
    
    //MotorID not known
    else
        this->errorFlag = 1;
        
    bridges.enablePwm(this->motor, 1);
}
    
    
enum BridgeDriver::Motors Motor::getMotor(){     
    return this->motor;
}

//A line consists of [ __(Local_Name)__ __(function)__ __(parameter1)__ __(parameter2)__ __(parameter3)__ ... and so on]
int Motor::interpret(LineData &lineData){

    //Order of Line: Local_Name, Function_Name, Param1, Param2, Param3,.......
    string func = lineData.word[1];
        
    /******************************************************************************/
    /***                       <Func: enableBrake>                              ***/
    /******************************************************************************/
    if (func.compare("enableBrake") == 0){
        
        if (lineData.numWords != 3){
            ErrorOut("Incorrect number of parameters", lineData.lineNumber);
            return -1;
        }
        
        //Initialize and Convert Parameters
        string enable = lineData.word[2];
        int enableValue = 0;
        
        int numValuesFound = sscanf(enable.c_str(), "%d", &enableValue);
        if (numValuesFound < 1){
            ErrorOut("Parameter Unknown, enableBrake value can't be converted", lineData.lineNumber);
            return -1;
        }
        
        //All syntax checking done by this point, if Dummy then return success in order to check the code, no need to perform functionality
        if (DummyMode)
            return 0; //Function operated successfully but doesn't return a value
            
        bridges.enableBraking(getMotor(), enableValue);
    }
        
    /******************************************************************************/
    /***                       <Func: forceBrake>                               ***/
    /******************************************************************************/
    else if (func.compare("forceBrake") == 0){
        
        if (lineData.numWords != 2){
            ErrorOut("Incorrect number of parameters", lineData.lineNumber);
            return -1;
        }
        
        //All syntax checking done by this point, if Dummy then return success in order to check the code, no need to perform functionality
        if (DummyMode)
            return 0; //Function operated successfully but doesn't return a value
            
        bridges.forceBrake(getMotor());
        
        //Record the settings for Pause and Resume
        currDir = 0;
        currSpeed = 0;
    }
    
    
    /******************************************************************************/
    /***                           <Func: drive>                                ***/
    /******************************************************************************/
    else if (func.compare("drive") == 0){
        
        if (lineData.numWords != 4){
            ErrorOut("Incorrect number of parameters", lineData.lineNumber);
            return -1;
        }
                
        //Initialize Parameters
        string speed = lineData.word[2];
        string dir = lineData.word[3];
        
        //Initialize Convertion Variables if needed
        float speedValue;
        int dirValue = 0;
        
        //Convert string to usable values
        int numValuesFound = sscanf(speed.c_str(), "%f", &speedValue);
        if (numValuesFound < 1){
            ErrorOut("Parameter Unknown, speed value can't be converted", lineData.lineNumber);
            return -1;
        }
        
        //Speed is given as a percentage of 100, so convert it to the value needed for the bridge.drive function
        speedValue = speedValue / 100;
        
        
        if (speedValue <= 0 && speedValue > 1.0){
            ErrorOut("Speed Value must be between 0 - 100", lineData.lineNumber);
            return -1;
        }
        
        if (dir.compare("CC") == 0 || dir.compare("cc") == 0)
            dirValue = -1; //Turn Clockwise
        else if (dir.compare("C") == 0 || dir.compare("c") == 0)
            dirValue = 1; //Turn CounterClockwise
            
        else{
            ErrorOut("Direction Value must be C or CC", lineData.lineNumber);
            return -1;
        }
        
        //All syntax checking done by this point, if Dummy then return success in order to check the code, no need to perform functionality
        if (DummyMode)
            return 0; //Function operated successfully but doesn't return a value
            
        bridges.drive(getMotor(), dirValue, speedValue); //Turn on the Motor
        
        //Record the settings for Pause and Resume
        currDir = dirValue;
        currSpeed = speedValue;
    }
    
    
    /******************************************************************************/
    /****                           <Func: off>                                ****/
    /******************************************************************************/
    else if (func.compare("off") == 0){
        
        if (lineData.numWords != 2){
            ErrorOut("Incorrect number of parameters", lineData.lineNumber);
            return -1;
        }
        
        //All syntax checking done by this point, if Dummy then return success in order to check the code, no need to perform functionality
        if (DummyMode)
            return 0; //Function operated successfully but doesn't return a value
            
        off();
    }
    
    else {
        ErrorOut("Unknown Command for Motor Class", lineData.lineNumber);
        return -1;
    }
    
    return 0; //Return success as 0 since no condition had to be met
}



//For stopping the entire system if an error occurs, can be called from main
int Motor::off(void){
    bridges.drive(getMotor(), 0, 0); //Turn off the Motor
    
    //Record the settings for Pause and Resume, and exit
    currDir = 0;
    currSpeed = 0;
    return 0;
}


//Stop the motor without changing the previous known settings, so that it will be saved on resume
int Motor::pause(void){
    bridges.drive(getMotor(), 0, 0); //Turn off the Motor
    return 0;
}

//Resume the motor using its previously known settings
int Motor::resume(void){
    bridges.drive(getMotor(), this->currDir, this->currSpeed); //Resume Motor from it's last known state
    return 0;
}
