#include "mbed.h"
#include "MCP23017.h"
#include "TextLCD.h"    //if using diagnostic

#ifndef BRIDGEDRIVER_H
#define BRIDGEDRIVER_H


#define DEFAULT_PWM_PERIOD 0.02

//pin definitions
#define PIN_CH1 P1_18 //hardware PWM1[1]
#define PIN_CH2 P2_0  //hardware PWM1[1]
#define PIN_CH3 P1_20 //hardware PWM1[2]
#define PIN_CH4 P2_1  //hardware PWM1[2]
#define PIN_CH5 P1_21 //hardware PWM1[3]
#define PIN_CH6 P1_23 //hardware PWM1[4]
#define PIN_CH7 P1_24 //hardware PWM1[5]
#define PIN_CH8 P1_26 //hardware PWM1[6]


#define EN_ADDR  0x4C //i2c address of enable pin port expander. 0x40 for rev 1.0, 0x4C for rev 2.0
#define LED_ADDR 0x4E //i2c address of button and LED port expander

/** BridgeDriver class
 * For controlling the Test Controller bridges.
 */
class BridgeDriver{
     
    
    public:
        /** Channel pair labels
         *
         * MOTOR_A - Channel 1 and 2.
         * MOTOR_B - Channel 3 and 4.
         * MOTOR_C - Channel 5 and 6.
         * MOTOR_D - Channel 7 and 8.
         */
        enum  Motors{ MOTOR_A, 
                  MOTOR_B, 
                  MOTOR_C, 
                  MOTOR_D 
                };
        
        /** Create BridgeDriver object
         *
         * Enabling PWM for a channel pair links the channels so that they are controlled together. This should be used when connecting a motor between these channels.
         * Disabling PWM for a channel pair allows individual on/off control for each channel.
         *
         * MOTOR_A - Channel 1 and 2.
         * MOTOR_B - Channel 3 and 4.
         * MOTOR_C - Channel 5 and 6.
         * MOTOR_D - Channel 7 and 8.
         * @param *i2c A pointer to the I2C bus that the Enable control and LED port expanders are on.
         * @param enPwmA (Optional) 1 enables PWM on channels 1 and 2 for motor control, 0 disables PWM on channels 1 and 2 for solenoid control. Defaults to 0.
         * @param enPwmB (Optional) 1 enables PWM on channels 3 and 4 for motor control, 0 disables PWM on channels 3 and 4 for solenoid control. Defaults to 0.
         * @param enPwmC (Optional) 1 enables PWM on channels 5 and 6 for motor control, 0 disables PWM on channels 5 and 6 for solenoid control. Defaults to 0.
         * @param enPwmD (Optional) 1 enables PWM on channels 7 and 8 for motor control, 0 disables PWM on channels 7 and 8 for solenoid control. Defaults to 0.
         * @param enAddr (Optional) Address of the Enable control port expander. May be necessary if not using a rev 1.0 board. TODO: update for rev 2.0.
         * @param ledAddr (Optional) Address of the LED control port expander. May be necessary if not using rev 1.0 or 2.0 board.
         */
        BridgeDriver( I2C *i2c, uint8_t enPwmA = 0, uint8_t enPwmB = 0, uint8_t enPwmC = 0, uint8_t enPwmD = 0, uint8_t enAddr = EN_ADDR, uint8_t ledAddr = LED_ADDR);
        
        /** Destroy BridgeDriver object
         */
        ~BridgeDriver();
        
        /** Enables/disables PWM for channel pairs
         *
         * Enabling PWM for a channel pair links the channels so that they are controlled together. This should be used when connecting a motor between these channels.
         * Disabling PWM for a channel pair allows individual on/off control for each channel.
         *
         * MOTOR_A - Channel 1 and 2.
         * MOTOR_B - Channel 3 and 4.
         * MOTOR_C - Channel 5 and 6.
         * MOTOR_D - Channel 7 and 8.
         * @param enPwmA 1 enables PWM on channels 1 and 2 for motor control, 0 disables PWM on channels 1 and 2 for solenoid control.
         * @param enPwmB 1 enables PWM on channels 3 and 4 for motor control, 0 disables PWM on channels 3 and 4 for solenoid control.
         * @param enPwmC 1 enables PWM on channels 5 and 6 for motor control, 0 disables PWM on channels 5 and 6 for solenoid control.
         * @param enPwmD 1 enables PWM on channels 7 and 8 for motor control, 0 disables PWM on channels 7 and 8 for solenoid control.
         */
        void enablePwm(uint8_t enPwmA, uint8_t enPwmB, uint8_t enPwmC, uint8_t enPwmD);
        
        /** Enables/disables PWM for a given channel pair
         * 
         * Enabling PWM for a channel pair links the channels so that they are controlled together. This should be used when connecting a motor between these channels.
         * Disabling PWM for a channel pair allows individual on/off control for each channel.
         *
         * MOTOR_A - Channel 1 and 2.
         * MOTOR_B - Channel 3 and 4.
         * MOTOR_C - Channel 5 and 6.
         * MOTOR_D - Channel 7 and 8.
         * @param motor Motor (channel pair) to enale/disable PWM.
         * @param enPwm 1 enables PWM on channel pair for motor control, 0 disables PWM on channel pair for solenoid control.
         */
        void enablePwm(Motors motor, uint8_t enPwm);

        /** Enables/disables braking for channel pairs
         * @param enBrakeA 1 drives output to GND when off, 0 floats output when off.
         * @param enBrakeB 1 drives output to GND when off, 0 floats output when off.       
         * @param enBrakeC 1 drives output to GND when off, 0 floats output when off.
         * @param enBrakeD 1 drives output to GND when off, 0 floats output when off.
         */
        void enableBraking(uint8_t enBrakeA, uint8_t enBrakeB, uint8_t enBrakeC, uint8_t enBrakeD);
        
        /** Enables/disables braking for given channel pair
         * @param enBrakeA 1 drives output to GND when off, 0 floats output when off.
         */
        void enableBraking(Motors motor, uint8_t enBrake);
        
        /** Force a specific channel to GND without changing braking default
         * Will fail if PWM is enabled on given channel. Use forceBrake(Motors motor) to force braking on channels with PWM enabled.
         * @param ch Channel to drive to GND
         * @return
         *      0 on success,
         *      non 0 on failure.
         */
        int forceBrake(uint8_t ch);
        
        /** Force a specific motor (channel pair) to GND without changing braking default
         * Will fail if PWM is disabled on given channel. Use forceBrake(uint8_t ch) to force braking on channels with PWM disabled.
         * @param motor Motor (channel pair) to drive to GND
         * @return
         *      0 on success,
         *      non 0 on failure.
         */
        int forceBrake(Motors motor);
        
        /** Force a specific channel to float without changing braking default
         * Will float both channels of a motor (channel pair) if PWM is enabled for the channel.
         * @param ch Channel to float.
         * @return
         *      0 on success,
         *      non 0 on failure.
         */
        int forceFloat(uint8_t ch);
        
        /** Force a specific motor to float without changing braking default
         * Will fail if PWM is disabled on given channel. Use forceFloat(uint8_t ch) to force floating on channels with PWM disabled.
         * @param motor Motor (channel pair) to float.
         * @return
         *      0 on success,
         *      non 0 on failure.
         */
        int forceFloat(Motors motor);
        
        /** Set all outputs with a single byte (bitwise control).
         * Will fail if PWM is enabled for any motor (channel pair).
         * @param state Sets all the outputs. Each bit controls one channel.
         *      1 drives high,
         *      0 brakes (drives to GND) or floats depending on setting of enBrake.
         * @return
         *      -1 on failure,
         *      0 if all outputs are set to 0,
         *      1 if any output is set to 1.
         */
        int drive(uint8_t state);
        
        /** Set the output of one channel
         * @param ch Channel to set.
         * @param on State to set.
         *      1 drives high,
         *      0 brakes (drives to GND) or floats depending on setting of enBrake.
         * @return
         *      -1 on fauilure,
         *      0 if set to 0,
         *      1 if set to 1.
         */
        int drive(uint8_t ch, uint8_t on);
        
        /** Control the speed of a motor
         * @param motor Motor (channel pair) to control.
         * @param speed Speed to set motor to. Acceptable range: -1.0 to 1.0. Speed of 0.0 will coast or brake depending on setting of enBrake.
         * @return 
         *      -1 on failure,
         *      speed of set output on success.
         */
        float drive(Motors motor, float speed);
        
        /** Control the speed of a motor
         * @param motor Motor (channel pair) to control.
         * @param speed Speed to set motor to. Acceptable range: 0.0 to 1.0. Speed of 0.0 will coast or brake depending on setting of enBrake.
         * @param dir Set direction to drive motor.
         *      1: forward,
         *      -1: reverse,
         *      0: brake, irregardless of setting of enBrake or speed.
         * @return 
         *      -1 on failure,
         *      speed of set output on success.
         */
        float drive(Motors motor, int8_t dir, float speed);
        
        
        /** Display diagnostic information on the LCD
         * @param *lcd Pointer to TextLCD object to display info on.
         */
        void diagnostic(TextLCD_I2C *lcd);
        
        /** Change the PWM period
         * @param newPWMperiod The new PWM period. In seconds. System default is 0.0002 (5kHz).
         */
        void setPWMperiod(float newPWMperiod);
        
        
        
    private:
        DigitalOut *_d[8];
        PwmOut *_p[8];
        MCP23017 *_EnCtl;
        MCP23017 *_IO;
        I2C *_i2c;
        uint8_t _enAddr;
        uint8_t _ledAddr;
        uint8_t _pwm[4];        //enabled full bridges
        uint8_t _pwmCh;         //currently actie PWM pins
        int8_t  _dir[4];        //dir of 1 means lower # ch is driving, dir of -1 means higher number ch is driving
        uint8_t _braking[4]; 
        uint8_t _oldLedState;   //for setLed functions. Kept external to overload function
        float _PWMperiod;
        
        void enableCh(uint8_t ch, uint8_t en);
        void setLed(uint8_t ch, uint8_t en);
        void setLed(uint8_t state);
        void setPwm(uint8_t ch, uint8_t en);
        
        
        
};
    


#endif