#include "motor.h"


Motor::Motor(PinName pwm, PinName dir, PinName enc_a, PinName enc_b,
             int pulses_per_rev, bool invert_dir)
    :
    encoder_(enc_a, enc_b, NC, NULL, QEI::X4_ENCODING),
    dir_(dir),
    speed_(pwm),
    kPulsesPerRev_(pulses_per_rev),
    kInvertDir_(invert_dir)
{
    Stop();
    offset_ = 0;
    speed_.period_us(PWM_PERIOD_US);
}

/**
 * Turn off motor
 */
void Motor::Stop()
{
    power_ = false;
    dir_=0;
    speed_=0;
}

/**
 * Start motor
 */
void Motor::Start()
{
    power_ = true;
}

/**
 * Is the motor allowed to move?
 */
bool Motor::has_power()
{
    return power_;
}

/**
 * Apply PWM value in [-1,1] to motor
 * Direction changes accordingly
 * Only has effect when this.has_power()
 */
void Motor::set_pwm(double pwm)
{
    // only when allowed
    if (power_) {
        if (pwm<0) {
            dir_ = kInvertDir_;
        } else {
            dir_ = !kInvertDir_;
        }
        if (abs(pwm)>1) {
            speed_ = 1;
        } else {
            speed_ = abs(pwm);
        }
    }
}

/**
 * Get motor angle [deg]
 */
double Motor::get_angle()
{
    return offset_ + 360.0 * ((double)get_pulses()) / kPulsesPerRev_;
}

/**
 * Set the current motor angle to a given value [deg] (stored in offset)
 */
void Motor::set_angle(double angle)
{
    offset_ = angle - 360.0 * ((double)get_pulses()) / kPulsesPerRev_;
}




/**
 * get encoder pulses; only for internal use
 */
int Motor::get_pulses()
{
    return kInvertDir_?-encoder_.getPulses():encoder_.getPulses();
}


