#include "ref.h"
#include "robot.h"
#include "ui.h"

/**
 * Reference implementation
 */
namespace ref
{

// current reference angles
double theta_1 = 0;
double theta_2 = 0;

 // current demo coord
int i_demo_coord = 0;
  

/**
 * Get reference x-coordinate using reference angles and forward kinematics
 */
double get_ref_x()
{
    return robot::ForwardKinematicsX(theta_1,theta_2);
}

/**
 * Get reference x-coordinate using reference angles and forward kinematics
 */
double get_ref_y()
{
    return robot::ForwardKinematicsY(theta_1,theta_2);
}

/**
 * Update reference signal depending on robot state
 */
void UpdateReference()
{
    switch(robot::state) {
        case robot::CALIBRATION:
            CalibrationReference();
            break;
        case robot::HOMING:
            HomingReference();
            break;
        case robot::DEMO:
            DemoReference();
            break;
    }
}

/**
 * Generate reference profile to bring robotto calibration position;
 * at start-up the motor angles are '0' (relative encoders)
 */
void CalibrationReference()
{
    // Check when motors can no longer move farther
    bool is_ready_1 = (theta_1 - robot::get_angle(1) > kCalibrationError);
    bool is_ready_2 = (theta_2 - robot::get_angle(2) > kCalibrationError);

    // Turn off motor if end-stop reached
    // Motor 2 _should_ arrive first
    if (is_ready_1) {
        robot::m1.Stop();
    } else {
        theta_1 += kCalibrationOmegaStep;
    }

    // Turn off motor if end-stop reached
    if (is_ready_2) {
        robot::m2.Stop();
    } else {
        theta_2 += kCalibrationOmegaStep;
    }

    if (is_ready_1 & is_ready_2) {
        ui::serial.printf("  - Both motors at end-stop.\r\n");
        robot::Stop();
        // Set the calibration angles as current position
        robot::SetCalibrationAngles();
        robot::GoToNextState();
    }
}

/**
 * Bring robot to home position: STARTING FROM CALIBRATION POSITION!!
 */
void HomingReference()
{
    // First move motor 1
    double dist_1 = kOriginTheta1 - theta_1;

    if (abs(dist_1)<=kCalibrationOmegaStep) {
        // Motor 1 ready
        theta_1 = kOriginTheta1;

        // Only move motor 2 after motor 1 at home
        double dist_2 = kOriginTheta2 - theta_2;

        if (abs(dist_2)<=kCalibrationOmegaStep) {
            // motor 2 also ready
            theta_2 = kOriginTheta2;

            ui::serial.printf("  - Robot at home position.\r\n");
            robot::GoToNextState();
        } else {
            // Motor 2 still to move
            theta_2 += dist_2 / abs(dist_2) * kCalibrationOmegaStep;
        }


    } else {
        // Motor 1 still to move
        theta_1 += dist_1 / abs(dist_1) * kCalibrationOmegaStep;
    }
}

/**
 * Cycle through demo coordinates to display straight lines
 */
void DemoReference()
{
    // Compare current setpoint and goal coord and slowly move linearly in the
    // right direction (straight line 2D)

    double x = robot::ForwardKinematicsX(theta_1,theta_2);
    double y = robot::ForwardKinematicsY(theta_1,theta_2);

    double x_goal = kDemoCoords[0][i_demo_coord];
    double y_goal = kDemoCoords[1][i_demo_coord];

    double delta_x = x_goal - x;
    double delta_y = y_goal - y;

    double dist = sqrt(pow(delta_x,2) + pow(delta_y,2));

    // Compute next setpoint (x,y)
    double x_new = x;
    double y_new = y;

    if (dist<kMaxStep) {
        x_new = x_goal;
        y_new = y_goal;

        ui::serial.printf("  - Reached Demo Coord %d\r\n",i_demo_coord);
        i_demo_coord = ++i_demo_coord % kNumDemoCoords;

        if (i_demo_coord == 0) {
            robot::is_calibrated = false;
            robot::SwitchState(robot::CALIBRATION);
        }
    } else {
        x_new += delta_x / dist * kMaxStep;
        y_new += delta_y / dist * kMaxStep;
    }

    // Compute motor angles
    theta_1 = robot::InverseKinematicsTheta(x_new, y_new, 1);
    theta_2 = robot::InverseKinematicsTheta(x_new, y_new, 2);
}

/**
 * Set current robot position as reference position.
 */
void SetPositionAsReference()
{
    theta_1 = robot::get_angle(1);
    theta_2 = robot::get_angle(2);
}

} // end namespace ref