#include "controller.h"

/**
 * Constructor
 */
Controller::Controller(double kp, double ki, double kd,
           double t_sample, int memory_size)
    :
    kSampleTime_(t_sample),
    kMemorySize_(memory_size)
{
    // set controller gains
    set_kp(kp);
    set_ki(ki);
    set_kd(kd);

    // create memory buffer
    e_memory_ = new double[kMemorySize_];

    // reset memory
    Reset();
}

/**
 * Reset memory
 */
void Controller::Reset()
{
    // clear memory buffer for moving average
    for (int i=0; i<kMemorySize_; ++i) {
        e_memory_[i] = 0;
    }
    i_memory_cur_ = 0;

    // clear stored p,i,d errors
    e_p_ = 0;
    e_i_ = 0;
    e_d_ = 0;

    // clear moving average filted values for d action
    e_p_avg_cur_ = 0;
    e_p_avg_prev_ = 0;

    // clear control signal
    u_ = 0;
}

/**
 * Update the control signal. The control signal is given as output but can
 * also be reached through get_u(), so as to not again update the controller!
 */
double Controller::Update(double e)
{
    // Store current error in memory array
    i_memory_cur_ = ++i_memory_cur_ % kMemorySize_;
    e_memory_[i_memory_cur_] = e;

    // Compute moving average of error
    e_p_avg_prev_ = e_p_avg_cur_;
    e_p_avg_cur_ = 0;
    for (int i=0; i< kMemorySize_; ++i) {
        e_p_avg_cur_ += e_memory_[i] / kMemorySize_;
    }

    // Proportional error
    e_p_ = e;

    // Integral error
    e_i_ += e_p_ * kSampleTime_;

    // Differential error on moving average of error signal
    e_d_ = (e_p_avg_cur_ - e_p_avg_prev_) / kSampleTime_;

    // Control signal
    u_ = e_p_*kp_ + e_i_*ki_ + e_d_*kd_;

    return u_;
}