//
// MBED Application Board
// Lightweight C12832 LCD library
// 2014, Alexander Medvedev, @medvdv
//

// 
// BC12832 LCD
// Base geometry: 128 x 32 px
// X - [0..127] pixels
// Y - [0..3]   8 bit rows  
// 512 byte total buffer size
//
 
#define LCD_X       128     // In pixels
#define LCD_Y       4       // In rows by 8 pixels each
#define LCD_SIZE    512     // In bytes, = LCD_X*LCD_Y
    
//
// Proportional font  
//

typedef struct {
    unsigned char first_code, glyphs_total;   
    const unsigned char*  widths;
    const unsigned char** glyphs;    
} lcd128font; 
 
//
// Progress bar shape 
// Drawed as BFFF...FFUU...UUUE
// B - Begin  one
// F - Filled zone
// U - Empty zone
// E - End one
//

#define LCD_BAR_B   0x1c
#define LCD_BAR_F   0x3e 
#define LCD_BAR_U   0x22
#define LCD_BAR_E   0x1c

//
// Special Glyphs for menu / interface
//

#define LCD_RADIO_OFF       128
#define LCD_RADIO_ON        129
#define LCD_CHECK_OFF       130
#define LCD_CHECK_ON        131
#define LCD_SHORT_SPACE     132
#define LCD_PLAY            133
#define LCD_PAUSE           134
#define LCD_FORWARD         135
#define LCD_REVERSE         136
#define LCD_BEGIN           137
#define LCD_END             138
#define LCD_PLAY_PAUSE      139
#define LCD_CLOCK0          140
#define LCD_CLOCK1          141
#define LCD_CLOCK2          142
#define LCD_CLOCK3          143
#define LCD_CLOCK4          144
#define LCD_CLOCK5          145
#define LCD_CLOCK6          146
#define LCD_CLOCK7          147

//
// Row text align
//

#define LCD_ALIGN_LEFT      0
#define LCD_ALIGN_CENTER    1
#define LCD_ALIGN_RIGHT     2

//
// lcd128 class
//

class lcd128 {
    
    // Interface with lcd
    SPI spi;
    DigitalOut rst, cs, a0;
     
    // Current font    
    lcd128font font;
    
    // Write options
    bool invert;
    bool bold;
    bool underline;
            
    // buffer 
    unsigned char buffer[LCD_SIZE];
         
    // Cursor position
    int X, Y;
         
    // SPI writer with data / command       
    void write(unsigned char byte, bool cmd = false);

    public:
    
    // Supply LCD connected pin's here for your design   
    lcd128(PinName mosi = p5, PinName sclk = p7, PinName a0 = p8, PinName cs = p11, PinName rst = p6);

    // LCD control
    void Reset();                           // Reset LCD, configure defaults
    void Power(bool power = true);          // Power off / on
    void InverseMode(bool inverse = true);  // Invert LCD off / on

    // Buffer -> LCD
    void Update();                          // Update LCD from buffer, actually shows drawed
 
    // All buffer operations
    void Clear(int row = -1);               // Clear all buffer or one concrete row
    void InverseRow(int row = -1, unsigned char mask = 255);    // Inverse all buffer or one concrete row 
    
    // Write mode
    void Invert(bool invert = true);        // Switch inverting of chars
    void Bold(bool bold = true);            // Switch bold mode (repeat each char row twice)
    void Underline(bool underline = true);  // Switch underline mode (last pixel of row always on)
        
    // Cursor    
    void XY(int x = 0, int y = 0);          // Change write position X in pixels, Y in rows
    
    // Vertical 8bit pixel row
    void Write(unsigned char byte);                  // One 8bit row
    void Write(unsigned char byte, int count);       // One 8bit row * 'count' times
    void Write(unsigned char * data, int size);       // 'size' 8bit rows
    void Write2(unsigned char * data, int size);      // Bold: 'size' 8bit rows * 2   
    
    // Character drawing  
    void Character(char chr);               // Draw one font character (with invert and bold opt-s)
                                            // ! automaticaly does next line if char is wider then space left 
    int CharacterWidth(char chr);           // Calculate one character width (with bold opt-n)

    // String drawing
    void String(char* str);                 // Draw string proportionally
    int StringWidth(char* str);             // Calculate string width in px 
    
    // Row drawer
    void Row(int Y, char* str = "", bool inverse = false, int align = 0); // Clear one text row and draw string on it
    
    // Progress bar
    void Bar(int width, float fill = 0.0);  // Progress bar - one row, width pixels, fill - 0.0..1.0 
}; 
