#ifndef QT1070_H
#define QT1070_H

#include "mbed.h"

/** AT42QT1070 class
* Seven-channel QTouch® Touch Sensor Driver
*
* Example:
* @code
* #include "mbed.h"
* #include "QT1070.h"
*
* DigitalOut led(LED1);
* I2C i2c(PB_7,PB_6);
* QT1070 touch(&i2c, PA_11);
*
* int main() {
*     touch.guard(0);     
*     touch.aksAve(5,0,0);    // Disable key5 AVE=0
*     touch.aksAve(6,0,0);    // Disable key6 AVE=0
*
*     while(1) {
*         if(touch.change()){
*             led=!led;
*             printf("Key %d!!!!\r\n", touch.key());
*         }
*     }
* }
* @endcode
*/
    
class QT1070{
    public:
        enum state{
            I2C_ADDR = 0x36,    //I2C adress 
            KEY_NUM = 0x07,     //Number of channel
            AVEMAX = 31,        //Averaging maximum
            AKSMAX = 3          //Group maximum
        };
        
        //Register map
        enum reg{
            REG_CHIPID  = 0,
            REG_FIRMW   = 1,
            REG_STATUS  = 2,
            REG_KEYSTAT = 3,
            REG_KEYSIG0 = 4,
            REG_REFDAT0 = 18,
            REG_NTHR0   = 32,
            REG_AVEAKS0 = 39,
            REG_DI0     = 46,
            REG_FOMOGRD = 53,
            REG_LOWPWR  = 54,
            REG_MAXOND  = 55,
            REG_CALIB   = 56,
            REG_RESET   = 57   
        };
        
        /** Create PCA9633 instance + config method
        * @param *i2c initialized I2C bus to use
        * @param change PinName if connected 
        */
        QT1070(I2C *i2c, PinName change=NC);
         
        /** Get key reg.
        * @rerun 0 if not, 2^key if hit
        */
        char key(void);
        
        /** Check a change in status 
        * @return true in change occurs
        */
        bool change(void);
        
        /** Get signal.
        * @param key <0,6> key number
        */
        unsigned short signal(char key);
        
        /** Get reference.
        * @param key <0,6> key number
        */
        unsigned short ref(char key);
        
        /** Set/Get negative threshold.
        * @param key <0,6> key number
        * @param val <1,255> (Getter if 0)
        */
        char nthresh(char key, char val=0);
        
        /** Set/Get adjacent supresssor group and averaging.
        * @param key <0,6> key number
        * @param group <0,3> adjacent key suppressor group of the key
        * @param ave <1,31> averaged signal value
        */
        char aksAve(char key, char group=1, char ave=8);
        
        /** set DI.
        * @param key <0,6> key number
        * @param val <2,255> how measurement before confirm key hit
        */
        void detectInt(char key, char val);
        
        /** Set fastout and Maxcal.
        * @param fo set the digital integrator of 4 for all channel
        * @param maxcal if set only the key timeout was recalibrated   
        */
        void foMaxCal(bool fo, bool maxcal=false);
        
        /** Set a key to be guard channel (priority filtering).
        * @param key <0,6> key number
        */
        void guard(char key);
        
        /** Set interval between key measurment to reduce pwr.
        * @param ms <8,2040>[ms] in step of 8ms
        */
        void lowpwr(unsigned short ms);
        
        /** Set how long any key can be touch before it recalibrates itself.
        * @param ms <0,40800>[ms] in step of 160ms 
        */
        void maxOnDuration(unsigned short ms);
        
        /** Trig a calibration cycle
        */
        void calibrate(void);
        
        /** Trig a RESET
        */
        void reset(void);
    
    protected:     
        void write(char reg, char data);
        void writeW(char reg, unsigned short data);
        char read(char reg);
        unsigned short readW(char reg);

    private:
        I2C *_i2c;
        DigitalIn _change;
};
 
#endif
