#ifndef PCA9633_H
#define PCA9633_H

#include "mbed.h"
/** PCA9633 class.
 *  PCA9633 4-bit Fm+ I2C-bus LED driver.
 *  This driver ignore !OE pin functionality.
 *  This driver disable i2c LED All Call address.
 *  This driver disable i2c SUB addresing.  
 *
 * Example:
 * @code
  * #include "mbed.h"
 * #include "PCA9633.h"
 * I2C smbus(PB_7,PB_6);
 *
 * PCA9633 leds(&smbus);
 * 
 * int main() {
 *     leds.config(true, true); //The output config was openDrain and inverted
 *     leds.pwm(32,0);     //LED0 12.5%
 *     leds.pwm(64,1);     //LED1 25%     
 *     leds.pwm(128,2);    //LED2 50%    
 *     leds.pwm(255,3);    //LED3 100%
 *     leds.ledout(PCA9633::GROUP) //All led was dim/blink and single configurable
 *     led.blink(64,1.5)           //Leds blink each 1.5s with 25% duty cycle
 *   
 *     while(1) {
 *     }
 * }
 * @endcode
 */ 
class PCA9633{
    public:
    
        enum state{
            OFF = 0,        //LED driver is off (default power-up state) 
            FULL = 1,       //LED driver is fully on
            SINGLE = 2,     //LED driver individual brightness can be controlled 
                            //through its PWMx register.
            GROUP = 3,      //LED driver individual brightness and group 
                            //dimming/blinking can be controlled through its 
                            //PWMx register and the PCA9633_GRPPWM registers.
            ALL = 4,        //Select all LED for pos
            I2C_ADDR = 0x62 //Base I2C address 
        };
        /** Create PCA9633 instance + config method
        * @param I2C initialized I2C bus to use
        * @param addr I2C address default 0x62
        * @param invert invert output state
        * @param output was opendrain or push/pull
        */
        PCA9633(I2C *i2c, char addr=PCA9633::I2C_ADDR, bool invert=false, bool openDrain=false);
        
        /** Configure
        * @param invert invert output state
        * @param output was opendrain or push/pull
        */
        void config(bool invert, bool openDrain);
        
        /** Soft Reset ALL PCA9633 on I2C bus
        */
        void softReset(void);
        
        
        /** Set PWM value for 1 or all LED
        * @param bright <0,255> duty cycle
        * @param pos select output<0,3> (all if >3)
        */
        void pwm(char bright, char pos=PCA9633::ALL);
        
        /** Set general dimmer (if ledout = single | group)
        * Disable BLINK
        * @param val slowest pwm superimposed on led pwm
        */
        void dim(char val);
        
        /** Blinking led (if ledout = single | group)
        * Disable DIMMER
        * @param duty <0,255> duty cycle
        * @param period <0.041,10.73> is seconds
        */
        void blink(char duty, float period);
        
        /** Set output state
        * @param state off, full(on, nopwm), single(pwm, no dim|blink), group(pwm+dim|blink)
        * @param pos select output<0,3> (all if >3)
        */
        void ledout(char state=PCA9633::GROUP, char pos=PCA9633::ALL);
        
    private:
        I2C *_i2c;
        char _addr;
};

#endif
