#ifndef MAX31855_H
#define MAX31855_H

#include "mbed.h"
/** MAX31855 class.
 *  Used for read MAX31855 Cold-Junction Compensated Thermocouple-to-Digital Converter
 *
 * Example:
 * @code
 * DigitalOut led(LED2);
 * Serial pc(USBTX,USBRX);
 * MAX31855 therm(p5,p6,p7,p8);
 * 
 * int main() 
 * {
 *     while(1) 
 *     {
 *         pc.printf("T=%f;Chip=%f\r\n",therm.thermocouple(),therm.chip());
 *         led  = therm.fault();
 *         wait(0.5);
 *     }
 * }
 * @endcode
 */  
class MAX31855
{
public:
    /** Create MAX31855 instance connected to spi & ncs
    * @param mosi SPI master out slave in pin (MAX31855 is only read device)
    * @param miso SPI master in slave out pin
    * @param sck SPI clock pin
    * @param ncs pin to connect at CS input
    */
    MAX31855(PinName mosi, PinName miso, PinName sck, PinName ncs);
    
    /** Create MAX31855 instance connected to spi & ncs
    * @param SPI bus instance (MAX31855 is only read device)
    * @param ncs pin to connect at CS input
    */
    MAX31855(SPI& spi, PinName ncs);
    
    /**Get Thermocouple temperature
    * @returns temperature [°C]
    */    
    float thermocouple(void);
    
    /**Get Chip temperature
    * @returns temperature [°C]
    */       
    float chip(void);
    
    /**Check if thermocouple disconnected
    */
    bool opened(void);
    
    /**Check if an error
    */
    bool fault(void);
    
    /**Check if thermocouple short-circuited to Vcc
    */
    bool scToVcc(void);
    
    /**Check if thermocouple shorted-circuited to GND
    */
    bool scToGnd(void);

#ifdef MBED_OPERATORS
    /** An operator shorthand for thermocouple()
     *
     * The float() operator can be used as a shorthand for thermocouple() to simplify common code sequences
     *
     * Example:
     * @code
     * float x = temp.thermocouple();
     * float x = temp;
     *
     * if(temp.thermocouple() > 20.25) { ... }
     * if(temp > 20.25) { ... }
     * @endcode
     */
    operator float(){return thermocouple();}
#endif
    
protected:
    void read(void);
      
private:
    SPI         _spi;
    DigitalOut  _ncs;
    float       _t;
    float       _chip_t;
    bool        _fault;
    bool        _scv;
    bool        _scg;
    bool        _oc;
};

#endif