#ifndef ADS1248_H
#define ADS1248_H

#include "mbed.h"
/** ADS1248 class.
 *  Used for read ADS1248/7/6 Temperature ADC
 *
 */  
class ADS1248
{
    public:
        /** Create ADS1248 instance
        * @param SPI SPI bus to use
        * @param cs pin to connect at CS input
        * @param rdy pin connect at !rdy output
        * @param start pin connect at start inpout (pullup if unused)
        */
        ADS1248(SPI& spi, PinName cs, PinName rdy, PinName start = NC);
        
        /** register name list of ADS1248
        */
        enum reg{
            MUX0    = 0x00,
            VBIAS   = 0x01,
            MUX1    = 0x02,
            SYS0    = 0x03,
            OFC0    = 0x04,
            OFC1    = 0x05,
            OFC2    = 0x06,
            FSC0    = 0x07,
            FSC1    = 0x08,
            FSC2    = 0x09,
            IDAC0   = 0x0A,
            IDAC1   = 0x0B,
            GPIOCFG = 0x0C,
            GPIODIR = 0x0D,
            GPIODAT = 0x0E};
        
        /** commande list of ADS1248
        */ 
        enum cmd{
            WAKEUP  = 0x00,
            SLEEP   = 0x02,
            SYNC    = 0x04,
            RESET   = 0x06,
            NOP     = 0xff,
            RDATA   = 0x12,
            RDATAC  = 0x14,
            SDATAC  = 0x16,
            RREG    = 0x20,
            WREG    = 0x40,
            SYSOCAL = 0x60,
            SYSGCAL = 0x61,
            SELFOCAL= 0x62};
    
        /** Access to output start pin
        * @param en start pin state
        */                  
        void start(bool en);
        
        /** Wait ADS1248 to be ready (pooling method)
        */     
        void waitReady(void);
        
        /** Sleep cmd of ADS1248 (ask waitReady)
        * @param en for wakeup or sleep
        */       
        void sleep(bool en);
        
        /** Synchronisation cmd of ADS1248 
        */
        void sync(void);
        
        /** Reset cmd of ADS1248
        */
        void reset(void);
        
        /** Read data when conversion was finished (ask waitReady)
        * @return 24bit data 2's complement
        */
        int read(void);
        
        /** Ask read
        * @return 24bit data 2's complement
        */
        operator int();
        
        /** Read ADS1248 register
        * @param reg is register address
        * @return register value
        */
        unsigned char readReg(unsigned char reg);
        
        /** Read ADS1248 registers
        * @param reg is first register address
        * @param buff pointer on buffer to write
        * @param len lenght of data to read
        */
        void readReg(unsigned char reg, unsigned char *buff, int len);
        
        /** Write ADS1248 register
        * @param reg is register address
        * @param val value to write
        */
        void writeReg(unsigned char reg, unsigned char val);
        
        /** Write ADS1248 registers
        * @param reg is first register address
        * @param buff pointer on buffer to read
        * @param len lenght of data to write
        */
        void writeReg(unsigned char reg, const unsigned char *buff, int len);
        
        /** System Offset Calibration cmd of ADS1248 (ask waitReady)
        */
        void systemOffsetCal(void);
        
        /** System Gain Calibration cmd of ADS1248 (ask waitReady)
        */
        void systemGainCal(void);
        
        /** Self Offset Calibration cmd of ADS1248 (ask waitReady)
        */
        void selfOffsetCal(void);
        
   
    private:
        SPI& _spi;
        DigitalOut  _cs;
        DigitalIn   _rdy;
        DigitalOut  _start;
};

#endif
