/**
 * @brief       VAC.cpp
 * @details     For the electronic measurement of currents: DC, AC, pulsed, 
 *              mixed ..., with a galvanic Isolation between the primary circuit
 *              (high power) and the secondary circuit (electronic circuit).
 *              Function file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        25/September/2017
 * @version     25/September/2017    The ORIGIN
 * @pre         VAC ( VACUUMSCHMELZE ) current sensor-modules.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */

#include "VAC.h"


VAC::VAC ( PinName INPUT )
    : _INPUT               ( INPUT )
{
    _VAC_V_OFFSET   =   0.0;
}


VAC::~VAC()
{
}



/**
 * @brief       VAC_GetVoltage   ( float, float )
 *
 * @details     It performs a new voltage measurement.
 *
 * @param[in]    VoltageDivider:    If there is a voltage divider before the ADC pin ( if not VoltageDivider = 1 ).
 * @param[in]    ADC_Vref:          Voltage reference for the ADC.
 *
 * @param[out]   NaN.
 *
 *
 * @return       The actual voltage.
 *
 *
 * @author      Manuel Caballero
 * @date        25/September/2017
 * @version     25/September/2017   The ORIGIN
 * @pre         NaN.
 * @warning     NaN.
 */
VAC::Vector_VAC_voltage_t  VAC::VAC_GetVoltage ( float VoltageDivider, float ADC_Vref )
{
    Vector_VAC_voltage_t myAuxVoltage;
    

    myAuxVoltage.INPUT_Voltage    =   VoltageDivider * _INPUT.read() * ADC_Vref;
    

    return   myAuxVoltage;
}



/**
 * @brief       VAC_CalculateCurrent   ( Vector_VAC_voltage_t ,uint32_t, float )
 *
 * @details     It calculates the actual current.
 *
 * @param[in]    myVoltages:    Both voltages, INPUT and Vref voltages.
 * @param[in]    myKN:          Turns ratio.
 * @param[in]    myRM:          Measuring resistance.
 *
 * @param[out]   NaN.
 *
 *
 * @return       The calculated current.
 *
 *
 * @author      Manuel Caballero
 * @date        25/September/2017
 * @version     25/September/2017   The ORIGIN
 * @pre         VAC_GetVoltage function MUST be called first.
 * @warning     NaN.
 */
VAC::Vector_VAC_current_t  VAC::VAC_CalculateCurrent ( Vector_VAC_voltage_t myVoltages, uint32_t myIPN, float myRM )
{
    Vector_VAC_current_t myAuxCurrent;
    
    
    myAuxCurrent.Current    =   ( ( myVoltages.INPUT_Voltage + _VAC_V_OFFSET ) * myIPN ) / myRM;
    


    return   myAuxCurrent;
}



/**
 * @brief       VAC_SetAutoOffset ( float , float )
 *
 * @details     It calculates the offset automatically ( at 0A current ).
 *
 * @param[in]    VoltageDivider:    If there is a voltage divider before the ADC pin ( if not VoltageDivider = 1 ).
 * @param[in]    ADC_Vref:          Voltage reference for the ADC.
 *
 * @param[out]   NaN.
 *
 *
 * @return       The actual offset volatge.
 *
 *
 * @author      Manuel Caballero
 * @date        25/September/2017
 * @version     25/September/2017   The ORIGIN
 * @pre         NaN.
 * @warning     This test has to be perfomed at 0A ( no current through
 *              the sensor at all ).
 */
float  VAC::VAC_SetAutoOffset ( float VoltageDivider, float ADC_Vref )
{
    uint32_t i          =   0;
    float    myVoltage  =   0;
    
    
    // Calculate the average, get a new measurement every 0.25s
    for ( i = 0; i < CALIBRATION_AVERAGE; i++ ){
        myVoltage   +=   _INPUT.read();
        wait ( 0.25 );
    }
    
    myVoltage   /=   ( float )CALIBRATION_AVERAGE;
    
    
    // Store the offset
    _VAC_V_OFFSET   =   0.0 - ( VoltageDivider * myVoltage * ADC_Vref );        // Offset is calculated at 0A
    


    return   _VAC_V_OFFSET;
}
