/**
 * @brief       VAC.h
 * @details     For the electronic measurement of currents: DC, AC, pulsed,
 *              mixed ..., with a galvanic Isolation between the primary circuit
 *              (high power) and the secondary circuit (electronic circuit).
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        25/September/2017
 * @version     25/September/2017    The ORIGIN
 * @pre         VAC ( VACUUMSCHMELZE ) current sensor-modules.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef VAC_H
#define VAC_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "VAC.h"

VAC    myCurrentTransducer   ( p20 );
Serial pc                    ( USBTX, USBRX );

Ticker newReading;
DigitalOut myled1            ( LED1 );
DigitalOut myled2            ( LED2 );


VAC::Vector_VAC_voltage_t      myVoltages;
float                          myVoltageOffset =   0;


void readDATA ( void )
{
    VAC::Vector_VAC_current_t      myCurrent;

    myled2    =  1;

    myVoltages          =    myCurrentTransducer.VAC_GetVoltage       ( 1.0, 3.3 );                            // NO Voltage divider that is why the numer 1, ADC Vref = 3.3 V
    myCurrent           =    myCurrentTransducer.VAC_CalculateCurrent ( myVoltages, 2000, 100 );               // KN = 2000, RM = 100 Ohms

    pc.printf( "Vout: %0.5f V IP: %0.5f A\r\n", myVoltages.INPUT_Voltage, myCurrent.Current );

    myled2    =  0;
}


int main()
{
    pc.baud ( 115200 );

    // [ OPTIONAL ] CALIBRATION. It calculates the offset to calibrate the future measurements.
    myled1    =  1;

    // It reads PIN3 from the device. NOTE: This MUST be done at 0A current!!!
    myVoltageOffset  =   myCurrentTransducer.VAC_SetAutoOffset ( 1.0, 3.3 );    // NO Voltage divider that is why the numer 1, ADC Vref = 3.3 V
    myled1    =  0;

    pc.printf( "Voff: %0.5f V\r\n", myVoltageOffset );
    // CALIBRATION ends here


    newReading.attach( &readDATA, 0.5 );                                        // the address of the function to be attached ( readDATA ) and the interval ( 0.5s )


    // Let the callbacks take care of everything
    while(1)
    {
        sleep();
    }
}

*/


/*!
 Library for the VAC Current transducer.
*/
class VAC
{
public:
#define CALIBRATION_AVERAGE  10             // Change it if you wish to calculate the offset with less measurements



#ifndef VECTOR_STRUCT_H
#define VECTOR_STRUCT_H
    typedef struct {
        float Current;
    } Vector_VAC_current_t;

    typedef struct {
        float INPUT_Voltage;
    } Vector_VAC_voltage_t;
#endif


    /** Create an VAC object connected to the specified pins.
      *
      * @param INPUT   Vout from the device
      */
    VAC ( PinName INPUT  );

    /** Delete VAC object.
     */
    ~VAC();

    /** It gets the voltage.
     */
    Vector_VAC_voltage_t  VAC_GetVoltage          ( float VoltageDivider = 1.0, float ADC_Vref = 3.3 );

    /** It calculates the offset automatically ( at 0A current ).
     */
    float                 VAC_SetAutoOffset       ( float VoltageDivider = 1.0, float ADC_Vref = 3.3 );

    /** It calculates the current.
     */
    Vector_VAC_current_t  VAC_CalculateCurrent    ( Vector_VAC_voltage_t myVoltages, uint32_t myKN, float myRM );



private:
    AnalogIn     _INPUT;
    float        _VAC_V_OFFSET;
};

#endif
