/**
 * @brief       TMP102.h
 * @details     Low-Power Digital Temperature Sensor With SMBus and Two-Wire Serial.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        2/July/2018
 * @version     2/July/2018    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef TMP102_H
#define TMP102_H

#include "mbed.h"


/**
    Example:
@code
#include "mbed.h"
#include "TMP102.h"

TMP102 myTMP102         ( I2C_SDA, I2C_SCL, TMP102::TMP102_ADDRESS_A0_GROUND, 400000 );
Serial pc               ( USBTX, USBRX );

DigitalOut  myled       ( LED1 );
Ticker      newReading;

TMP102::TMP102_status_t         aux;
TMP102::TMP102_vector_data_t    myTMP102_Data;
uint32_t                        myState = 0;


void changeDATA ( void )
{
    myState = 1;
}


int main()
{
    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;


    // Read TLOW value
    aux  =   myTMP102.TMP102_Read_T_LOW_Register         ( &myTMP102_Data );

    // Read THIGH value
    aux  =   myTMP102.TMP102_Read_T_HIGH_Register        ( &myTMP102_Data );

    // Set polarity LOW
    aux  =   myTMP102.TMP102_SetPolarityAlertPinOutput   ( TMP102::TMP102_CONFIGURATION_POL_ALERT_PIN_ACTIVE_LOW );

    // Set Normal mode operation
    aux  =   myTMP102.TMP102_SetModeOperation            ( TMP102::TMP102_CONFIGURATION_EM_NORMAL_MODE_OPERATION );

    // Set Conversion Rate: 4Hz
    aux  =   myTMP102.TMP102_SetConversionRate           ( TMP102::TMP102_CONFIGURATION_CR_4_HZ );

    // Set Shutdown mode
    aux  =   myTMP102.TMP102_SetShutdownMode             ( TMP102::TMP102_CONFIGURATION_SD_ENABLED );


    newReading.attach( &changeDATA, 1 );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        myled = 1;

        if ( myState == 1 ) {
            // New reading
            aux  =   myTMP102.TMP102_TriggerSingleTemperatureConversion ();

            // Wait until the conversion is finished
            do {
                aux      =   myTMP102.TMP102_ReadConfigurationRegister ( &myTMP102_Data );
                wait_ms(1);
            } while( ( myTMP102_Data.ConfigurationRegister & TMP102::TMP102_CONFIGURATION_OS_MASK ) == TMP102::TMP102_CONFIGURATION_OS_BUSY );  // [TODO] Dangerous!!! The uC may get stuck here if something goes wrong!
                                                                                                                                                // [WORKAROUND] Insert a counter.

            // Read the new temperature value
            aux  =   myTMP102.TMP102_GetTemperature ( &myTMP102_Data );

            // Send data through the UART
            pc.printf( "Temperature: %0.4f C \r\n", myTMP102_Data.Temperature );
            myState  =   0;                                                             // Reset the variable
        }

        myled = 0;
    }
}
@endcode
*/


/*!
 Library for the TMP102 Low-Power Digital Temperature Sensor With SMBus and Two-Wire Serial.
*/
class TMP102
{
public:
    /**
    * @brief   DEFAULT ADDRESSES
    */
    typedef enum {
        TMP102_ADDRESS_A0_GROUND     =   ( 0b1001000 << 1 ),     /*!<   TMP102 I2C Address: A0 connected to Ground           */
        TMP102_ADDRESS_A0_VPOS       =   ( 0b1001001 << 1 ),     /*!<   TMP102 I2C Address: A0 connected to V+               */
        TMP102_ADDRESS_A0_SDA        =   ( 0b1001010 << 1 ),     /*!<   TMP102 I2C Address: A0 connected to SDA              */
        TMP102_ADDRESS_A0_SCL        =   ( 0b1001011 << 1 )      /*!<   TMP102 I2C Address: A0 connected to SCL              */
    } TMP102_addresses_t;


// REGISTERS
    /**
      * @brief   REGISTERS BYTE
      */
    typedef enum {
        TMP102_TEMPERATURE_REGISTER     =   0x00,               /*!<  Temperature Register      ( Read Only  )              */
        TMP102_CONFIGURATION_REGISTER   =   0x01,               /*!<  Configuration Register    ( Read/Write )              */
        TMP102_T_LOW_REGISTER           =   0x02,               /*!<  T LOW Register            ( Read/Write )              */
        TMP102_T_HIGH_REGISTER          =   0x03                /*!<  T HIGH Register           ( Read/Write )              */
    } TMP102_registers_byte_t;



// TEMPERATURE REGISTER
    /**
      * @brief   TEMPERATURE REGISTER
      */
    typedef enum {
        TMP102_RESOLUTION_12_BITS_MASK    =   0xFFF0,           /*!<  TMP102 configured as a 12-bit                         */
        TMP102_RESOLUTION_13_BITS_MASK    =   0xFFF8            /*!<  TMP102 configured as a 13-bit                         */
    } TMP102_temperature_register_t;



// CONFIGURATION REGISTER
    /**
      * @brief   SHUTDOWN MODE
      */
    typedef enum {
        TMP102_CONFIGURATION_SD_MASK        =   ( 1U << 8 ),    /*!<  Shutdown Mode Mask                                    */
        TMP102_CONFIGURATION_SD_ENABLED     =   ( 1U << 8 ),    /*!<  Shutdown Mode Enabled                                 */
        TMP102_CONFIGURATION_SD_DISABLED    =   ( 0U << 8 )     /*!<  Shutdown Mode Disabled                                */
    } TMP102_configuration_shutdown_mode_t;


    /**
      * @brief   THERMOSTAT MODE
      */
    typedef enum {
        TMP102_CONFIGURATION_TM_MASK            =   ( 1U << 9 ),    /*!<  Thermostat Mode Mask                           */
        TMP102_CONFIGURATION_TM_COMPARATOR_MODE =   ( 0U << 9 ),    /*!<  Thermostat Mode: Comparator mode               */
        TMP102_CONFIGURATION_TM_INTERRUPT_MODE  =   ( 1U << 9 )     /*!<  Thermostat Mode: Interrupt mode                */
    } TMP102_configuration_thermostat_mode_t;


    /**
      * @brief   POLARITY
      */
    typedef enum {
        TMP102_CONFIGURATION_POL_MASK                   =   ( 1U << 10 ),    /*!<  Polarity Mask                                                                      */
        TMP102_CONFIGURATION_POL_ALERT_PIN_ACTIVE_LOW   =   ( 0U << 10 ),    /*!<  Polarity: ALERT pin becomes active low ( default )                                 */
        TMP102_CONFIGURATION_POL_ALERT_PIN_ACTIVE_HIGH  =   ( 1U << 10 )     /*!<  Polarity: ALERT pin becomes active high and the state of the ALERT pin is inverted */
    } TMP102_configuration_polarity_t;


    /**
      * @brief   FAULT QUEUE
      */
    typedef enum {
        TMP102_CONFIGURATION_F1F0_MASK                   =   ( 0b11 << 11 ),    /*!<  Fault Queue Mask                                                               */
        TMP102_CONFIGURATION_F1F0_CONSECUTIVE_FAULTS_1   =   ( 0b00 << 11 ),    /*!<  Fault Queue: consecutive faults: 1                                             */
        TMP102_CONFIGURATION_F1F0_CONSECUTIVE_FAULTS_2   =   ( 0b01 << 11 ),    /*!<  Fault Queue: consecutive faults: 2                                             */
        TMP102_CONFIGURATION_F1F0_CONSECUTIVE_FAULTS_4   =   ( 0b10 << 11 ),    /*!<  Fault Queue: consecutive faults: 4                                             */
        TMP102_CONFIGURATION_F1F0_CONSECUTIVE_FAULTS_6   =   ( 0b11 << 11 )     /*!<  Fault Queue: consecutive faults: 6                                             */
    } TMP102_configuration_fault_queue_t;


    /**
      * @brief   CONVERTER RESOLUTION
      */
    typedef enum {
        TMP102_CONFIGURATION_R1R0_MASK                   =   ( 0b11 << 13 )     /*!<  Converter resolution bits Mask                                                 */
    } TMP102_configuration_converter_resolution_t;


    /**
      * @brief   ONE-SHOT
      */
    typedef enum {
        TMP102_CONFIGURATION_OS_MASK                                =   ( 1U << 15 ),      /*!<  One-Shot Mask                                                       */
        TMP102_CONFIGURATION_OS_START_SINGLE_TEMPERATURE_CONVERSION =   ( 1U << 15 ),      /*!<  One-Shot It starts a single temperature conversion                  */
        TMP102_CONFIGURATION_OS_BUSY                                =   ( 0U << 15 )       /*!<  One-Shot During the conversion, the OS bit reads '0'                */
    } TMP102_configuration_one_shot_t;



    /**
      * @brief   EXTENDED-MODE BIT
      */
    typedef enum {
        TMP102_CONFIGURATION_EM_MASK                                =   ( 1U << 4 ),      /*!<  Extended-Mode Mask                                                  */
        TMP102_CONFIGURATION_EM_NORMAL_MODE_OPERATION               =   ( 0U << 4 ),      /*!<  Extended-Mode: Normal Mode operation                                */
        TMP102_CONFIGURATION_EM_EXTENDED_MODE_OPERATION             =   ( 1U << 4 )       /*!<  Extended-Mode: Extended Mode operation                              */
    } TMP102_configuration_extended_mode_bit_t;


    /**
      * @brief   ALERT BIT
      */
    typedef enum {
        TMP102_CONFIGURATION_AL_MASK                                =   ( 1U << 5 )       /*!<  Alert Mask                                                          */
    } TMP102_configuration_alert_bit_t;


    /**
      * @brief   CONVERSION RATE
      */
    typedef enum {
        TMP102_CONFIGURATION_CR_MASK                                =   ( 0b11 << 6 ),    /*!<  Conversion Rate Mask                                                */
        TMP102_CONFIGURATION_CR_0_25_HZ                             =   ( 0b00 << 6 ),    /*!<  Conversion Rate: 0.25Hz                                             */
        TMP102_CONFIGURATION_CR_1_HZ                                =   ( 0b01 << 6 ),    /*!<  Conversion Rate: 1Hz                                                */
        TMP102_CONFIGURATION_CR_4_HZ                                =   ( 0b10 << 6 ),    /*!<  Conversion Rate: 4Hz ( default )                                    */
        TMP102_CONFIGURATION_CR_8_HZ                                =   ( 0b11 << 6 )     /*!<  Conversion Rate: 8Hz                                                */
    } TMP102_configuration_conversion_rate_t;



// HIGH-TEMPERATURE REGISTER
    /**
      * @brief   TEMPERATURE REGISTER
      */
    typedef enum {
        TMP102_THIGH_RESOLUTION_12_BITS_MASK    =   0xFFF0,                             /*!<  THIGH configured as a 12-bit                         */
        TMP102_THIGH_RESOLUTION_13_BITS_MASK    =   0xFFF8                              /*!<  THIGH configured as a 13-bit                         */
    } TMP102_high_temperature_register_t;



// LOW-TEMPERATURE REGISTER
    /**
      * @brief   TEMPERATURE REGISTER
      */
    typedef enum {
        TMP102_TLOW_RESOLUTION_12_BITS_MASK    =   0xFFF0,                             /*!<  TLOW configured as a 12-bit                            */
        TMP102_TLOW_RESOLUTION_13_BITS_MASK    =   0xFFF8                              /*!<  TLOW configured as a 13-bit                            */
    } TMP102_low_temperature_register_t;




#ifndef TMP102_VECTOR_STRUCT_H
#define TMP102_VECTOR_STRUCT_H
    typedef struct {
        float    Temperature;

        int16_t  TemperatureRegister;
        uint16_t ConfigurationRegister;
        int16_t  TLOW_Register;
        int16_t  THIGH_Register;
    } TMP102_vector_data_t;
#endif


    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        TMP102_SUCCESS    =       0,
        TMP102_FAILURE    =       1,
        I2C_SUCCESS       =       0                                             /*!<   I2C communication was fine        */
    } TMP102_status_t;




    /** Create an TMP102 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency
      */
    TMP102 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete TMP102 object.
     */
    ~TMP102();

    /** It gets the temperature register value ( raw temperature ).
    */
    TMP102_status_t  TMP102_ReadTemperatureRegister             ( TMP102_vector_data_t* myTemperatureRegister         );

    /** It gets the temperature value.
      */
    TMP102_status_t  TMP102_GetTemperature                      ( TMP102_vector_data_t* myTemperature                 );

    /** It gets the low temperature register value.
      */
    TMP102_status_t  TMP102_Read_T_LOW_Register                 ( TMP102_vector_data_t* myTLOW_Register               );

    /** It updates the low temperature register value.
      */
    TMP102_status_t  TMP102_Write_T_LOW_Register                ( TMP102_vector_data_t myTLOW_Register                );

    /** It gets the high temperature register value.
      */
    TMP102_status_t  TMP102_Read_T_HIGH_Register                ( TMP102_vector_data_t* myTHIGH_Register              );

    /** It updates the high temperature register value.
      */
    TMP102_status_t  TMP102_Write_T_HIGH_Register               ( TMP102_vector_data_t myTHIGH_Register               );

    /** It gets the configuration register value.
      */
    TMP102_status_t  TMP102_ReadConfigurationRegister           ( TMP102_vector_data_t* myConfigurationRegister       );

    /** It enables shutdown/continuous mode operation.
      */
    TMP102_status_t  TMP102_SetShutdownMode                     ( TMP102_configuration_shutdown_mode_t mySDMode       );

    /** It enables comparator/interrupt mode operation.
      */
    TMP102_status_t  TMP102_SetThermostatMode                   ( TMP102_configuration_thermostat_mode_t myTMMode     );

    /** The polarity bit allows the user to adjust the polarity of the ALERT pin output.
      */
    TMP102_status_t  TMP102_SetPolarityAlertPinOutput           ( TMP102_configuration_polarity_t myPOLMode           );

    /** The number of fault conditions required to generate an alert may be programmed using the fault queue.
      */
    TMP102_status_t  TMP102_SetConsecutiveFaultsQueue           ( TMP102_configuration_fault_queue_t myF1F0Mode       );

    /** When the device is in Shutdown Mode, writing a 1 to the OS bit starts a single temperature conversion.
      */
    TMP102_status_t  TMP102_TriggerSingleTemperatureConversion  ( void                                                );

    /** It sets the device into Normal/Extended mode operation.
      */
    TMP102_status_t  TMP102_SetModeOperation                    ( TMP102_configuration_extended_mode_bit_t myEMMode   );

    /** It sets the conversion rate for the device.
      */
    TMP102_status_t  TMP102_SetConversionRate                   ( TMP102_configuration_conversion_rate_t myCR         );




private:
    I2C      _i2c;
    uint32_t _TMP102_Addr;
};

#endif
