/**
 * @brief       SHT3X.h
 * @details     Humidity and Temperature Sensor.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef SHT3X_H_
#define SHT3X_H_

#include "mbed.h"


/**
    Example:

@code
#include "mbed.h"
#include "SHT3X.h"

SHT3X mySHT3X ( I2C_SDA, I2C_SCL, SHT3X::SHT3X_ADDRESS_B, 400000 );             // I2C_SDA | I2C_SCL
Serial pc     ( USBTX, USBRX );                                                 // tx, rx

DigitalOut  myled   ( LED1 );
Ticker      newAction;


//@brief Constants.


//@brief Variables.
volatile uint32_t myState;                                                      // State that indicates when to perform a new sample


//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    SHT3X::SHT3X_status_t aux;
    SHT3X::SHT3X_data_t   mySHT3X_Data;

    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;

    // Set a software reset
    aux  =   mySHT3X.SHT3X_SetSoftReset ();
    wait_ms(2U);

    // Clear the STATUS register
    aux  =   mySHT3X.SHT3X_ClearStatus ();

    // Disable the Heater
    aux  =   mySHT3X.SHT3X_SetHeater ( SHT3X::SHT3X_HEATER_DISABLE );

    // Do not perform periodic measurement
    aux  =   mySHT3X.SHT3X_SetBreakCommand ();

    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 1U );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        if ( myState == 1UL ) {
            myled = 1U;

            // Trigger to get a new data set, no clock stretching and repeatibility high
            aux  =   mySHT3X.SHT3X_OneShotTriggerAllData ( SHT3X::SHT3X_CLOCK_STRETCHING_DISABLED_REPEATABILITY_HIGH );
            wait_ms(15U);

            // Get all the data
            aux  =   mySHT3X.SHT3X_OneShotGetAllRawData ( &mySHT3X_Data.rawData );


            // Process all the data
            mySHT3X.SHT3X_ProccessData ( mySHT3X_Data.rawData, &mySHT3X_Data.data );

            // Send data through the UART
            pc.printf ( "T: %0.2f C, RH: %0.2f %%\r\n", mySHT3X_Data.data.temperature, mySHT3X_Data.data.relativeHumidity );


            // Reset the variables
            myState  =   0UL;
            myled    =   0U;
        }
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
// @return       N/A.
//
// @author      Manuel Caballero
// @date        25/March/2021
// @version     25/March/2021   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState  =   1UL;
}
@endcode
*/


/*!
 Library for the SHT3X Humidity and Temperature Sensor.
*/
class SHT3X
{
public:
    /**
    * @brief   DEFAULT ADDRESSES
    */
    typedef enum {
        SHT3X_ADDRESS_A     =   ( 0x44 << 1U ),             /*!<   SHT3X ADDR pin connected to logic low   */
        SHT3X_ADDRESS_B     =   ( 0x45 << 1U )              /*!<   SHT3X ADDR pin connected to logic high  */
    } SHT3X_address_t;



    /**
      * @brief   COMMAND REGISTERS
      */
    typedef enum {
        SHT3X_CLOCK_STRETCHING_ENABLED_REPEATABILITY_HIGH     = 0x2C06,   /*!<  High repeatability measurement with clock stretching enabled    */
        SHT3X_CLOCK_STRETCHING_ENABLED_REPEATABILITY_MEDIUM   = 0x2C0D,   /*!<  Medium repeatability measurement with clock stretching enabled  */
        SHT3X_CLOCK_STRETCHING_ENABLED_REPEATABILITY_LOW      = 0x2C10,   /*!<  Low repeatability measurement with clock stretching enabled     */

        SHT3X_CLOCK_STRETCHING_DISABLED_REPEATABILITY_HIGH    = 0x2400,   /*!<  High repeatability measurement with clock stretching disabled   */
        SHT3X_CLOCK_STRETCHING_DISABLED_REPEATABILITY_MEDIUM  = 0x240B,   /*!<  Medium repeatability measurement with clock stretching disabled */
        SHT3X_CLOCK_STRETCHING_DISABLED_REPEATABILITY_LOW     = 0x2416,   /*!<  Low repeatability measurement with clock stretching disabled    */
    } SHT3X_command_registers_single_shot_mode_t;


    typedef enum {
        SHT3X_HEATER_ENABLE                                   = 0x306D,   /*!<  Heater enable command                                           */
        SHT3X_HEATER_DISABLE                                  = 0x3066,   /*!<  Heater disable command                                          */
    } SHT3X_command_registers_heater_t;


    typedef enum {
        SHT3X_MPS_0_5_REPEATABILITY_HIGH                      = 0x2032,   /*!<  High repeatability 0.5 measurement per second                   */
        SHT3X_MPS_0_5_REPEATABILITY_MEDIUM                    = 0x2024,   /*!<  Medium repeatability 0.5 measurement per second                 */
        SHT3X_MPS_0_5_REPEATABILITY_LOW                       = 0x202F,   /*!<  Low repeatability 0.5 measurement per second                    */

        SHT3X_MPS_1_REPEATABILITY_HIGH                        = 0x2130,   /*!<  High repeatability 1 measurement per second                     */
        SHT3X_MPS_1_REPEATABILITY_MEDIUM                      = 0x2126,   /*!<  Medium repeatability 1 measurement per second                   */
        SHT3X_MPS_1_REPEATABILITY_LOW                         = 0x212D,   /*!<  Low repeatability 1 measurement per second                      */

        SHT3X_MPS_2_REPEATABILITY_HIGH                        = 0x2236,   /*!<  High repeatability 2 measurement per second                     */
        SHT3X_MPS_2_REPEATABILITY_MEDIUM                      = 0x2220,   /*!<  Medium repeatability 2 measurement per second                   */
        SHT3X_MPS_2_REPEATABILITY_LOW                         = 0x222B,   /*!<  Low repeatability 2 measurement per second                      */

        SHT3X_MPS_4_REPEATABILITY_HIGH                        = 0x2334,   /*!<  High repeatability 4 measurement per second                     */
        SHT3X_MPS_4_REPEATABILITY_MEDIUM                      = 0x2322,   /*!<  Medium repeatability 4 measurement per second                   */
        SHT3X_MPS_4_REPEATABILITY_LOW                         = 0x2329,   /*!<  Low repeatability 4 measurement per second                      */

        SHT3X_MPS_10_REPEATABILITY_HIGH                       = 0x2737,   /*!<  High repeatability 10 measurement per second                    */
        SHT3X_MPS_10_REPEATABILITY_MEDIUM                     = 0x2721,   /*!<  Medium repeatability 10 measurement per second                  */
        SHT3X_MPS_10_REPEATABILITY_LOW                        = 0x272A,   /*!<  Low repeatability 10 measurement per second                     */
    } SHT3X_command_registers_periodic_data_mode_t;


    typedef enum {
        SHT3X_FETCH_DATA                                      = 0xE000,   /*!<  Fetch data command                                              */
        SHT3X_PERIODIC_MESUREMENT_WITH_ART                    = 0x2B32,   /*!<  ART command                                                     */
        SHT3X_BREAK                                           = 0x3093,   /*!<  Break command                                                   */
        SHT3X_SOFT_RESET                                      = 0x30A2,   /*!<  Software reset command                                          */
        SHT3X_STATUS_REGISTER                                 = 0xF32D,   /*!<  Status command                                                  */
        SHT3X_CLEAR_STATUS_REGISTER                           = 0x3041    /*!<  Clear Status register command                                   */
    } SHT3X_command_registers_t;



// GENERAL CALL
    /**
      * @brief   RESET
      */
    typedef enum {
        SHT3X_GENERAL_CALL_RESET_ADDRESS_BYTE = 0x00,     /*!<  Reset. Address byte                                   */
        SHT3X_GENERAL_CALL_RESET_SECOND_BYTE  = 0x06      /*!<  Reset. Second byte                                    */
    } SHT3X_general_call_t;



    /**
     * @brief   STATUS REGISTER.
     *           NOTE: N/A.
     */
    /* ALERT_PENDING_STATUS <15>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_ALERT_PENDING_STATUS_MASK                        =   ( 1U << 15U ),    /*!<  ALERT_PENDING_STATUS mask                            */
        STATUS_ALERT_PENDING_STATUS_NO_PENDING_ALERTS           =   ( 0U << 15U ),    /*!<  No pending alerts                                    */
        STATUS_ALERT_PENDING_STATUS_AT_LEAST_ONE_PENDING_ALERT  =   ( 1U << 15U )     /*!<  At least one pending alert                 [Default] */
    } SHT3X_status_alert_pending_status_t;


    /* HEATER_STATUS <13>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_HEATER_STATUS_MASK                               =   ( 1U << 13U ),    /*!<  HEATER_STATUS mask                                   */
        STATUS_HEATER_OFF                                       =   ( 0U << 13U ),    /*!<  Heater off                                 [Default] */
        STATUS_HEATER_ON                                        =   ( 1U << 13U )     /*!<  Heater on                                            */
    } SHT3X_status_heater_status_t;


    /* RH_TRACKING_ALERT <11>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_RH_TRACKING_ALERT_MASK                           =   ( 1U << 11U ),    /*!<  RH_TRACKING_ALERT mask                               */
        STATUS_RH_TRACKING_ALERT_NO_ALERT                       =   ( 0U << 11U ),    /*!<  No alert                                   [Default] */
        STATUS_RH_TRACKING_ALERT_ALERT                          =   ( 1U << 11U )     /*!<  Alert                                                */
    } SHT3X_status_rh_tracking_alert_t;


    /* T_TRACKING_ALERT <10>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_T_TRACKING_ALERT_MASK                            =   ( 1U << 10U ),    /*!<  T_TRACKING_ALERT mask                                */
        STATUS_T_TRACKING_ALERT_NO_ALERT                        =   ( 0U << 10U ),    /*!<  No alert                                   [Default] */
        STATUS_T_TRACKING_ALERT_ALERT                           =   ( 1U << 10U )     /*!<  Alert                                                */
    } SHT3X_status_t_tracking_alert_t;


    /* SYSTEM_RESET_DETECTED <4>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_SYSTEM_RESET_DETECTED_MASK                       =   ( 1U << 4U ),     /*!<  SYSTEM_RESET_DETECTED mask                                              */
        STATUS_SYSTEM_RESET_DETECTED_NO_RESET_DETECTED          =   ( 0U << 4U ),     /*!<  No reset detected since last 'clear status register' command            */
        STATUS_SYSTEM_RESET_DETECTED_RESET_DETECTED             =   ( 1U << 4U )      /*!<  Reset detected (hard reset, soft reset command or supply fail)[Default] */
    } SHT3X_status_system_reset_detected_t;


    /* COMMAND_STATUS <1>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_COMMAND_STATUS_MASK                              =   ( 1U << 10U ),    /*!<  COMMAND_STATUS mask                                  */
        STATUS_COMMAND_STATUS_COMMAND_EXECUTED_SUCCESSFULLY     =   ( 0U << 10U ),    /*!<  Last command executed successfully         [Default] */
        STATUS_COMMAND_STATUS_LAST_COMMAND_NOT_PROCESSED        =   ( 1U << 10U )     /*!<  Last command not processed                           */
    } SHT3X_status_command_status_t;


    /* WRITE_DATA_CHECKSUM_STATUS <0>
     *    NOTE: N/A.
     */
    typedef enum {
        STATUS_WRITE_DATA_CHECKSUM_STATUS_MASK                  =   ( 1U << 0U ),     /*!<  COMMAND_STATUS mask                                  */
        STATUS_WRITE_DATA_CHECKSUM_STATUS_CHECKSUM_CORRECT      =   ( 0U << 0U ),     /*!<  Checksum of last write transfer was correct[Default] */
        STATUS_WRITE_DATA_CHECKSUM_STATUS_CHECKSUM_FAILED       =   ( 1U << 0U )      /*!<  Checksum of last write transfer failed               */
    } SHT3X_status_write_data_checksum_status_t;





#ifndef SHT3X_VECTOR_STRUCT_H
#define SHT3X_VECTOR_STRUCT_H
    /* RAW DATA  */
    typedef struct {
        uint16_t rawRelativeHumidity;
        uint16_t rawTemperature;

        uint8_t  temperatureCRC;
        uint8_t  relativeHumidityCRC;
    } SHT3X_raw_data_t;


    /* DATA: Temperature and Relative Humidity  */
    typedef struct {
        float relativeHumidity;
        float temperature;

        uint8_t  temperatureCRC;
        uint8_t  relativeHumidityCRC;
    } SHT3X_final_data_t;


    /* STATUS REGISTER  */
    typedef struct {
        uint16_t status;
        uint8_t  statusCRC;
    } SHT3X_status_data_t;



    /* USER: User's variables  */
    typedef struct {
        /* Output data   */
        SHT3X_raw_data_t    rawData;
        SHT3X_final_data_t  data;

        /* Status register  */
        SHT3X_status_data_t status;
    } SHT3X_data_t;
#endif





    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        SHT3X_SUCCESS     =       0,
        SHT3X_FAILURE     =       1,
        I2C_SUCCESS       =       0
    } SHT3X_status_t;



    /**
      * @brief   FUNCTION PROTOTYPES
      */
    /** Create an SHT3X object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    SHT3X ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete SHT3X object.
     */
    ~SHT3X();

    /** It triggers all the raw data in single shot mode.
      */
    SHT3X_status_t  SHT3X_OneShotTriggerAllData     ( SHT3X_command_registers_single_shot_mode_t mode       );

    /** It gets all the raw data in single shot mode.
      */
    SHT3X_status_t  SHT3X_OneShotGetAllRawData      ( SHT3X_raw_data_t* rawData                             );

    /** It sets periodic measurement with ART (accelerated response time).
      */
    SHT3X_status_t  SHT3X_SetPeriodicMeasurementART ( void                                                  );

    /** It sets the break command (stop periodic data acquisition mode).
      */
    SHT3X_status_t  SHT3X_SetBreakCommand           ( void                                                  );

    /** It perfoms a software reset.
      */
    SHT3X_status_t  SHT3X_SetSoftReset              ( void                                                  );

    /** It perfoms a reset through a general call address.
      */
    SHT3X_status_t  SHT3X_SetGeneralCallReset       ( void                                                  );

    /** It sets the heater.
      */
    SHT3X_status_t  SHT3X_SetHeater                 ( SHT3X_command_registers_heater_t heater               );

    /** It gets the status register.
      */
    SHT3X_status_t  SHT3X_GetStatus                 ( SHT3X_status_data_t* status                           );

    /** It clears the status register.
      */
    SHT3X_status_t  SHT3X_ClearStatus               ( void                                                  );

    /** It sets the periodic data aquisition mode.
      */
    SHT3X_status_t  SHT3X_SetPeriodicAquisitionMode ( SHT3X_command_registers_periodic_data_mode_t mo       );

    /** It gets the all raw data (in periodic aquisition mode).
      */
    SHT3X_status_t  SHT3X_GetAllRawDataFetchData    ( SHT3X_raw_data_t* rawData                             );

    /** It processes all data: Temperature and Relative Humidity.
      */
    void            SHT3X_ProccessData              ( SHT3X_raw_data_t rawData, SHT3X_final_data_t* data    );

private:
    I2C      _i2c;
    uint32_t _SHT3X_Addr;
};

#endif /* SHT3X_H */
