/**
 * @brief       SHT3X.c
 * @details     Humidity and Temperature Sensor.
 *              Functions file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */

 #include "SHT3X.h"


SHT3X::SHT3X ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c         ( sda, scl )
    , _SHT3X_Addr  ( addr )
{
    _i2c.frequency( freq );
}


SHT3X::~SHT3X()
{
}




 /**
 * @brief       SHT3X_OneShotTriggerAllData    ( SHT3X_command_registers_single_shot_mode_t  )
 * @details     It triggers all the raw data in single shot mode.
 *
 * @param[in]    mode:            Clock Stretching/No Stretching and the repeatability.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_OneShotTriggerAllData.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_OneShotTriggerAllData ( SHT3X_command_registers_single_shot_mode_t mode )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( mode >> 8U );
  cmd[1]   =  ( mode & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_OneShotGetAllRawData ( SHT3X_raw_data_t* )
 * @details     It gets all the raw data in single shot mode.
 *
 * @param[in]    N/A.
 *
 * @param[out]   rawData:         All the raw data and their CRC values.
 *
 *
 * @return      Status of SHT3X_OneShotGetAllRawData.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     SHT3X_OneShotTriggerAllData function must be called first. The user MUST respect the
 *              measurement duration as indicated in the datasheet (Table4.System timing specification, p7):
 *              
 *                                     Typ.     Max
 *              Low repeatability      2.5ms    4ms
 *              Medium repeatability   4.5ms    6ms
 *              High repeatability    12.5ms   15ms.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_OneShotGetAllRawData ( SHT3X_raw_data_t* rawData )
{
  char        cmd[6] = { 0U };
  uint32_t    aux;

  /* Read the register  */
  aux = _i2c.read ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  rawData->rawTemperature        =   cmd[0];
  rawData->rawTemperature      <<=   8U;
  rawData->rawTemperature       |=   cmd[1];

  rawData->temperatureCRC        =   cmd[2];

  rawData->rawRelativeHumidity   =   cmd[3];
  rawData->rawRelativeHumidity <<=   8U;
  rawData->rawRelativeHumidity  |=   cmd[4];

  rawData->relativeHumidityCRC   =   cmd[5];


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetPeriodicMeasurementART ( void )
 * @details     It sets periodic measurement with ART (accelerated response time).
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_SetPeriodicMeasurementART.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetPeriodicMeasurementART ( void )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_PERIODIC_MESUREMENT_WITH_ART >> 8U );
  cmd[1]   =  ( SHT3X_PERIODIC_MESUREMENT_WITH_ART & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetBreakCommand ( void )
 * @details     It sets the break command (stop periodic data acquisition mode).
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_SetBreakCommand.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetBreakCommand ( void )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_BREAK >> 8U );
  cmd[1]   =  ( SHT3X_BREAK & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetSoftReset ( void )
 * @details     It performs a software reset.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_SetSoftReset.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetSoftReset ( void )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_SOFT_RESET >> 8U );
  cmd[1]   =  ( SHT3X_SOFT_RESET & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetGeneralCallReset ( void )
 * @details     It perfoms a reset through a general call address.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_SetGeneralCallReset.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetGeneralCallReset ( void )
{
  char        cmd;
  uint32_t    aux;
  
  /* Write the register  */
  cmd   =  SHT3X_GENERAL_CALL_RESET_SECOND_BYTE;
  aux   =  _i2c.write ( SHT3X_GENERAL_CALL_RESET_ADDRESS_BYTE, &cmd, 1U, false );
  


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetHeater ( SHT3X_command_registers_heater_t )
 * @details     It sets the heater.
 *
 * @param[in]    heater:          Enable/Disable.
 *
 * @param[out]   N/A
 *
 *
 * @return      Status of SHT3X_SetHeater.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetHeater ( SHT3X_command_registers_heater_t heater )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( heater >> 8U );
  cmd[1]   =  ( heater & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );


  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_GetStatus ( SHT3X_status_data_t* )
 * @details     It gets the status register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   status:          The wole status register and its CRC.
 *
 *
 * @return      Status of SHT3X_GetStatus.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_GetStatus ( SHT3X_status_data_t* status )
{
  char        cmd[3] = { 0U };
  uint32_t    aux    = SHT3X_SUCCESS;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_STATUS_REGISTER >> 8U );
  cmd[1]   =  ( SHT3X_STATUS_REGISTER & 0xFF );
  aux     |=  _i2c.write ( _SHT3X_Addr, &cmd[0], 2U, true );
  
  /* Read the register   */
  aux     |=  _i2c.read ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  status->status     =   cmd[0];
  status->status   <<=   8U;
  status->status    |=   cmd[1];

  status->statusCRC  =   cmd[2];



  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_ClearStatus ( void )
 * @details     It clears the status register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of SHT3X_ClearStatus.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_ClearStatus ( void )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_CLEAR_STATUS_REGISTER >> 8U );
  cmd[1]   =  ( SHT3X_CLEAR_STATUS_REGISTER & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_SetPeriodicAquisitionMode ( SHT3X_command_registers_periodic_data_mode_t )
 * @details     It sets the periodic data aquisition mode.
 *
 * @param[in]    mo:              Periodic aquisition mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return      Status of SHT3X_SetPeriodicAquisitionMode.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_SetPeriodicAquisitionMode ( SHT3X_command_registers_periodic_data_mode_t mo )
{
  char        cmd[2] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( mo >> 8U );
  cmd[1]   =  ( mo & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_GetAllRawDataFetchData ( SHT3X_raw_data_t* )
 * @details     It gets the all raw data (in periodic aquisition mode).
 *
 * @param[in]    N/A.
 *
 * @param[out]   rawData:         Raw data and their CRC values.
 *
 *
 * @return      Status of SHT3X_GetAllRawDataFetchData.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
SHT3X::SHT3X_status_t  SHT3X::SHT3X_GetAllRawDataFetchData ( SHT3X_raw_data_t* rawData )
{
  char        cmd[6] = { 0U };
  uint32_t    aux;

  /* Write the register  */
  cmd[0]   =  ( SHT3X_FETCH_DATA >> 8U );
  cmd[1]   =  ( SHT3X_FETCH_DATA & 0xFF );
  aux      =  _i2c.write ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), true );
  
  /* Read the register   */
  aux      =  _i2c.read ( _SHT3X_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  rawData->rawTemperature        =   cmd[0];
  rawData->rawTemperature      <<=   8U;
  rawData->rawTemperature       |=   cmd[1];

  rawData->temperatureCRC        =   cmd[2];

  rawData->rawRelativeHumidity   =   cmd[3];
  rawData->rawRelativeHumidity <<=   8U;
  rawData->rawRelativeHumidity  |=   cmd[4];

  rawData->relativeHumidityCRC   =   cmd[5];



  if ( aux == I2C_SUCCESS )
  {
      return   SHT3X_SUCCESS;
  }
  else
  {
      return   SHT3X_FAILURE;
  }
}



/**
 * @brief       SHT3X_ProccessData ( SHT3X_raw_data_t , SHT3X_final_data_t* )
 * @details     It clears the status register.
 *
 * @param[in]    rawData:         Data to be processed.
 *
 * @param[out]   data:            Data result.
 *
 *
 * @return      N/A.
 *
 * @author      Manuel Caballero
 * @date        25/March/2021
 * @version     25/March/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
void  SHT3X::SHT3X_ProccessData ( SHT3X_raw_data_t rawData, SHT3X_final_data_t* data )
{
  /* Process the temperature value  */
  data->temperature  =   ( -45.0 + ( 175.0 * ( rawData.rawTemperature / ( 65536.0 - 1.0 ) ) ) );
  
  /* Process the relative humidity value  */
  data->relativeHumidity  =   100.0 * ( rawData.rawRelativeHumidity / ( 65536.0 - 1.0 ) );
}
