/**
 * @brief       SHT2X.h
 * @details     Humidity and Temperature Sensor IC.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        3/September/2018
 * @version     3/September/2018    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef SHT2X_H_
#define SHT2X_H_

#include "mbed.h"


/**
    Example:

@code
#include "mbed.h"
#include "SHT2X.h"

SHT2X mySHT2X ( I2C_SDA, I2C_SCL, SHT2X::SHT2X_ADDRESS, 400000 );
Serial pc     ( USBTX, USBRX );                                                 // tx, rx

DigitalOut  myled       ( LED1 );
Ticker      newAction;


//@brief Constants.
//
#define TX_BUFF_SIZE  64                                                        //   UART buffer size


//@brief Variables.
//
volatile uint32_t myState;                                                      //   State that indicates when to perform an ADC sample


//
// @brief   FUNCTION PROTOTYPES
//
void    changeDATA     ( void );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
//
int main()
{
    char     myBatteryMessage[5]  =  "";
    uint32_t aux;
    SHT2X::SHT2X_vector_data_t    mySHT2X_Data;

    pc.baud ( 115200 );


    myled   =   1;
    wait(3);
    myled   =   0;

    // Perform a software reset
    aux  =   mySHT2X.SHT2X_SoftReset ();
    wait_ms ( 15 );

    // Configure the device: Resolution Temperature: 14-bit and Humidity: 12-bit, Heater is disabled
    aux  =   mySHT2X.SHT2X_Conf            ( SHT2X::USER_REGISTER_RESOLUTION_12RH_14TEMP, SHT2X::USER_REGISTER_HEATER_DISABLED );

    // Get the serial number
    aux  =   mySHT2X.SHT2X_GetSerialNumber ( &mySHT2X_Data );

    // Transmit result through the UART
    pc.printf( "ID: %ld\r\n", mySHT2X_Data.SerialNumber );

    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 0.5f );                                      // the address of the function to be attached ( changeDATA ) and the interval ( 0.5s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        switch ( myState ) {
            default:
            case 1:
                myled = 1;

                // Trigger a new temperature measurement
                aux  =   mySHT2X.SHT2X_TriggerTemperature ( SHT2X::SHT2X_NO_HOLD_MASTER_MODE );
                break;

            case 2:
                // Get the temperature
                aux  =   mySHT2X.SHT2X_ReadTemperature    ( &mySHT2X_Data );

                // Trigger a new relative humidity measurement
                aux  =   mySHT2X.SHT2X_TriggerHumidity    ( SHT2X::SHT2X_NO_HOLD_MASTER_MODE );
                break;

            case 3:
                // Get the relative humidity
                aux  =   mySHT2X.SHT2X_ReadHumidity       ( &mySHT2X_Data );

                // Get the battery status
                aux  =   mySHT2X.SHT2X_BatteryStatus      ( &mySHT2X_Data );

                // Prepare the message for the battery status
                if ( ( mySHT2X_Data.BatteryStatus & SHT2X::USER_REGISTER_STATUS_END_BATTERY_MASK ) == SHT2X::USER_REGISTER_STATUS_END_BATTERY_HIGH_2V25 ) {
                    strcpy( myBatteryMessage, "GOOD" );
                } else {
                    strcpy( myBatteryMessage, "BAD" );
                }

                // Transmit result through the UART
                pc.printf ( "Temperature: %0.2f C | RH: %0.2f | Battery: %s\r\n", mySHT2X_Data.Temperature, mySHT2X_Data.RelativeHumidity, myBatteryMessage );


                // Reset the variables
                myState  =   0UL;
                myled    =   0;
                break;
        }
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
//
// @return       N/A..
//
//
// @author      Manuel Caballero
// @date        3/September/2018
// @version     3/September/2018   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState++;
}
@endcode
*/


/*!
 Library for the SHT2X Humidity and Temperature Sensor IC.
*/
class SHT2X
{
public:
    /**
    * @brief   DEFAULT ADDRESS
    */
    typedef enum {
        SHT2X_ADDRESS     =   ( 0b1000000 << 1U )                                   /*!<   SHT2X I2C Address                */
    } SHT2X_address_t;



    /**
      * @brief   COMMAND REGISTERS
      */
    typedef enum {
        SHT2X_TRIGGER_TEMPERATURE_MEASUREMENT_HOLD_MASTER      =   0xE3,            /*!<  Hold master                           */
        SHT2X_TRIGGER_HUMIDITY_MEASUREMENT_HOLD_MASTER         =   0xE5,            /*!<  Hold master                           */
        SHT2X_TRIGGER_TEMPERATURE_MEASUREMENT_NO_HOLD_MASTER   =   0xF3,            /*!<  No Hold master                        */
        SHT2X_TRIGGER_HUMIDITY_MEASUREMENT_NO_HOLD_MASTER      =   0xF5,            /*!<  No Hold master                        */
        SHT2X_WRITE_USER_REGISTER                              =   0xE6,            /*!<  Write user register                   */
        SHT2X_READ_USER_REGISTER                               =   0xE7,            /*!<  Read user register                    */
        SHT2X_SOFT_RESET                                       =   0xFE,            /*!<  Software reset                        */
        SHT2X_SERIAL_NUMBER_FIRST_MEMORY_ACCESS_MSB            =   0xFA,            /*!<  Serial number first memory access     */
        SHT2X_SERIAL_NUMBER_FIRST_MEMORY_ACCESS_LSB            =   0x0F,            /*!<  Serial number first memory access     */
        SHT2X_SERIAL_NUMBER_SECOND_MEMORY_ACCESS_MSB           =   0xFC,            /*!<  Serial number second memory access    */
        SHT2X_SERIAL_NUMBER_SECOND_MEMORY_ACCESS_LSB           =   0xC9             /*!<  Serial number second memory access    */
    } SHT2X_command_registers_t;



// MASTER MODE
    /**
      * @brief   MODE
      */
    typedef enum {
        SHT2X_HOLD_MASTER_MODE                 =   0x01,           /*!<  SHT2X HOLD MASTER MODE enabled                       */
        SHT2X_NO_HOLD_MASTER_MODE              =   0x00            /*!<  SHT2X NO HOLD MASTER MODE enabled                    */
    } SHT2X_master_mode_t;



// USER REGISTER
    /*
        NOTE:   Reserved bits must not be changed and default
                values of respective reserved bits may change over time without prior notice. Therefore, for any writing to user
                register, default values of reserved bits must be read first.
    */
    /**
      * @brief   RESOLUTION
      */
    typedef enum {
        USER_REGISTER_RESOLUTION_MASK                =   0x81,           /*!<  SHT2X Measurement Resolution                         */
        USER_REGISTER_RESOLUTION_12RH_14TEMP         =   0x00,           /*!<  SHT2X 12b RH 14b Temp.                               */
        USER_REGISTER_RESOLUTION_8RH_12TEMP          =   0x01,           /*!<  SHT2X 9b  RH 12b Temp.                               */
        USER_REGISTER_RESOLUTION_10RH_13TEMP         =   0x80,           /*!<  SHT2X 10b RH 13b Temp.                               */
        USER_REGISTER_RESOLUTION_11RH_11TEMP         =   0x81            /*!<  SHT2X 11b RH 11b Temp.                               */
    } SHT2X_measurement_resolution_t;



    /**
      * @brief   STATUS END OF BATTERY
      */
    typedef enum {
        USER_REGISTER_STATUS_END_BATTERY_MASK         =   0x40,           /*!<  End battery mask                                      */
        USER_REGISTER_STATUS_END_BATTERY_HIGH_2V25    =   0x00,           /*!<  VDD > 2.25V.                                          */
        USER_REGISTER_STATUS_END_BATTERY_LOW_2V25     =   0x40            /*!<  VDD < 2.25V.                                          */
    } SHT2X_status_end_battery_t;



    /**
      * @brief   ON-CHIP HEATER
      */
    typedef enum {
        USER_REGISTER_HEATER_MASK                     =   0x04,           /*!<  ON-chip heater mask                                   */
        USER_REGISTER_HEATER_ENABLED                  =   0x04,           /*!<  Heater enabled.                                       */
        USER_REGISTER_HEATER_DISABLED                 =   0x00            /*!<  Heater disabled.                                      */
    } SHT2X_on_chip_heater_t;



    /**
      * @brief   OTP RELOAD
      */
    typedef enum {
        USER_REGISTER_OTP_MASK                        =   0x02,           /*!<  OTP heater mask                                       */
        USER_REGISTER_OTP_ENABLED                     =   0x00,           /*!<  OTP enabled.                                          */
        USER_REGISTER_OTP_DISABLED                    =   0x02            /*!<  OTP disabled.                                         */
    } SHT2X_otp_t;



#ifndef SHT2X_VECTOR_STRUCT_H
#define SHT2X_VECTOR_STRUCT_H
    typedef struct {
        float    RelativeHumidity;
        float    Temperature;

        uint16_t RawRelativeHumidity;
        uint16_t RawTemperature;

        uint8_t  BatteryStatus;
        uint64_t SerialNumber;
    } SHT2X_vector_data_t;
#endif


    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        SHT2X_SUCCESS     =       0,
        SHT2X_FAILURE     =       1,
        I2C_SUCCESS       =       0                                           /*!<   I2C communication was fine     */
    } SHT2X_status_t;




    /** Create an SHT2X object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    SHT2X ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete SHT2X object.
     */
    ~SHT2X();

    /** It configures the SHT2X device.
    */
    SHT2X_status_t  SHT2X_Conf               ( SHT2X_measurement_resolution_t myResolution, SHT2X_on_chip_heater_t myHeater   );

    /** Rebooting the SHT2X sensor switching the power off and on again.
      */
    SHT2X_status_t  SHT2X_SoftReset          ( void                                                                           );

    /** It triggers a new temperature measurement.
      */
    SHT2X_status_t  SHT2X_TriggerTemperature ( SHT2X_master_mode_t myMode                                                     );

    /** It reads a new temperature measurement.
      */
    SHT2X_status_t  SHT2X_ReadTemperature    ( SHT2X_vector_data_t* myTemperature                                             );

    /** It reads a new raw temperature measurement.
      */
    SHT2X_status_t  SHT2X_ReadRawTemperature ( SHT2X_vector_data_t* myRawTemperature                                          );

    /** It triggers a new humidity measurement.
      */
    SHT2X_status_t  SHT2X_TriggerHumidity    ( SHT2X_master_mode_t myMode                                                     );

    /** It reads a new humidity measurement.
      */
    SHT2X_status_t  SHT2X_ReadHumidity       ( SHT2X_vector_data_t* myHumidity                                                );

    /** It reads a new raw humidity measurement.
      */
    SHT2X_status_t  SHT2X_ReadRawHumidity    ( SHT2X_vector_data_t* myRawHumidity                                             );

    /** It reads the user register to check the battery status.
      */
    SHT2X_status_t  SHT2X_BatteryStatus      ( SHT2X_vector_data_t* myBattStatus                                              );

    /** It gets the serial number ( Electronic Identification Code ).
      */
    SHT2X_status_t  SHT2X_GetSerialNumber    ( SHT2X_vector_data_t* mySerialNumber                                            );

private:
    I2C      _i2c;
    uint32_t _SHT2X_Addr;
};

#endif /* SHT2X_H */
