/**
 * @brief       PCF8591.h
 * @details     8-bit A/D and D/A converter.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     19/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef PCF8591_H
#define PCF8591_H

#include "mbed.h"


/**
    Example:

[TODO]

*/


/*!
 Library for the PCF8591 8-bit A/D and D/A converter.
*/
class PCF8591
{
public:
    /**
    * @brief   DEFAULT ADDRESSES
    */
    typedef enum {
        PCF8591_ADDRESS_0     =   ( 0x48 << 1 ),                   /*!<   A2 A1 A0: 000                                            */
        PCF8591_ADDRESS_1     =   ( 0x49 << 1 ),                   /*!<   A2 A1 A0: 001                                            */
        PCF8591_ADDRESS_2     =   ( 0x4A << 1 ),                   /*!<   A2 A1 A0: 010                                            */
        PCF8591_ADDRESS_3     =   ( 0x4B << 1 ),                   /*!<   A2 A1 A0: 011                                            */
        PCF8591_ADDRESS_4     =   ( 0x4C << 1 ),                   /*!<   A2 A1 A0: 100                                            */
        PCF8591_ADDRESS_5     =   ( 0x4D << 1 ),                   /*!<   A2 A1 A0: 101                                            */
        PCF8591_ADDRESS_6     =   ( 0x4E << 1 ),                   /*!<   A2 A1 A0: 110                                            */
        PCF8591_ADDRESS_7     =   ( 0x4F << 1 )                    /*!<   A2 A1 A0: 111                                            */
    } PCF8591_address_t;


// CONTROL BYTE
    /**
      * @brief   ANALOG OUTPUT ENABLE FLAG
      */
    typedef enum {
        PCF8591_DAC_ENABLED        =   1,                                /*!<  PCF8591 DAC is enabled.                                */
        PCF8591_DAC_DISABLE        =   0                                 /*!<  PCF8591 DAC is disabled                                */
    } PCF8591_dac_status_t;



    /**
      * @brief   ANALOG INPUT PROGRAMMING
      */
    typedef enum {
        PCF8591_FOUR_SINGLE_ENDED_INPUTS             =   0,              /*!<  PCF8591 Scale in kg.                                   */
        PCF8591_THREE_DIFFERENTIAL_INPUTS            =   1,              /*!<  PCF8591 Scale in  g.                                   */
        PCF8591_SINGLE_ENDED_AND_DIFFERENTIAL_MIXED  =   2,              /*!<  PCF8591 Scale in mg.                                   */
        PCF8591_TWO_DIFFERENTIAL_INPUTS              =   3               /*!<  PCF8591 Scale in ug.                                   */
    } PCF8591_analog_input_programming_t;



    /**
      * @brief   AUTO-INCREMENT FLAG
      */
    typedef enum {
        PCF8591_AUTO_INCREMENT_ENABLED               =   1,              /*!<  PCF8591 Auto-increment is enabled.                     */
        PCF8591_AUTO_INCREMENT_DISABLED              =   0               /*!<  PCF8591 Auto-increment is disabled                     */
    } PCF8591_auto_increment_status_t;



    /**
      * @brief   A/D CHANNEL NUMBER
      */
    typedef enum {
        PCF8591_CHANNEL_0                            =   0,              /*!<  PCF8591 Channel 0 Active                               */
        PCF8591_CHANNEL_1                            =   1,              /*!<  PCF8591 Channel 1 Active                               */
        PCF8591_CHANNEL_2                            =   2,              /*!<  PCF8591 Channel 2 Active                               */
        PCF8591_CHANNEL_3                            =   3               /*!<  PCF8591 Channel 3 Active                               */
    } PCF8591_channel_number_t;




#ifndef VECTOR_STRUCT_H
#define VECTOR_STRUCT_H
    typedef struct {
        uint8_t ADC_Channel_0;
        uint8_t ADC_Channel_1;
        uint8_t ADC_Channel_2;
        uint8_t ADC_Channel_3;
    } PCF8591_vector_data_t;
#endif



    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        PCF8591_SUCCESS     =       0,
        PCF8591_FAILURE     =       1,
        I2C_SUCCESS         =       0                           /*!<   I2C communication was fine                                                          */
    } PCF8591_status_t;




    /** Create an PCF8591 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    PCF8591 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete PCF8591 object.
     */
    ~PCF8591();

    /** It configures the ADC.
     */
    PCF8591_status_t  PCF8591_SetADC        ( PCF8591_analog_input_programming_t myAnalogInputs, PCF8591_auto_increment_status_t myAutoIncrement, PCF8591_channel_number_t myADCchannel );

    /** It performs a reading from the ADC device.
     */
    PCF8591_status_t  PCF8591_ReadADC       ( PCF8591_vector_data_t* myADC_Data );

    /** It configures the DAC.
     */
    PCF8591_status_t  PCF8591_SetDAC        ( PCF8591_dac_status_t myDAC_Status );

    /** It sets a new DAC value.
     */
    PCF8591_status_t  PCF8591_NewDACValue   ( uint8_t myNewDACValue );




private:
    I2C         i2c;
    uint32_t    PCF8591_Addr;
    uint32_t    _ANALOG_INPUT_PROGRAMMING;
    uint32_t    _AUTO_INCREMENT_STATUS;
    uint32_t    _CHANNEL_NUMBER;
    uint32_t    _DAC_STATUS;
};

#endif
