/**
 * @brief       PCF85063.cpp
 * @details     Tiny Real-Time Clock/calendar.
 *              Function file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ). All rights reserved.
 */

#include "PCF85063.h"


PCF85063::PCF85063 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c           ( sda, scl )
    , _PCF85063_Addr ( addr )
{
    _i2c.frequency( freq );
}


PCF85063::~PCF85063()
{
}



/**
 * @brief       PCF85063_SetTestMode ( PCF85063_control_1_ext_test_t )
 *
 * @details     It sets the external clock test mod.
 *
 * @param[in]    myEXT_TEST:        External clock test mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetTestMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetTestMode ( PCF85063_control_1_ext_test_t myEXT_TEST )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~CONTROL_1_EXT_TEST_MASK ) | myEXT_TEST );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetRTCMode ( PCF85063_control_1_stop_t )
 *
 * @details     It sets the RTC clock mode.
 *
 * @param[in]    mySTOP:          External clock test mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetRTCMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetRTCMode ( PCF85063_control_1_stop_t mySTOP )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~CONTROL_1_STOP_MASK ) | mySTOP );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SoftwareReset ( void )
 *
 * @details     It performs a software reset.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SoftwareReset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         For a software reset, 01011000 (58h) must be sent to register Control_1. Datasheet p6, 
 *              Table 6. Control_1 - control and status register 1 (address 00h) bit description.
 *
 *              The PCF85063TP resets to:
 *                -Time    — 00:00:00
 *                -Date    — 20000101
 *                -Weekday — Saturday
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SoftwareReset ( void )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  cmd[1]   =   0x58;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetCorrectionInterruptMode ( PCF85063_control_1_cie_t )
 *
 * @details     It sets the correction interrupt mode.
 *
 * @param[in]    myCIE:           Correction interrupt enable.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetCorrectionInterruptMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetCorrectionInterruptMode ( PCF85063_control_1_cie_t myCIE )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~CONTROL_1_CIE_MASK ) | myCIE );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_Set12_24_HourMode ( PCF85063_data_t )
 *
 * @details     It sets 12 or 24 hour mode.
 *
 * @param[in]    my12_24:         12 or 24 hour mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_Set12_24_HourMode.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_Set12_24_HourMode ( PCF85063_data_t my12_24 )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~CONTROL_1_12_24_MASK ) | my12_24.Time12H_24HMode );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetInternalOscillatorCapacitor ( PCF85063_control_1_cap_sel_t )
 *
 * @details     It sets the internal oscillator capacitor.
 *
 * @param[in]    myCAP_SEL:       Internal oscillator capacitor selection.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetInternalOscillatorCapacitor.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetInternalOscillatorCapacitor ( PCF85063_control_1_cap_sel_t myCAP_SEL )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_1;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~CONTROL_1_CAP_SEL_MASK ) | myCAP_SEL );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetMinuteInterrupts ( PCF85063_control_2_mi_t , PCF85063_control_2_hmi_t )
 *
 * @details     It enables/disables minute/half minute interrupt.
 *
 * @param[in]    myMI:            Minute interrupt.
 * @param[in]    myHMI:           Half minute interrupt.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetMinuteInterrupts.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetMinuteInterrupts ( PCF85063_control_2_mi_t myMI, PCF85063_control_2_hmi_t myHMI )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_2;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( ( cmd[1] & ~( CONTROL_2_MI_MASK | CONTROL_2_HMI_MASK ) ) | myMI | myHMI );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetTimerFlag ( PCF85063_control_2_tf_t* )
 *
 * @details     It gets the status of the timer flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTF:            Timer flag.
 *
 *
 * @return       Status of PCF85063_GetTimerFlag.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetTimerFlag ( PCF85063_control_2_tf_t* myTF )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_CONTROL_2;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  *myTF  =   (PCF85063_control_2_tf_t)( cmd & CONTROL_2_TF_MASK );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_ClearTimerFlag ( void )
 *
 * @details     It resets the status of the timer flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_ClearTimerFlag.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_ClearTimerFlag ( void )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_2;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( cmd[1] & ~CONTROL_2_TF_MASK );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetClockOutputFrequency ( PCF85063_control_2_cof_t )
 *
 * @details     It sets the clock output frequency.
 *
 * @param[in]    myCOF:           CLKOUT control.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetClockOutputFrequency.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetClockOutputFrequency ( PCF85063_control_2_cof_t myCOF )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_CONTROL_2;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( cmd[1] & ~CONTROL_2_COF_MASK ) | myCOF;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetOffset ( PCF85063_offset_mode_t , int8_t )
 *
 * @details     It sets the offset.
 *
 * @param[in]    myMODE:          Offset mode.
 * @param[in]    myOFFSET:        from +63 to -64.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetOffset ( PCF85063_offset_mode_t myMODE, int8_t myOFFSET )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;
  
  /*  Check offset values  */
  if ( ( myOFFSET < -64 ) || ( myOFFSET > 63 ) )
  {
    return   PCF85063_FAILURE;
  }
  else
  {
    /* Read the register   */
    cmd[0]   =   PCF85063_OFFSET;
    aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
    /* Mask it and update it with the new value  */
    cmd[1]   =   ( cmd[1] & ~( OFFSET_MODE_MASK | 0b01111111 ) ) | ( myMODE | myOFFSET );
    aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
  }



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_WriteByteRAM ( PCF85063_data_t )
 *
 * @details     It writes into the RAM byte register.
 *
 * @param[in]    myData:          Data to be written in the RAM byte register.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_WriteByteRAM.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_WriteByteRAM ( PCF85063_data_t myData )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_RAM_BYTE;
  cmd[1]   =   myData.ramByte;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_ReadByteRAM ( PCF85063_data_t* )
 *
 * @details     It writes into the RAM byte register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myData:          Data to be read from the RAM byte register.
 *
 *
 * @return       Status of PCF85063_ReadByteRAM.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_ReadByteRAM ( PCF85063_data_t* myData )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_RAM_BYTE;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myData->ramByte   =   cmd;




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_CheckOscillatorClockIntegrityFlag ( PCF85063_data_t* )
 *
 * @details     It checks oscillator clock integrity flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myOS:            CLKOUT control.
 *
 *
 * @return       Status of PCF85063_CheckOscillatorClockIntegrityFlag.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_CheckOscillatorClockIntegrityFlag ( PCF85063_data_t* myOS )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_SECONDS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myOS->os   =   (PCF85063_seconds_os_t)cmd;




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_ClearOscillatorClockIntegrityFlag ( void )
 *
 * @details     It clears oscillator clock integrity flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_ClearOscillatorClockIntegrityFlag.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_ClearOscillatorClockIntegrityFlag ( void )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_SECONDS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
  /* Mask it and update it with the new value  */
  cmd[1]   =   ( cmd[1] & ~SECONDS_OS_MASK );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetAM_PM_Indicator ( PCF85063_data_t )
 *
 * @details     It sets the AM/PM indicator ( only for 12-hour mode ).
 *
 * @param[in]    myAM_PM_Indicator: AM/PM indicator.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetAM_PM_Indicator.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     This function only works when the device is set for 12-hour mode, otherwise it
 *              will return FAILURE.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetAM_PM_Indicator ( PCF85063_data_t myAM_PM_Indicator )
{   
  char     cmd[2]  =  { 0U };
  uint32_t aux;
  
  /* This function only works when the device is set for 12-hour mode  */
  if ( myAM_PM_Indicator.Time12H_24HMode == CONTROL_1_12_24_24_HOUR_MODE )
  {
    return   PCF85063_FAILURE;
  }
  else
  {
    /* Read the register   */
    cmd[0]   =   PCF85063_HOURS;
    aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
    aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 1U );
  
    /* Mask it and update it with the new value  */
    cmd[1]   =   ( cmd[1] & ~HOURS_AMPM_MASK ) | myAM_PM_Indicator.TimeAM_PM_Mode;
    aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
  }



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetAM_PM_Indicator ( PCF85063_data_t* )
 *
 * @details     It gets the AM/PM indicator ( only for 12-hour mode ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAM_PM_Indicator: AM/PM indicator.
 *
 *
 * @return       Status of PCF85063_GetAM_PM_Indicator.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     This function only works when the device is set for 12-hour mode, otherwise it
 *              will return FAILURE.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetAM_PM_Indicator ( PCF85063_data_t* myAM_PM_Indicator )
{
  char     cmd  =  0U;
  uint32_t aux;
  
  /* This function only works when the device is set for 12-hour mode  */
  if ( myAM_PM_Indicator->Time12H_24HMode == CONTROL_1_12_24_24_HOUR_MODE )
  {
    return   PCF85063_FAILURE;
  }
  else
  {
    /* Read the register   */
    cmd      =   PCF85063_HOURS;
    aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
    aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
    /* Parse the data  */
    myAM_PM_Indicator->TimeAM_PM_Mode  =   (PCF85063_hours_ampm_t)cmd;
  }



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetDay ( PCF85063_data_t* )
 *
 * @details     It gets the day ( BCD format ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myActualDay:     Current day.
 *
 *
 * @return       Status of PCF85063_GetDay.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetDay ( PCF85063_data_t* myActualDay )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_DAYS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myActualDay->BCDday  =   ( cmd & ( DAYS_DAYS_TEN_PLACE_MASK | DAYS_DAYS_UNIT_PLACE_MASK ) );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetDay ( PCF85063_data_t )
 *
 * @details     It sets the day ( BCD format ).
 *
 * @param[in]    myNewDay:        New day ( BCD format ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetDay.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetDay ( PCF85063_data_t myNewDay )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_DAYS;
  cmd[1]   =   ( myNewDay.BCDday & ( DAYS_DAYS_TEN_PLACE_MASK | DAYS_DAYS_UNIT_PLACE_MASK ) );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetWeekday ( PCF85063_data_t* )
 *
 * @details     It gets the weekday.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myActualWeekday: Current weekday.
 *
 *
 * @return       Status of PCF85063_GetWeekday.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetWeekday ( PCF85063_data_t* myActualWeekday )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_WEEKDAYS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myActualWeekday->weekday  =   (PCF85063_weekdays_weekdays_t)( cmd & WEEKDAYS_WEEKDAYS_MASK );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetWeekday ( PCF85063_data_t )
 *
 * @details     It sets the weekday.
 *
 * @param[in]    myNewWeekday:    New weekday.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetWeekday.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetWeekday ( PCF85063_data_t myNewWeekday )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_WEEKDAYS;
  cmd[1]   =   myNewWeekday.weekday;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetMonth ( PCF85063_data_t* )
 *
 * @details     It gets the month ( BCD format ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myActualMonth:   Current month.
 *
 *
 * @return       Status of PCF85063_GetMonth.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetMonth ( PCF85063_data_t* myActualMonth )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_MONTHS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myActualMonth->BCDmonth  =   (PCF85063_months_months_t)( cmd & MONTHS_MONTHS_MASK );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetMonth ( PCF85063_data_t )
 *
 * @details     It sets the month ( BCD format ).
 *
 * @param[in]    myNewMonth:      New month.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetMonth.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetMonth ( PCF85063_data_t myNewMonth )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_MONTHS;
  cmd[1]   =   myNewMonth.BCDmonth;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetYear ( PCF85063_data_t* )
 *
 * @details     It gets the year ( BCD format ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myActualYear:    Current year.
 *
 *
 * @return       Status of PCF85063_GetYear.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetYear ( PCF85063_data_t* myActualYear )
{
  char     cmd  =  0U;
  uint32_t aux;

  /* Read the register   */
  cmd      =   PCF85063_YEARS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd, 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myActualYear->BCDyear  =   cmd;




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetYear ( PCF85063_data_t )
 *
 * @details     It sets the year ( BCD format ).
 *
 * @param[in]    myNewYear:       New year.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetYear.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetYear ( PCF85063_data_t myNewYear )
{
  char     cmd[2]  =  { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]   =   PCF85063_YEARS;
  cmd[1]   =   myNewYear.BCDyear;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_GetTime ( PCF85063_data_t* )
 *
 * @details     It gets the time ( BCD format ).
 *
 * @param[in]    N/A.
 *
 * @param[out]   myActualTime:    Current time ( BCD format: HHMMSS ).
 *
 *
 * @return       Status of PCF85063_GetTime.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_GetTime ( PCF85063_data_t* myActualTime )
{
  char     cmd[3]  =  { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_SECONDS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  if ( myActualTime->Time12H_24HMode == CONTROL_1_12_24_24_HOUR_MODE )
  {
    myActualTime->BCDtime   =   ( cmd[2] & ( HOURS_24_HOUR_MODE_TEN_PLACE_MASK | HOURS_HOURS_UNIT_PLACE_MASK ) );
  }
  else
  {
    myActualTime->BCDtime   =   ( cmd[2] & ( HOURS_12_HOUR_MODE_TEN_PLACE_MASK | HOURS_HOURS_UNIT_PLACE_MASK ) );
  }

  myActualTime->BCDtime <<=   8U;
  myActualTime->BCDtime  |=   ( cmd[1] & ( MINUTES_MINUTES_TEN_PLACE_MASK | MINUTES_MINUTES_UNIT_PLACE_MASK ) );
  myActualTime->BCDtime <<=   8U;
  myActualTime->BCDtime  |=   ( cmd[0] & ( SECONDS_SECONDS_TEN_PLACE_MASK | SECONDS_SECONDS_UNIT_PLACE_MASK ) );



  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}



/**
 * @brief       PCF85063_SetTime ( PCF85063_data_t )
 *
 * @details     It sets the time ( BCD format ).
 *
 * @param[in]    myNewTime:       New current time ( BCD format: HHMMSS ).
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of PCF85063_SetTime.
 *
 *
 * @author      Manuel Caballero
 * @date        19/February/2019
 * @version     19/February/2019     The ORIGIN
 * @pre         N/A
 * @warning     N/A.
 */
PCF85063::PCF85063_status_t  PCF85063::PCF85063_SetTime ( PCF85063_data_t myNewTime )
{
  char     cmd[4]  =  { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]   =   PCF85063_SECONDS;
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], 1U, true );
  aux      =   _i2c.read  ( _PCF85063_Addr, &cmd[1], 3U );
  
  /* Mask it and update the register  */
  cmd[1]  &=  ~( SECONDS_SECONDS_TEN_PLACE_MASK | SECONDS_SECONDS_UNIT_PLACE_MASK );
  cmd[1]  |=   (char)( myNewTime.BCDtime & 0x0000FF );
  cmd[2]  &=  ~( MINUTES_MINUTES_TEN_PLACE_MASK | MINUTES_MINUTES_UNIT_PLACE_MASK );
  cmd[2]  |=   (char)( ( myNewTime.BCDtime & 0x00FF00 ) >> 8U );

  if ( myNewTime.Time12H_24HMode == CONTROL_1_12_24_24_HOUR_MODE )
  {
    cmd[3]  &=  ~( HOURS_24_HOUR_MODE_TEN_PLACE_MASK | HOURS_HOURS_UNIT_PLACE_MASK );
  }
  else
  {
    cmd[3]  &=  ~( HOURS_12_HOUR_MODE_TEN_PLACE_MASK | HOURS_HOURS_UNIT_PLACE_MASK );
  }
  cmd[3]  |=   (char)( ( myNewTime.BCDtime & 0xFF0000 ) >> 16U );
  aux      =   _i2c.write ( _PCF85063_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   PCF85063_SUCCESS;
  }
  else
  {
    return   PCF85063_FAILURE;
  }
}