/**
 * @brief       MicroForce_FMA.h
 * @details     MicroForce Sensors, Compensated/Amplified.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        30/June/2021
 * @version     30/June/2021    The ORIGIN
 * @pre         N/A.
 * @warning     This driver only supports SPI communication.
 * @pre         This code belongs to Nimbus Centre ( https://www.nimbus.cit.ie ).
 */
#ifndef MicroForce_FMA_H
#define MicroForce_FMA_H

#include "mbed.h"
/**
    Example:
@code
#include "mbed.h"
#include "MicroForce_FMA.h"

MicroForce_FMA myMicroForce_FMA ( PB_5, PB_4, PB_3, PB_8, 500000 );             // MOSI: PB_5 | MISO: PB_4 | SCLK: PB_3 | CS: PB_8 | FREQ: 500kHz;
Serial pc     ( USBTX, USBRX );                                                 // tx, rx

DigitalOut  myled   ( LED1 );
Ticker      newAction;


//@brief Constants.


//@brief Variables.
volatile uint32_t myState;                                                      // State that indicates when to perform a new sample


//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    MicroForce_FMA::MicroForce_FMA_status_t aux;
    MicroForce_FMA::MicroForce_FMA_data_t   myMicroForce_FMA_Data;

    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;


    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 1U );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 5s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        if ( myState == 1UL ) {
            myled = 1U;

            do {
                // Get all the data  
                aux  =   myMicroForce_FMA.MicroForce_FMA_GetAllRawData ( &myMicroForce_FMA_Data.status, &myMicroForce_FMA_Data.force.raw_bridge_data, &myMicroForce_FMA_Data.temperature.raw_11bit_temperature );
                wait_ms(1U);
            } while( myMicroForce_FMA_Data.status.status_bits != MicroForce_FMA::MicroForce_FMA_STATUS_BITS_NORMAL_OPERATION );

            // It processes all the data  
            myMicroForce_FMA_Data.transfer_function             =  MicroForce_FMA::MicroForce_FMA_TRANSFER_FUNCTION_20_TO_80;
            myMicroForce_FMA_Data.force_range                   =  FMA_FORCE_RANGE_15_N;
            myMicroForce_FMA_Data.force.bridge_data             =  myMicroForce_FMA.MicroForce_FMA_CalculateForce ( myMicroForce_FMA_Data.transfer_function, myMicroForce_FMA_Data.force_range, myMicroForce_FMA_Data.force.raw_bridge_data );
            myMicroForce_FMA_Data.temperature.temperature_data  =  myMicroForce_FMA.MicroForce_FMA_Calculate11bitTemperature ( myMicroForce_FMA_Data.temperature.raw_11bit_temperature );


            // Send data through the UART
            pc.printf ( "Force: %d N | Temp: %d C\r\n", (uint32_t)myMicroForce_FMA_Data.force.bridge_data, (uint32_t)myMicroForce_FMA_Data.temperature.temperature_data );

            // Reset the variables
            myState  =   0UL;
            myled    =   0U;
        }
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
// @return       N/A.
//
// @author      Manuel Caballero
// @date        30/June/2021
// @version     30/June/2021   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState  =   1UL;
}
@endcode
*/


/*!
 Library for the MicroForce_FMA. MicroForce Sensors, Compensated/Amplified.
*/
class MicroForce_FMA
{
public:
    /**
    * @brief   STATUS BITS
    */
    typedef enum {
        MicroForce_FMA_STATUS_BITS_MASK                   =   ( 0b11 << 6 ),      /*!<   STATUS BITS mask                 */
        MicroForce_FMA_STATUS_BITS_NORMAL_OPERATION       =   ( 0b00 << 6 ),      /*!<   Normal operation, valid data     */
        MicroForce_FMA_STATUS_BITS_DEVICE_IN_COMMAND_MODE =   ( 0b01 << 6 ),      /*!<   Device in command mode           */
        MicroForce_FMA_STATUS_BITS_STALE_DATA             =   ( 0b10 << 6 ),      /*!<   Stale data                       */
        MicroForce_FMA_STATUS_BITS_DIAGNOSTIC_CONDITION   =   ( 0b11 << 6 )       /*!<   Diagnostic condition             */
    } MicroForce_FMA_status_bits_t;


    /**
      * @brief   TRANSFER FUCNTION
      */
    typedef enum {
        MicroForce_FMA_TRANSFER_FUNCTION_10_TO_90         =   0U,                 /*!<   Tranfer function A: 10% to 90%   */
        MicroForce_FMA_TRANSFER_FUNCTION_20_TO_80         =   1U                  /*!<   Tranfer function C: 20% to 80%   */
    } MicroForce_FMA_transfer_function_t;


    /**
      * @brief   FORCE RANGE
      */
    #define FMA_FORCE_RANGE_5_N     5U                  /*!<   Force range:  5N                 */
    #define FMA_FORCE_RANGE_15_N    15U                 /*!<   Force range: 15N                 */
    #define FMA_FORCE_RANGE_25_N    25U                 /*!<   Force range: 25N                 */



#ifndef MicroForce_FMA_STRUCT_H
#define MicroForce_FMA_STRUCT_H
    typedef struct {
        MicroForce_FMA_status_bits_t status_bits;
    } MicroForce_FMA_status_bits_data_t;


    typedef struct {
        uint16_t  raw_bridge_data;
        float     bridge_data;                                              /*!<   Force data in Newton (N)           */
    } MicroForce_FMA_bridge_data_t;


    typedef struct {
        uint8_t   raw_8bit_temperature;
        uint16_t  raw_11bit_temperature;
        float     temperature_data;                                         /*!<   Temperature data in Celsius degree */
    } MicroForce_FMA_temperature_data_t;


    /* User's data   */
    typedef struct {
        MicroForce_FMA_transfer_function_t  transfer_function;              /*!<   Transfer function. Calibration     */
        uint8_t                             force_range;                    /*!<   Force Range                        */

        MicroForce_FMA_status_bits_data_t   status;                         /*!<   Status bits. Device condition      */

        MicroForce_FMA_bridge_data_t        force;                          /*!<   Force data                         */
        MicroForce_FMA_temperature_data_t   temperature;                    /*!<   Temperature data                   */
    } MicroForce_FMA_data_t;
#endif



    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        MicroForce_FMA_SUCCESS     =       0,
        MicroForce_FMA_FAILURE     =       1,
        SPI_SUCCESS                =       1
    } MicroForce_FMA_status_t;




    /** Create an MicroForce_FMA object connected to the specified SPI pins.
      *
      * @param mosi     SPI Master Output Slave Input
      * @param miso     SPI Master Input Slave Output
      * @param sclk     SPI clock
      * @param cs       SPI Chip Select
      * @param freq     SPI frequency in Hz.
      */
    MicroForce_FMA ( PinName mosi, PinName miso, PinName sclk, PinName cs, uint32_t freq );

    /** Delete MicroForce_FMA object.
     */
    ~MicroForce_FMA();

    /** It gets the raw force data.
    */
    MicroForce_FMA_status_t  MicroForce_FMA_GetRawForce               ( MicroForce_FMA_status_bits_data_t* myStatus, uint16_t* myRawBridgeData                            );

    /** It calculates the force data in Newton (N).
        */
    float                    MicroForce_FMA_CalculateForce            ( MicroForce_FMA_transfer_function_t myCalibration, uint8_t myForceRange, uint16_t myRawBridgeData                                    );

    /** It gets the raw 8-bit temperature.
        */
    MicroForce_FMA_status_t  MicroForce_FMA_GetRaw8bitTemperature     ( MicroForce_FMA_status_bits_data_t* myStatus, uint8_t* myRaw8bitTemp                               );

    /** It gets the raw 11-bit temperature.
        */
    MicroForce_FMA_status_t  MicroForce_FMA_GetRaw11bitTemperature    ( MicroForce_FMA_status_bits_data_t* myStatus, uint16_t* myRaw11bitTemp                             );

    /** It calculates the 8-bit temperature in Celsius degree.
        */
    float                    MicroForce_FMA_Calculate8bitTemperature  ( uint8_t myRawTemperature                                                                                                            );

    /** It calculates the 11-bit temperature in Celsius degree.
        */
    float                    MicroForce_FMA_Calculate11bitTemperature ( uint16_t myRawTemperature                                                                                                           );

    /** It gets the all raw data ( bridge data and 11-bit temperature ).
        */
    MicroForce_FMA_status_t  MicroForce_FMA_GetAllRawData             ( MicroForce_FMA_status_bits_data_t* myStatus, uint16_t* myRawBridgeData, uint16_t* myRaw11bitTemp  );

private:
    SPI         _spi;
    DigitalOut  _cs;
};

#endif
