/**
 * @brief       MICS_VZ_89TE.h
 * @details     Integrated VOC Sensor Module with CO2 equivalent.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        27/May/2021
 * @version     27/May/2021    The ORIGIN
 * @pre         N/A
 * @warning     N/A
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef MICS_VZ_89TE_H_
#define MICS_VZ_89TE_H_

#include "mbed.h"


/**
    Example:

@code
#include "mbed.h"
#include "MICS_VZ_89TE.h"

MICS_VZ_89TE myMICS_VZ_89TE ( I2C_SDA, I2C_SCL, MICS_VZ_89TE::MICS_VZ_89TE_ADDRESS, 100000 );               // I2C_SDA | I2C_SCL
Serial pc     ( USBTX, USBRX );                                                 // tx, rx

DigitalOut  myled   ( LED1 );
Ticker      newAction;


//@brief Constants.


//@brief Variables.
volatile uint32_t myState;                                                      // State that indicates when to perform a new sample


//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    MICS_VZ_89TE::MICS_VZ_89TE_status_t aux;
    MICS_VZ_89TE::MICS_VZ_89TE_data_t   myMICS_VZ_89TE_Data;

    pc.baud ( 115200 );

    myled   =   1;
    wait(3);
    myled   =   0;

    // It triggers the Revision command and gets the result
    aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_TriggersRevision ();
    wait_ms (100);
    aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_GetRevision ( &myMICS_VZ_89TE_Data.device );

    // It triggers the R0 ( calibration ) command and gets the result
    aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_TriggersR0 ();
    wait_ms (100);
    aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_GetR0 ( &myMICS_VZ_89TE_Data.r0_raw, &myMICS_VZ_89TE_Data.r0 );

    pc.printf ( "Revision: %d | R0: %d\r\n", myMICS_VZ_89TE_Data.device, myMICS_VZ_89TE_Data.r0 );


    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 5U );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 5s )

    // Let the callbacks take care of everything
    while(1) {
        sleep();

        if ( myState == 1UL ) {
            myled = 1U;

            // Trigger to get a new data value
            aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_TriggersStatus ();
            wait_ms (100);

            // Get the values: tVOC, CO2_equ and Resistor
            aux  =   myMICS_VZ_89TE.MICS_VZ_89TE_GetUpdateValues ( &myMICS_VZ_89TE_Data.status, &myMICS_VZ_89TE_Data.values );

            // Send data through the UART
            pc.printf ( "tVOC: %d [ppb], CO2_equ: %d [ppm], R: %d [kOhms]\r\n", (uint32_t)myMICS_VZ_89TE_Data.values.tvoc, (uint32_t)myMICS_VZ_89TE_Data.values.co2_equ, (uint32_t)myMICS_VZ_89TE_Data.values.resistor );

            // Reset the variables
            myState  =   0UL;
            myled    =   0U;
        }
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
// @return       N/A.
//
// @author      Manuel Caballero
// @date        09/June/2021
// @version     09/June/2021   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState  =   1UL;
}
@endcode
*/


/*!
 Library for the MICS_VZ_89TE Integrated VOC Sensor Module with CO2 equivalent.
*/
class MICS_VZ_89TE
{
public:
    /**
    * @brief   DEFAULT ADDRESS
    */
    typedef enum {
        MICS_VZ_89TE_ADDRESS   =   ( 0x70 << 1U )       /*!<   I2C slave address byte, Default              */
    } MICS_VZ_89TE_address_t;



    /**
      * @brief   COMMAND LIST
      */
    typedef enum {
        MICS_VZ_89TE_SET_PPM_CO2  =   0b00001000,       /*!<  To send the ppmCO2 value given by an external analyser to the VZ89TE in order to recalibrate its outputs  */
        MICS_VZ_89TE_GET_STATUS   =   0b00001100,       /*!<  This command is used to read the VZ89TE status coded on 6x bytes + 1 CRC byte                             */
        MICS_VZ_89TE_GET_REVISION =   0b00001101,       /*!<  This command will return the revision code of the module                                                  */
        MICS_VZ_89TE_GET_R0       =   0b00010000        /*!<  This command is used to read the R0 (calibration) value in [kOhms]                                        */
    } MICS_VZ_89TE_command_list_t;





#ifndef MICS_VZ_89TE_VECTOR_STRUCT_H
#define MICS_VZ_89TE_VECTOR_STRUCT_H
    /* Status command  */
    typedef struct {
        uint8_t d1;                   /*!<  Raw tVOC                                                */
        uint8_t d2;                   /*!<  Raw CO2_equ                                             */
        uint8_t d3;                   /*!<  RS first byte(MSB)                                      */
        uint8_t d4;                   /*!<  RS second byte                                          */
        uint8_t d5;                   /*!<  RS third byte(LSB)                                      */
        uint8_t d6;                   /*!<  Status                                                  */
        uint8_t d7;                   /*!<  CRC                                                     */
    } MICS_VZ_89TE_get_status_t;


    /* Output values  */
    typedef struct {
        float     tvoc;               /*!<  tVOC [13...242] in ppb                                  */
        float     co2_equ;            /*!<  CO2 equivalent [13...242] in ppm                        */
        uint32_t  resistor;           /*!<  Resistor value in Ohms                                  */
    } MICS_VZ_89TE_output_values_t;


    /* Revision  */
    typedef struct {
        uint8_t year;                  /*!<  Year                                                   */
        uint8_t month;                 /*!<  Month                                                  */
        uint8_t day;                   /*!<  Day                                                    */
        uint8_t ascii_code;            /*!<  ASCII code for a charter                               */
        uint8_t crc;                   /*!<  CRC                                                    */
    } MICS_VZ_89TE_revision_t;


    /* R0: Raw data  */
    typedef struct {
        uint8_t r0_lsb;                 /*!<  R0 LSB                                                 */
        uint8_t r0_msb;                 /*!<  R0 MSB                                                 */
        uint8_t crc;                    /*!<  CRC                                                    */
    } MICS_VZ_89TE_r0_t;



    /* USER: User's global variables     */
    typedef struct {
        /* Output  */
        MICS_VZ_89TE_get_status_t     status;   /*!<  Raw values: tVOC, CO2_equ and Resistor                  */
        MICS_VZ_89TE_output_values_t  values;   /*!<  Output values: tVOC, CO2_equ and Resistor               */


        /* Calibration   */
        MICS_VZ_89TE_r0_t             r0_raw;   /*!<  R0 calibration raw value                                */
        uint16_t                      r0;       /*!<  R0 calibration value in kOhms                           */


        /* Device identification   */
        MICS_VZ_89TE_revision_t       device;   /*!<  Device. The revision code of the module                 */
    } MICS_VZ_89TE_data_t;
#endif




    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        MICS_VZ_89TE_SUCCESS           =       0,
        MICS_VZ_89TE_FAILURE           =       1,
        MICS_VZ_89TE_DATA_CORRUPTED    =       2,
        I2C_SUCCESS             =       0
    } MICS_VZ_89TE_status_t;



    /**
      * @brief   FUNCTION PROTOTYPES
      */
    /** Create an MICS_VZ_89TE object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    MICS_VZ_89TE ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete MICS_VZ_89TE object.
     */
    ~MICS_VZ_89TE();

    /** It triggers the STATUS command.
    */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_TriggersStatus   ( void                                                                      );

    /** It gets all the raw values.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_GetStatus        ( MICS_VZ_89TE_get_status_t* status                                         );

    /** It gets all the values: tVOC, CO2_equ and Resistor.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_GetUpdateValues  ( MICS_VZ_89TE_get_status_t* status, MICS_VZ_89TE_output_values_t* values   );

    /** It triggers the Revsion command.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_TriggersRevision ( void                                                                      );

    /** It gets Revision values.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_GetRevision      ( MICS_VZ_89TE_revision_t* revision                                         );

    /** It triggers the R0 command.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_TriggersR0       ( void                                                                      );

    /** It gets R0 value.
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_GetRawR0         ( MICS_VZ_89TE_r0_t* rawR0                                                  );

    /** It gets R0 value (in kOhms).
      */
    MICS_VZ_89TE_status_t MICS_VZ_89TE_GetR0            ( MICS_VZ_89TE_r0_t* rawR0, uint16_t* r0                                    );



private:
    /** It gets the CRC byte.
    */
    uint8_t               MICS_VZ_89TE_GetCRC           ( char* buffer, uint8_t size                                                );

    I2C      _i2c;
    uint32_t _MICS_VZ_89TE_Addr;
};

#endif /* MICS_VZ_89TE_H */
