/**
 * @brief       MCP4725.h
 * @details     12-Bit Digital-to-Analog Converter with EEPROM Memory.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        7/September/2017
 * @version     7/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef MCP4725_H
#define MCP4725_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "MCP4725.h"

MCP4725 myDACSensor        ( I2C_SDA, I2C_SCL, MCP4725::MCP4725_ADDRESS_LOW, 400000 );


Ticker newDACOutput;
DigitalOut myled(LED1);

MCP4725::MCP4725_status_t        aux;
uint8_t                 myState = 0;


void changeDATA ( void )
{
    MCP4725::Vector_new_dac_value_t myNewDAC_Value;

    myled = 1;

    switch ( myState ) {
        default:
        case 0:
            // Vout ~ 0V
            myNewDAC_Value.DAC_New_Value   =   0;
            aux = myDACSensor.MCP4725_SetNewValue ( MCP4725::FAST_MODE, myNewDAC_Value );

            myState                      =   1;
            break;

        case 1:
            // Vout = ~ ( Vref * 0.5 )
            myNewDAC_Value.DAC_New_Value   =   2048;
            aux = myDACSensor.MCP4725_SetNewValue ( MCP4725::WRITE_DAC_AND_EEPROM_REGISTER_MODE, myNewDAC_Value );

            myState                      =   2;
            break;

        case 2:
            // Vout ~ Vref
            myNewDAC_Value.DAC_New_Value   =   4095;
            aux = myDACSensor.MCP4725_SetNewValue ( MCP4725::WRITE_DAC_REGISTER_MODE, myNewDAC_Value );

            myState                      =   0;
            break;
    }

    myled = 0;
}


int main()
{
    MCP4725::Vector_data_t           myDefaultData;


    // Reset and wake the device up
    aux = myDACSensor.MCP4725_Reset  ();
    aux = myDACSensor.MCP4725_WakeUp ();

    // Read the default data in both EEPROM and DAC
    aux = myDACSensor.MCP4725_GetDAC_Data    ( &myDefaultData );
    aux = myDACSensor.MCP4725_GetEEPROM_Data ( &myDefaultData );


    newDACOutput.attach( &changeDATA, 0.5 );                      // the address of the function to be attached ( changeDATA ) and the interval ( 0.5s )

    // Let the callbacks take care of everything
    while(1)  sleep();
}
*/


/*!
 Library for the MCP4725 12-Bit Digital-to-Analog Converter with EEPROM Memory.
*/
class MCP4725
{
public:
    /**
    * @brief   DEFAULT ADDRESSES ( NOTE1: The A2 and A1 are programmed to '00' (default), if not requested by customer.
    *                              NOTE2: On my board, the A2 and A1 are programmed to '01'. )
    */
    typedef enum {
        MCP4725_ADDRESS_LOW     =   ( 0x62 << 1 ),                   /*!<   A0 pin ties to GND                                            */
        MCP4725_ADDRESS_HIGH    =   ( 0x63 << 1 )                    /*!<   A0 pin ties to VDD                                            */
    } MCP4725_address_t;


    /**
      * @brief   COMMANDS
      */
#define MCP4725_GENERAL_CALL             0x00              /*!<   The MCP4725 device acknowledges the general call address                             */

    /* General Call Commands */
    /**
      * @brief   GENERAL CALL COMMANDS
      */
#define MCP4725_GENERAL_CALL_RESET       0x06              /*!<  Perform an internal reset similar to a power-on-reset (POR).                          */
#define MCP4725_GENERAL_CALL_WAKE_UP     0x09              /*!<  The power-down bits of the DAC register are set to a normal operation.                */



    /* Commands Registers */
    /**
      * @brief   WRITE COMMAND TYPE
      */
    typedef enum {
        FAST_MODE                              =   0,           /*!<  This command is used to change the DAC register. EEPROM is not affected.              */
        WRITE_DAC_REGISTER_MODE                =   1,           /*!<  Load configuration bits and data code to the DAC Register.                            */
        WRITE_DAC_AND_EEPROM_REGISTER_MODE     =   2            /*!<  Load configuration bits and data code to the DAC Register and also write the EEPROM.  */
    } MCP4725_write_command_type_t;



    /**
      * @brief   POWER-DOWN MODE
      */
    typedef enum {
        NORMAL_MODE                                 =   0,      /*!<  Normal Mode.                                                                          */
        POWER_DOWN_1KOHM_RESISTIVE_LOAD_MODE        =   1,      /*!<  Power-Down Mode. 1 k‎Ω resistor to ground.                                             */
        POWER_DOWN_100KOHM_RESISTIVE_LOAD_MODE      =   2,      /*!<  Power-Down Mode. 100 k‎Ω resistor to ground.                                           */
        POWER_DOWN_500KOHM_RESISTIVE_LOAD_MODE      =   3       /*!<  Power-Down Mode. 500 k‎Ω resistor to ground.                                           */
    } MCP4725_operation_mode_t;



    /**
      * @brief   READY/#BUSY BIT
      */
    typedef enum {
        EEPROM_BUSY                                 =   0,      /*!<  EEPROM write is not completed.                                                        */
        EEPROM_READY                                =   1       /*!<  EEPROM write is complete.                                                             */
    } MCP4725_eeprom_status_t;




#ifndef VECTOR_STRUCT_H
#define VECTOR_STRUCT_H
    typedef struct {
        uint16_t EEPROM_Data;
        uint16_t DAC_Data;
    } Vector_data_t;

    typedef struct {
        uint32_t DAC_New_Value;
    } Vector_new_dac_value_t;
#endif





    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        MCP4725_SUCCESS     =       0,
        MCP4725_FAILURE     =       1,
        I2C_SUCCESS         =       0                           /*!<   I2C communication was fine                                                          */
    } MCP4725_status_t;




    /** Create an MCP4725 object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency in Hz.
      */
    MCP4725 ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete MCP4725 object.
     */
    ~MCP4725();

    /** It performs an internal reset similar to a power-on-reset ( POR ).
     */
    MCP4725_status_t  MCP4725_Reset                         ( void );

    /** The power-down bits of the DAC register are set to a normal operation.
     */
    MCP4725_status_t  MCP4725_WakeUp                        ( void );

    /** It configures the power mode of the device.
     */
    MCP4725_status_t  MCP4725_PowerMode                     ( MCP4725_write_command_type_t myWriteCMD, MCP4725_operation_mode_t myPowerMode );

    /** It sets a new output value.
     */
    MCP4725_status_t  MCP4725_SetNewValue                   ( MCP4725_write_command_type_t myWriteCMD, Vector_new_dac_value_t myDACNewValue );

    /** It gets the eeprom status.
     */
    MCP4725_status_t  MCP4725_EEPROM_Status                 ( MCP4725_eeprom_status_t* myEEPROM_Status );

    /** It gets the last eeprom stored value.
     */
    MCP4725_status_t  MCP4725_GetEEPROM_Data                ( Vector_data_t* myEEPROMData );

    /** It gets the last dac stored value.
     */
    MCP4725_status_t  MCP4725_GetDAC_Data                   ( Vector_data_t* myDACData );




private:
    I2C      i2c;
    uint32_t MCP4725_Addr;
};

#endif
