/**
 * @brief       MAX7219.h
 * @details     Serially Interfaced, 8-Digit LED Display Drivers.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */

#include "MAX7219.h"


MAX7219::MAX7219 ( PinName mosi, PinName miso, PinName sclk, PinName cs, uint32_t freq )
    : _spi ( mosi, miso, sclk )
    , _cs  ( cs )
{
    _spi.frequency( freq );
}


MAX7219::~MAX7219()
{
}



/**
 * @brief       MAX7219_Mode   ( MAX7219_shutdown_reg_t )
 *
 * @details     It puts the device in shutdown mode.
 *
 * @param[in]    myMAX7219mode:     Shutdown or Normal operation mode.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_Mode.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_Mode  ( MAX7219_shutdown_reg_t myMAX7219mode )
{
    char  cmd[]         =    { SHUTDOWN, 0 };
    int   mySPI_status  =    0;


    cmd[ 1 ]     =   myMAX7219mode;

    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;


    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}



/**
 * @brief       MAX7219_DisplayTest   ( MAX7219_display_test_reg_t )
 *
 * @details     It turns all the LEDs on ( Test mode enabled ) or normal operation.
 *
 * @param[in]    myMAX7219DisplayTestMode:  Mode: Test or Normal operation.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_DisplayTest.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_DisplayTest ( MAX7219_display_test_reg_t myMAX7219DisplayTestMode )
{
    char  cmd[]         =    { DISPLAY_TEST, 0 };
    int   mySPI_status  =    0;


    cmd[ 1 ]     =   myMAX7219DisplayTestMode;


    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;



    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}



/**
 * @brief       MAX7219_DecodeMode   ( MAX7219_decode_mode_reg_t )
 *
 * @details     It enables and configures the decode-mode or turns it off.
 *
 * @param[in]    myMAX7219DecodeMode:   Decode-mode option.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_DecodeMode.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_DecodeMode ( MAX7219_decode_mode_reg_t myMAX7219DecodeMode )
{
    char  cmd[]         =    { DECODE_MODE, 0 };
    int   mySPI_status  =    0;


    cmd[ 1 ]     =   myMAX7219DecodeMode;


    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;



    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}



/**
 * @brief       MAX7219_SetIntensity   ( MAX7219_intensity_reg_t )
 *
 * @details     It configures the intensity of the device.
 *
 * @param[in]    myMAX7219Intensity:    Intensity option.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_SetIntensity.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_SetIntensity ( MAX7219_intensity_reg_t myMAX7219Intensity )
{
    char  cmd[]         =    { INTENSITY, 0 };
    int   mySPI_status  =    0;


    cmd[ 1 ]     =   myMAX7219Intensity;


    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;



    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}



/**
 * @brief       MAX7219_SetScanLimit   ( MAX7219_scan_limit_reg_t )
 *
 * @details     It configures duty cycle of the device.
 *
 * @param[in]    MAX7219_scan_limit_reg_t:  Duty cycle option.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_SetScanLimit.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_SetScanLimit ( MAX7219_scan_limit_reg_t myMAX7219ScanLimit )
{
    char  cmd[]         =    { SCAN_LIMIT, 0 };
    int   mySPI_status  =    0;


    cmd[ 1 ]     =   myMAX7219ScanLimit;


    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;



    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}



/**
 * @brief       MAX7219_SetDigit   ( MAX7219_digit_t , MAX7219_code_b_font_reg_t , MAX7219_code_b_dp_t )
 *
 * @details     It sets the digit to be written and its value.
 *
 * @param[in]    myMAX7219Digit:        Digit to be written.
 * @param[in]    myCharacter:           The current value.
 * @param[in]    myDP_status:           DP enabled/disabled.
 *
 * @param[out]   NaN.
 *
 *
 * @return       Status of MAX7219_SetDigit.
 *
 *
 * @author      Manuel Caballero
 * @date        9/October/2017
 * @version     9/October/2017   The ORIGIN
 * @pre         NaN
 * @warning     NaN.
 */
MAX7219::MAX7219_status_t  MAX7219::MAX7219_SetDigit ( MAX7219_digit_t myMAX7219Digit, MAX7219_code_b_font_reg_t myCharacter, MAX7219_code_b_dp_t myDP_status )
{
    char  cmd[]         =    { 0, 0 };
    int   mySPI_status  =    0;


    switch ( myMAX7219Digit ) {
        default:
        case SET_DIGIT_0:
            cmd[ 0 ]     =   DIGIT_0;
            break;

        case SET_DIGIT_1:
            cmd[ 0 ]     =   DIGIT_1;
            break;

        case SET_DIGIT_2:
            cmd[ 0 ]     =   DIGIT_2;
            break;

        case SET_DIGIT_3:
            cmd[ 0 ]     =   DIGIT_3;
            break;

        case SET_DIGIT_4:
            cmd[ 0 ]     =   DIGIT_4;
            break;

        case SET_DIGIT_5:
            cmd[ 0 ]     =   DIGIT_5;
            break;

        case SET_DIGIT_6:
            cmd[ 0 ]     =   DIGIT_6;
            break;

        case SET_DIGIT_7:
            cmd[ 0 ]     =   DIGIT_7;
            break;
    }


    if ( myDP_status == DP_ENABLED )
        cmd[ 1 ]     =   ( myCharacter | 0x80 );
    else
        cmd[ 1 ]     =   myCharacter;


    _cs          =   0;
    mySPI_status =   _spi.write ( &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), &cmd[0], 0 );
    _cs          =   1;



    if ( ( mySPI_status / ( sizeof( cmd )/sizeof( cmd[0] ) ) ) == SPI_SUCCESS )
        return   MAX7219_SUCCESS;
    else
        return   MAX7219_FAILURE;
}
