/**
 * @brief       LPS25HB.cpp
 * @details     MEMS pressure sensor: 260-1260 hPa absolute digital output barometer.
 *              Function file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ). 
 */

#include "LPS25HB.h"


LPS25HB::LPS25HB ( PinName sda, PinName scl, uint32_t addr, uint32_t freq )
    : _i2c          ( sda, scl )
    , _LPS25HB_Addr ( addr )
{
    _i2c.frequency( freq );
}


LPS25HB::~LPS25HB()
{
}



/**
 * @brief       LPS25HB_GetReferencePressure ( LPS25HB_data_t* )
 *
 * @details     It gets raw reference pressure.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myREFL:          Raw reference pressure.
 *
 *
 * @return       Status of LPS25HB_GetReferencePressure.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function uses auto-increment to read more than one register in a raw.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetReferencePressure ( LPS25HB_data_t* myREFL )
{
  char     cmd[3]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_REF_P_XL | 0x80 );                                      // Auto-increment enabled
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );

  /* Mask it and update it with the new value  */
  myREFL->rawReferencePressure   =   cmd[2];
  myREFL->rawReferencePressure <<=   8UL;
  myREFL->rawReferencePressure  |=   cmd[1];
  myREFL->rawReferencePressure <<=   8UL;
  myREFL->rawReferencePressure  |=   cmd[0];



  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetReferencePressure ( LPS25HB_data_t )
 *
 * @details     It sets raw reference pressure.
 *
 * @param[in]    myREFL:          Raw reference pressure.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetReferencePressure.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function uses auto-increment to write more than one register in a raw.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetReferencePressure ( LPS25HB_data_t myREFL )
{
  char     cmd[4]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_REF_P_XL | 0x80 );                                      // Auto-increment enabled
  cmd[1]  =   (char)( myREFL.rawReferencePressure );                            // REF_P_XL
  cmd[2]  =   (char)( myREFL.rawReferencePressure >> 8UL );                     // REF_P_L
  cmd[3]  =   (char)( myREFL.rawReferencePressure >> 16UL );                    // REF_P_H
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );




  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetDeviceID ( LPS25HB_data_t* )
 *
 * @details     It gets the device ID.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myID:            Device ID.
 *
 *
 * @return       Status of LPS25HB_GetDeviceID.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetDeviceID ( LPS25HB_data_t* myID )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Write the register   */
  cmd   =  LPS25HB_WHO_AM_I;                                              
  aux   =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux   =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );

  /* Parse data  */
  myID->deviceID   =   cmd;



  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetTemperatureResolution ( LPS25HB_data_t )
 *
 * @details     It sets temperature resolution.
 *
 * @param[in]    myAVGT:          Temperature resolution.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetTemperatureResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetTemperatureResolution ( LPS25HB_data_t myAVGT )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_RES_CONF;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( RES_CONF_AVGT_MASK );
  cmd[1] |=  myAVGT.avgt;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetTemperatureResolution ( LPS25HB_data_t* )
 *
 * @details     It gets temperature resolution.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAVGT:          Temperature resolution.
 *
 *
 * @return       Status of LPS25HB_GetTemperatureResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetTemperatureResolution ( LPS25HB_data_t* myAVGT )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_RES_CONF;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myAVGT->avgt   =   (LPS25HB_res_conf_avgt_t)( cmd & RES_CONF_AVGT_MASK );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetPressureResolution ( LPS25HB_data_t )
 *
 * @details     It sets pressure resolution.
 *
 * @param[in]    myAVGP:          Pressure resolution.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetPressureResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetPressureResolution ( LPS25HB_data_t myAVGP )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_RES_CONF;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( RES_CONF_AVGP_MASK );
  cmd[1] |=  myAVGP.avgp;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetPressureResolution ( LPS25HB_data_t* )
 *
 * @details     It gets pressure resolution.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAVGP:          Pressure resolution.
 *
 *
 * @return       Status of LPS25HB_GetPressureResolution.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetPressureResolution ( LPS25HB_data_t* myAVGP )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_RES_CONF;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myAVGP->avgp   =   (LPS25HB_res_conf_avgp_t)( cmd & RES_CONF_AVGP_MASK );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetPowerMode ( LPS25HB_ctrl_reg1_pd_t )
 *
 * @details     It sets power mode.
 *
 * @param[in]    myPD:            Power mode: Power-down/Active mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetPowerMode.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetPowerMode ( LPS25HB_ctrl_reg1_pd_t myPD )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG1;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG1_PD_MASK );
  cmd[1] |=   myPD;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetOutputDataRate ( LPS25HB_data_t )
 *
 * @details     It sets the output data rate.
 *
 * @param[in]    myODR:           Output data rate.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetOutputDataRate.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetOutputDataRate ( LPS25HB_data_t myODR )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG1;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG1_ODR_MASK );
  cmd[1] |=   myODR.odr;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetOutputDataRate ( LPS25HB_data_t* )
 *
 * @details     It gets the output data rate.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myODR:           Output data rate.
 *
 *
 * @return       Status of LPS25HB_GetOutputDataRate.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetOutputDataRate ( LPS25HB_data_t* myODR )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG1;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myODR->odr   =   (LPS25HB_ctrl_reg1_odr_t)( cmd & CTRL_REG1_ODR_MASK );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetInterruptGeneration ( LPS25HB_ctrl_reg1_diff_en_t )
 *
 * @details     It sets the interrupt generation enable.
 *
 * @param[in]    myDIFF_EN:       Interrupt generation enabled/disabled.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetInterruptGeneration.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetInterruptGeneration ( LPS25HB_ctrl_reg1_diff_en_t myDIFF_EN )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG1;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG1_DIFF_EN_MASK );
  cmd[1] |=   myDIFF_EN;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetBlockDataUpdate ( LPS25HB_ctrl_reg1_bdu_t )
 *
 * @details     It sets the block data update.
 *
 * @param[in]    myBDU:           Block data update.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetBlockDataUpdate.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetBlockDataUpdate ( LPS25HB_ctrl_reg1_bdu_t myBDU )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG1;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG1_BDU_MASK );
  cmd[1] |=   myBDU;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetResetAutozero ( void )
 *
 * @details     It sets the reset autozero function.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetResetAutozero.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetResetAutozero ( void )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG1;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG1_RESET_AZ_MASK );
  cmd[1] |=   CTRL_REG1_RESET_AZ_RESET_AUTOZERO_FUNCTION;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetResetAutozero ( LPS25HB_data_t* )
 *
 * @details     It gets the reset autozero function.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRESET_AZ:      Reset autozero function value.
 *
 *
 * @return       Status of LPS25HB_GetResetAutozero.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetResetAutozero ( LPS25HB_data_t* myRESET_AZ )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG1;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myRESET_AZ->reset_az   =  (LPS25HB_ctrl_reg1_reset_az_t)( CTRL_REG1_RESET_AZ_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetRebootMemoryContent ( void )
 *
 * @details     It sets the reboot memory content.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetRebootMemoryContent.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetRebootMemoryContent ( void )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_BOOT_MASK );
  cmd[1] |=   CTRL_REG2_BOOT_REBOOT_MODE;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetRebootMemoryContent ( LPS25HB_data_t* )
 *
 * @details     It gets the reboot memory content.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myBOOT:          Reboot memory content value.
 *
 *
 * @return       Status of LPS25HB_GetRebootMemoryContent.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetRebootMemoryContent ( LPS25HB_data_t* myBOOT )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG2;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myBOOT->boot   =  (LPS25HB_ctrl_reg2_boot_t)( CTRL_REG2_BOOT_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetFIFOEnable ( LPS25HB_data_t )
 *
 * @details     It sets the FIFO enable/disable.
 *
 * @param[in]    myFIFO_EN:       FIFO enable/disable.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetFIFOEnable.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetFIFOEnable ( LPS25HB_data_t myFIFO_EN )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_FIFO_EN_MASK );
  cmd[1] |=   myFIFO_EN.fifo_en;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetFIFOEnable ( LPS25HB_data_t* )
 *
 * @details     It gets the FIFO enable/disable.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myFIFO_EN:       FIFO enable/disable.
 *
 *
 * @return       Status of LPS25HB_GetFIFOEnable.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetFIFOEnable ( LPS25HB_data_t* myFIFO_EN )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG2;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myFIFO_EN->fifo_en   =  (LPS25HB_ctrl_reg2_fifo_en_t)( CTRL_REG2_FIFO_EN_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetFIFOMeanDec ( LPS25HB_ctrl_reg2_fifo_mean_dec_t )
 *
 * @details     It enables/disables the decimate the output pressure to 1Hz with FIFO Mean mode.
 *
 * @param[in]    myFIFO_MEAN_DEC: It enables/disables the decimate the output pressure to 1Hz with FIFO Mean mode.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetFIFOMeanDec.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetFIFOMeanDec ( LPS25HB_ctrl_reg2_fifo_mean_dec_t myFIFO_MEAN_DEC )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_FIFO_MEAN_DEC_MASK );
  cmd[1] |=   myFIFO_MEAN_DEC;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetSoftwareReset ( void )
 *
 * @details     It sets the software reset.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetSoftwareReset.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetSoftwareReset ( void )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_SWRESET_MASK );
  cmd[1] |=   CTRL_REG2_SWRESET_SW_RESET;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetSoftwareReset ( LPS25HB_data_t* )
 *
 * @details     It gets the software reset flag value.
 *
 * @param[in]    N/A.
 *
 * @param[out]   mySWRESET:       Software reset flag value.
 *
 *
 * @return       Status of LPS25HB_GetSoftwareReset.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetSoftwareReset ( LPS25HB_data_t* mySWRESET )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG2;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  mySWRESET->swreset   =  (LPS25HB_ctrl_reg2_swreset_t)( CTRL_REG2_SWRESET_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetAutozero ( LPS25HB_data_t )
 *
 * @details     It sets the autozero enable.
 *
 * @param[in]    myAUTOZERO:      Enable/disable autozero.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetAutozero.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetAutozero ( LPS25HB_data_t myAUTOZERO )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_AUTOZERO_MASK );
  cmd[1] |=   myAUTOZERO.autozero;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetAutozero ( LPS25HB_data_t* )
 *
 * @details     It gets the software reset flag value.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myAUTOZERO:      Autozero enable value.
 *
 *
 * @return       Status of LPS25HB_GetAutozero.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetAutozero ( LPS25HB_data_t* myAUTOZERO )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG2;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myAUTOZERO->autozero   =  (LPS25HB_ctrl_reg2_autozero_t)( CTRL_REG2_AUTOZERO_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetOneShot ( void )
 *
 * @details     It sets the one-shot mode.
 *
 * @param[in]    N/A.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetOneShot.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetOneShot ( void )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG2;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG2_ONE_SHOT_MASK );
  cmd[1] |=   CTRL_REG2_ONE_SHOT_NEW_DATASET;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetOneShot ( LPS25HB_data_t* )
 *
 * @details     It gets the one-shot mode flag.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myONE_SHOT:      One-shot flag value.
 *
 *
 * @return       Status of LPS25HB_GetOneShot.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetOneShot ( LPS25HB_data_t* myONE_SHOT )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG2;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myONE_SHOT->one_shot   =  (LPS25HB_ctrl_reg2_one_shot_t)( CTRL_REG2_ONE_SHOT_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetInterruptActiveMode ( LPS25HB_ctrl_reg3_int_h_l_t )
 *
 * @details     It sets the interrupt active mode.
 *
 * @param[in]    myINT_H_L:       Interrupt active high/low.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetInterruptActiveMode.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetInterruptActiveMode ( LPS25HB_ctrl_reg3_int_h_l_t myINT_H_L )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG3;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG3_INT_H_L_MASK );
  cmd[1] |=   myINT_H_L;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetDrainSelectionMode ( LPS25HB_ctrl_reg3_pp_od_t )
 *
 * @details     It sets the Push-pull/open drain selection on interrupt pads.
 *
 * @param[in]    myPP_OD:         Push-pull/open drain
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetDrainSelectionMode.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetDrainSelectionMode ( LPS25HB_ctrl_reg3_pp_od_t myPP_OD )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG3;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG3_PP_OD_MASK );
  cmd[1] |=   myPP_OD;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetDataSignalOnPin ( LPS25HB_ctrl_reg3_int_s2_t )
 *
 * @details     It sets the Data signal on INT_DRDY pin control bits.
 *
 * @param[in]    myINT_S:         Data signal on INT_DRDY pin control bits
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetDataSignalOnPin.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetDataSignalOnPin ( LPS25HB_ctrl_reg3_int_s2_t myINT_S )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG3;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG3_INT_S2_MASK );
  cmd[1] |=   myINT_S;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetINT_DRDY_Behaviour ( LPS25HB_data_t )
 *
 * @details     It sets the INT_DRDY behaviour.
 *
 * @param[in]    myIntConfig:     Interrupt configuration parameters.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetINT_DRDY_Behaviour.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetINT_DRDY_Behaviour ( LPS25HB_data_t myIntConfig )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_CTRL_REG4;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( CTRL_REG4_F_EMPTY_MASK | CTRL_REG4_F_FTH_MASK | CTRL_REG4_F_OVR_MASK | CTRL_REG4_DRDY_MASK );
  cmd[1] |=   ( myIntConfig.f_empty | myIntConfig.f_fth | myIntConfig.f_ovr | myIntConfig.drdy );
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetINT_DRDY_Behaviour ( LPS25HB_data_t* )
 *
 * @details     It gets INT_DRDY behaviour.
 *
 * @param[in]    N/A
 *
 * @param[out]   myIntConfig:     Interrupt configuration parameters.
 *
 *
 * @return       Status of LPS25HB_GetINT_DRDY_Behaviour.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetINT_DRDY_Behaviour ( LPS25HB_data_t* myIntConfig )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_CTRL_REG4;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myIntConfig->f_empty   =  (LPS25HB_ctrl_reg4_f_empty_t)( CTRL_REG4_F_EMPTY_MASK & cmd );
  myIntConfig->f_fth     =  (LPS25HB_ctrl_reg4_f_fth_t)( CTRL_REG4_F_FTH_MASK & cmd );
  myIntConfig->f_ovr     =  (LPS25HB_ctrl_reg4_f_ovr_t)( CTRL_REG4_F_OVR_MASK & cmd );
  myIntConfig->drdy      =  (LPS25HB_ctrl_reg4_drdy_t)( CTRL_REG4_DRDY_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetInterruptConfiguration ( LPS25HB_data_t )
 *
 * @details     It sets the interrupt configuration parameters.
 *
 * @param[in]    myIntConfig:     Interrupt configuration parameters.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetInterruptConfiguration.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetInterruptConfiguration ( LPS25HB_data_t myIntConfig )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_INTERRUPT_CFG;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( INTERRUPT_CFG_LIR_MASK | INTERRUPT_CFG_PL_E_MASK | INTERRUPT_CFG_PH_E_MASK );
  cmd[1] |=   ( myIntConfig.lir | myIntConfig.pl_e | myIntConfig.ph_e );
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetInterruptConfiguration ( LPS25HB_data_t* )
 *
 * @details     It gets interrupt configuration parameters.
 *
 * @param[in]    N/A
 *
 * @param[out]   myIntConfig:     Interrupt configuration parameters.
 *
 *
 * @return       Status of LPS25HB_GetInterruptConfiguration.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetInterruptConfiguration ( LPS25HB_data_t* myIntConfig )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_INTERRUPT_CFG;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myIntConfig->lir   =  (LPS25HB_interrupt_cfg_lir_t)( INTERRUPT_CFG_LIR_MASK & cmd );
  myIntConfig->pl_e  =  (LPS25HB_interrupt_cfg_pl_e_t)( INTERRUPT_CFG_PL_E_MASK & cmd );
  myIntConfig->ph_e  =  (LPS25HB_interrupt_cfg_ph_e_t)( INTERRUPT_CFG_PH_E_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetInterruptSource ( LPS25HB_data_t* )
 *
 * @details     It reads the interrupt source register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myIntSource:     Interrupt source.
 *
 *
 * @return       Status of LPS25HB_GetInterruptSource.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetInterruptSource ( LPS25HB_data_t* myIntSource )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_INT_SOURCE;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myIntSource->int_source  =  (char)( 0b00000111 & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetStatusRegister ( LPS25HB_data_t* )
 *
 * @details     It reads the status register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myStatusRegister:  Status register.
 *
 *
 * @return       Status of LPS25HB_GetStatusRegister.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetStatusRegister ( LPS25HB_data_t* myStatusRegister )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_STATUS_REG;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myStatusRegister->status_reg  =  (char)( 0b11110011 & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetRawPressure ( LPS25HB_data_t* )
 *
 * @details     It gets the raw pressure.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawPressure:   Raw pressure.
 *
 *
 * @return       Status of LPS25HB_GetRawPressure.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function uses auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetRawPressure ( LPS25HB_data_t* myRawPressure )
{
  char     cmd[3]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_PRESS_OUT_XL | 0x80 );                                         // Force auto-increment
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  myRawPressure->rawPressure   =   cmd[2];
  myRawPressure->rawPressure <<=   8UL;
  myRawPressure->rawPressure  |=   cmd[1];
  myRawPressure->rawPressure <<=   8UL;
  myRawPressure->rawPressure  |=   cmd[0];
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetRawTemperature ( LPS25HB_data_t* )
 *
 * @details     It gets the raw temperature.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myRawTemperature:  Raw temperature.
 *
 *
 * @return       Status of LPS25HB_GetRawTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function uses auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetRawTemperature ( LPS25HB_data_t* myRawTemperature )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_TEMP_OUT_L | 0x80 );                                         // Force auto-increment
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  myRawTemperature->rawTemperature   =   cmd[1];
  myRawTemperature->rawTemperature <<=   8UL;
  myRawTemperature->rawTemperature  |=   cmd[0];
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetFIFO_Mode ( LPS25HB_data_t )
 *
 * @details     It sets the FIFO mode selection.
 *
 * @param[in]    myFIFOmode:      FIFO mode selection.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetFIFO_Mode.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetFIFO_Mode ( LPS25HB_data_t myFIFOmode )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_FIFO_CTRL;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( FIFO_CTRL_F_MODE_MASK );
  cmd[1] |=   myFIFOmode.f_mode;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetFIFO_Mode ( LPS25HB_data_t* )
 *
 * @details     It gets the FIFO mode selection.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myFIFOmode:      FIFO mode selection.
 *
 *
 * @return       Status of LPS25HB_GetFIFO_Mode.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetFIFO_Mode ( LPS25HB_data_t* myFIFOmode )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_FIFO_CTRL;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myFIFOmode->f_mode   =  (LPS25HB_fifo_ctrl_f_mode_t)( FIFO_CTRL_F_MODE_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetFIFO_Threshold ( LPS25HB_data_t )
 *
 * @details     It sets the FIFO threshold (watermark) level selection.
 *
 * @param[in]    myFIFOthreshold: FIFO threshold (watermark) level selection.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetFIFO_Threshold.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetFIFO_Threshold ( LPS25HB_data_t myFIFOthreshold )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Update the register   */
  cmd[0]  =   LPS25HB_FIFO_CTRL;                                                     
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[1], 1U );
  
  cmd[1] &=  ~( FIFO_CTRL_WTM_POINT_MASK );
  cmd[1] |=   myFIFOthreshold.wtm_point;
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetFIFO_Threshold ( LPS25HB_data_t* )
 *
 * @details     It gets the FIFO threshold (watermark) level selection.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myFIFOthreshold: FIFO threshold (watermark) level selection.
 *
 *
 * @return       Status of LPS25HB_GetFIFO_Threshold.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetFIFO_Threshold ( LPS25HB_data_t* myFIFOthreshold )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_FIFO_CTRL;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myFIFOthreshold->wtm_point   =  (LPS25HB_fifo_ctrl_wtm_point_t)( FIFO_CTRL_WTM_POINT_MASK & cmd );
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetFIFO_Status ( LPS25HB_data_t* )
 *
 * @details     It reads the FIFO status register.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myFIFOstatus:    FIFO threshold (watermark) level selection.
 *
 *
 * @return       Status of LPS25HB_GetFIFO_Status.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetFIFO_Status ( LPS25HB_data_t* myFIFOstatus )
{
  char     cmd  = 0U;
  uint32_t aux;

  /* Read the register   */
  cmd  =   LPS25HB_FIFO_STATUS;                                                     
  aux  =   _i2c.write ( _LPS25HB_Addr, &cmd, 1U, true );
  aux  =   _i2c.read  ( _LPS25HB_Addr, &cmd, 1U );
  
  /* Parse the data  */
  myFIFOstatus->FIFOstatus   =  cmd;
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetFIFO_ThresholdValue ( LPS25HB_data_t* )
 *
 * @details     It gets the FIFO threshold value.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myFIFOthresholdValue:  Threshold value for pressure interrupt generation.
 *
 *
 * @return       Status of LPS25HB_GetFIFO_ThresholdValue.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function implemets auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetFIFO_ThresholdValue ( LPS25HB_data_t* myFIFOthresholdValue )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_THS_P_L | 0x80 );                                   // Force auto-increment
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  myFIFOthresholdValue->ths_p   =   cmd[1];
  myFIFOthresholdValue->ths_p <<=   8U;
  myFIFOthresholdValue->ths_p  |=   cmd[0];
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetFIFO_ThresholdValue ( LPS25HB_data_t* )
 *
 * @details     It sets the FIFO threshold value.
 *
 * @param[in]    myFIFOthresholdValue:  Threshold value for pressure interrupt generation.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetFIFO_ThresholdValue.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function implements auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetFIFO_ThresholdValue ( LPS25HB_data_t myFIFOthresholdValue )
{
  char     cmd[3]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_THS_P_L | 0x80 );                                   // Force auto-increment  
  cmd[1]  =   (char)( myFIFOthresholdValue.ths_p & 0xFF );
  cmd[2]  =   (char)( ( myFIFOthresholdValue.ths_p >> 8U ) & 0xFF );
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );

  
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetPressureOffset ( LPS25HB_data_t* )
 *
 * @details     It gets the Pressure offset value.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myPressureOffset:  Pressure offset.
 *
 *
 * @return       Status of LPS25HB_GetPressureOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function implemets auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetPressureOffset ( LPS25HB_data_t* myPressureOffset )
{
  char     cmd[2]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_RPDS_L | 0x80 );                                        // Force auto-increment
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], 1U, true );
  aux     =   _i2c.read  ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ) );
  
  /* Parse the data  */
  myPressureOffset->rpds   =   cmd[1];
  myPressureOffset->rpds <<=   8U;
  myPressureOffset->rpds  |=   cmd[0];
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_SetPressureOffset ( LPS25HB_data_t* )
 *
 * @details     It sets the Pressure offset value.
 *
 * @param[in]    myPressureOffset:  Pressure offset.
 *
 * @param[out]   N/A.
 *
 *
 * @return       Status of LPS25HB_SetPressureOffset.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         This function implements auto-increment.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_SetPressureOffset ( LPS25HB_data_t myPressureOffset )
{
  char     cmd[3]  = { 0U };
  uint32_t aux;

  /* Read the register   */
  cmd[0]  =   ( LPS25HB_RPDS_L | 0x80 );                                        // Force auto-increment  
  cmd[1]  =   (char)( myPressureOffset.rpds & 0xFF );
  cmd[2]  =   (char)( ( myPressureOffset.rpds >> 8U ) & 0xFF );
  aux     =   _i2c.write ( _LPS25HB_Addr, &cmd[0], sizeof( cmd )/sizeof( cmd[0] ), false );

  
 


  if ( aux == I2C_SUCCESS )
  {
    return   LPS25HB_SUCCESS;
  }
  else
  {
    return   LPS25HB_FAILURE;
  }
}



/**
 * @brief       LPS25HB_GetPressure ( LPS25HB_data_t* )
 *
 * @details     It gets the current pressure in mbar.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myPressure:        Current pressure in mbar.
 *
 *
 * @return       Status of LPS25HB_GetPressure.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetPressure ( LPS25HB_data_t* myPressure )
{
  LPS25HB::LPS25HB_status_t aux;

  /* Read raw pressure   */
  aux  =   LPS25HB_GetRawPressure ( &(*myPressure) );

  /* Check if the pressure value is negative   */
  if ( ( myPressure->rawPressure & 0x800000 ) == 0x800000 )
  {
    /* Negative pressure   */
    myPressure->rawPressure |=   0xFF000000;
  }
  else
  {
    /* Positive pressure   */
  }
  
  
  /* Calculate pressure  */
  myPressure->pressure   =   (float)( myPressure->rawPressure / 4096.0 );
  


  return aux;
}



/**
 * @brief       LPS25HB_GetTemperature ( LPS25HB_data_t* )
 *
 * @details     It gets the current temperature in Celsius degrees.
 *
 * @param[in]    N/A.
 *
 * @param[out]   myTemperature:     Current temperature in Celsius degrees.
 *
 *
 * @return       Status of LPS25HB_GetTemperature.
 *
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019     The ORIGIN
 * @pre         N/A.
 * @warning     N/A.
 */
LPS25HB::LPS25HB_status_t LPS25HB::LPS25HB_GetTemperature ( LPS25HB_data_t* myTemperature )
{
  LPS25HB::LPS25HB_status_t aux;

  /* Read raw temperature   */
  aux  =   LPS25HB_GetRawTemperature ( &(*myTemperature) );

  /* Calculate temperature  */
  myTemperature->temperature   =   (float)( 42.5 + ( myTemperature->rawTemperature / 480.0 ) );
  


  return aux;
}