/**
 * @brief       LPS25HB.h
 * @details     MEMS pressure sensor: 260-1260 hPa absolute digital output barometer.
 *              Header file.
 *
 *
 * @return      N/A
 *
 * @author      Manuel Caballero
 * @date        10/June/2019
 * @version     10/June/2019    The ORIGIN
 * @pre         N/A.
 * @warning     N/A
 * @pre         This code belongs to AqueronteBlog ( http://unbarquero.blogspot.com ).
 */
#ifndef LPS25HB_H
#define LPS25HB_H

#include "mbed.h"


/**
    Example:
@code
#include "mbed.h"
#include "LPS25HB.h"

LPS25HB myLPS25HB ( I2C_SDA, I2C_SCL, LPS25HB::LPS25HB_ADDRESS_1, 400000 );     // I2C_SDA | I2C_SCL
Serial pc         ( USBTX, USBRX );                                             // tx, rx

DigitalOut  myled   ( LED1 );
Ticker      newAction;


//@brief Constants.


//@brief Variables.
volatile uint32_t myState;                                                      //   State that indicates when to perform a new sample    


//@brief   FUNCTION PROTOTYPES
void    changeDATA     ( void );


//@brief FUNCTION FOR APPLICATION MAIN ENTRY.
int main()
{
    LPS25HB::LPS25HB_status_t aux;
    LPS25HB::LPS25HB_data_t   myLPS25HB_Data;

    pc.baud ( 115200 );


    myled   =   1;
    wait(3);
    myled   =   0;

    // Perform a software reset  
    aux  =   myLPS25HB.LPS25HB_SetSoftwareReset ();

    do {
        aux  =   myLPS25HB.LPS25HB_GetSoftwareReset ( &myLPS25HB_Data );        // Dangerous!!! The uC may get stuck here...
        // [WORKAROUND] Insert a counter
    } while ( myLPS25HB_Data.swreset == LPS25HB::CTRL_REG2_SWRESET_SW_RESET );

    // Reboot memory content  
    aux  =   myLPS25HB.LPS25HB_SetRebootMemoryContent ();

    do {
        aux  =   myLPS25HB.LPS25HB_GetRebootMemoryContent ( &myLPS25HB_Data );  // Dangerous!!! The uC may get stuck here...
        // [WORKAROUND] Insert a counter
    } while ( myLPS25HB_Data.boot == LPS25HB::CTRL_REG2_BOOT_REBOOT_MODE );

    // Set device in lOW-POWER mode  
    aux  =   myLPS25HB.LPS25HB_SetPowerMode ( LPS25HB::CTRL_REG1_PD_POWER_DOWN_MODE );

    // Get device ID  
    aux  =   myLPS25HB.LPS25HB_GetDeviceID ( &myLPS25HB_Data );
    pc.printf ( "Device ID: %x\r\n", myLPS25HB_Data.deviceID );

    // Set temperature resolution: 64 internal average 
    myLPS25HB_Data.avgt  =   LPS25HB::RES_CONF_AVGT_64;
    aux  =   myLPS25HB.LPS25HB_SetTemperatureResolution ( myLPS25HB_Data );

    // Set pressure resolution: 512 internal average 
    myLPS25HB_Data.avgp  =   LPS25HB::RES_CONF_AVGP_512;
    aux  =   myLPS25HB.LPS25HB_SetPressureResolution ( myLPS25HB_Data );

    // Set ODR: One-shot mode enabled  
    myLPS25HB_Data.odr   =   LPS25HB::CTRL_REG1_ODR_ONE_SHOT_MODE;
    aux  =   myLPS25HB.LPS25HB_SetOutputDataRate ( myLPS25HB_Data );

    // Interrupt generation disabled  
    aux  =   myLPS25HB.LPS25HB_SetInterruptGeneration ( LPS25HB::CTRL_REG1_DIFF_EN_ENABLED );

    // Block data update: output registers not updated until MSB and LSB have been read  
    aux  =   myLPS25HB.LPS25HB_SetBlockDataUpdate ( LPS25HB::CTRL_REG1_BDU_1 );

    // FIFO disabled 
    myLPS25HB_Data.fifo_en   =   LPS25HB::CTRL_REG2_FIFO_EN_DISABLED;
    aux  =   myLPS25HB.LPS25HB_SetFIFOEnable ( myLPS25HB_Data );

    // Autozero: Normal mode 
    myLPS25HB_Data.autozero  =   LPS25HB::CTRL_REG2_AUTOZERO_NORMAL_MODE;
    aux  =   myLPS25HB.LPS25HB_SetAutozero ( myLPS25HB_Data );

    // Set device in ACTIVE mode  
    aux  =   myLPS25HB.LPS25HB_SetPowerMode ( LPS25HB::CTRL_REG1_PD_ACTIVE_MODE );


    myState  =   0UL;                                                           // Reset the variable
    newAction.attach( &changeDATA, 1U );                                        // the address of the function to be attached ( changeDATA ) and the interval ( 1s )


    // Let the callbacks take care of everything
    while(1) {
        sleep();

        if ( myState == 1UL ) {
            myled = 1U;

            // Trigger to get a new data value  
            aux  =   myLPS25HB.LPS25HB_SetOneShot ();

            // Wait until the conversion is done  
            do {
                aux  =   myLPS25HB.LPS25HB_GetOneShot ( &myLPS25HB_Data );
            } while( myLPS25HB_Data.one_shot == LPS25HB::CTRL_REG2_ONE_SHOT_NEW_DATASET );                       // Dangerous!!! The uC may get stuck here...
            // [WORKAROUND] Insert a counter

            // Wait until there is a new data ( both pressure and temperature )  
            do {
                aux  =   myLPS25HB.LPS25HB_GetStatusRegister ( &myLPS25HB_Data );
            } while( ( myLPS25HB_Data.status_reg & ( LPS25HB::STATUS_REG_P_DA_MASK | LPS25HB::STATUS_REG_T_DA_MASK ) ) != ( LPS25HB::STATUS_REG_P_DA_NEW_DATA | LPS25HB::STATUS_REG_T_DA_NEW_DATA ) ); // Dangerous!!! The uC may get stuck here...
            // [WORKAROUND] Insert a counter

            // Get pressure  
            aux  =   myLPS25HB.LPS25HB_GetPressure ( &myLPS25HB_Data );

            // Get temperature 
            aux  =   myLPS25HB.LPS25HB_GetTemperature ( &myLPS25HB_Data );

            // Send data through the UART    
            pc.printf ( "T: %0.1f C, P: %0.1f mbar\r\n", myLPS25HB_Data.temperature, myLPS25HB_Data.pressure );


            // Reset the variables   
            myState  =   0UL;
            myled    =   0U;
        }
    }
}


// @brief       changeDATA ( void  )
//
// @details     It changes myState variable
//
// @param[in]    N/A
//
// @param[out]   N/A.
//
// @return       N/A.
//
// @author      Manuel Caballero
// @date        31/May/2019
// @version     31/May/2019   The ORIGIN
// @pre         N/A
// @warning     N/A.
void changeDATA ( void )
{
    myState  =   1UL;
}
@endcode
*/


/*!
 Library for the LPS25HB MEMS pressure sensor: 260-1260 hPa absolute digital output barometer.
*/
class LPS25HB
{
public:
    /**
    * @brief   DEFAULT ADDRESSES
    */
    typedef enum {
        LPS25HB_ADDRESS_0     =   ( 0b1011100 << 1U ),       /*!<   I2C slave address byte, SDO/SA0 = GND        */
        LPS25HB_ADDRESS_1     =   ( 0b1011101 << 1U )        /*!<   I2C slave address byte, SDO/SA0 = VDD        */
    } LPS25HB_addresses_t;



    /**
      * @brief   REGISTERS
      */
    typedef enum {
        LPS25HB_REF_P_XL        =   0x08,           /*!<  Reference pressure registers                  */
        LPS25HB_REF_P_L         =   0x09,           /*!<  Reference pressure registers                  */
        LPS25HB_REF_P_H         =   0x0A,           /*!<  Reference pressure registers                  */
        LPS25HB_WHO_AM_I        =   0x0F,           /*!<  Who am I register                             */
        LPS25HB_RES_CONF        =   0x10,           /*!<  Resolution register                           */
        LPS25HB_CTRL_REG1       =   0x20,           /*!<  Control registers                             */
        LPS25HB_CTRL_REG2       =   0x21,           /*!<  Control registers                             */
        LPS25HB_CTRL_REG3       =   0x22,           /*!<  Control registers                             */
        LPS25HB_CTRL_REG4       =   0x23,           /*!<  Control registers                             */
        LPS25HB_INTERRUPT_CFG   =   0x24,           /*!<  Interrupt registers                           */
        LPS25HB_INT_SOURCE      =   0x25,           /*!<  Interrupt registers                           */
        LPS25HB_STATUS_REG      =   0x27,           /*!<  Status register                               */
        LPS25HB_PRESS_OUT_XL    =   0x28,           /*!<  Pressure output register                      */
        LPS25HB_PRESS_OUT_L     =   0x29,           /*!<  Pressure output register                      */
        LPS25HB_PRESS_OUT_H     =   0x2A,           /*!<  Pressure output register                      */
        LPS25HB_TEMP_OUT_L      =   0x2B,           /*!<  Temperature output registers                  */
        LPS25HB_TEMP_OUT_H      =   0x2C,           /*!<  Temperature output registers                  */
        LPS25HB_FIFO_CTRL       =   0x2E,           /*!<  FIFO configure registers                      */
        LPS25HB_FIFO_STATUS     =   0x2F,           /*!<  FIFO configure registers                      */
        LPS25HB_THS_P_L         =   0x30,           /*!<  Pressure threshold registers                  */
        LPS25HB_THS_P_H         =   0x31,           /*!<  Pressure threshold registers                  */
        LPS25HB_RPDS_L          =   0x39,           /*!<  Pressure offset registers                     */
        LPS25HB_RPDS_H          =   0x3A            /*!<  Pressure offset registers                     */
    } LPS25HB_registers_t;



    /**
      * @brief   REF_P_XL REGISTER. Reference pressure (LSB data) ( Default: 0x00 )
      */
    typedef enum {
        REF_P_XL_MASK         =   0xFF            /*!<  REF_P_XL mask                                 */
    } LPS25HB_rep_p_xl_t;


    /**
      * @brief   REF_P_L REGISTER. Reference pressure (middle part) ( Default: 0x00 )
      */
    typedef enum {
        REF_P_L_MASK          =   0xFF            /*!<  REF_P_L mask                                  */
    } LPS25HB_rep_p_l_t;


    /**
      * @brief   REF_P_H REGISTER. Reference pressure (MSB data) ( Default: 0x00 )
      */
    typedef enum {
        REF_P_H_MASK          =   0xFF            /*!<  REF_P_H mask                                  */
    } LPS25HB_rep_p_h_t;


    /**
      * @brief   WHO_AM_I REGISTER.
      */
    typedef enum {
        WHO_AM_I_MASK         =   0xFF,           /*!<  WHO_AM_I mask                                */
        WHO_AM_I_VALUE        =   0xBD            /*!<  WHO_AM_I: 0xBD                               */
    } LPS25HB_who_am_i_t;


    /**
      * @brief   RES_CONF REGISTER.
      */
    /* AVGT <3:2>
     *    NOTE: Temperature internal average configuration.
     */
    typedef enum {
        RES_CONF_AVGT_MASK    =   ( 0b11 << 2U ),   /*!<  AVGT mask                                    */
        RES_CONF_AVGT_8       =   ( 0b00 << 2U ),   /*!<  AVGT Nr. internal average  8                 */
        RES_CONF_AVGT_16      =   ( 0b01 << 2U ),   /*!<  AVGT Nr. internal average 16                 */
        RES_CONF_AVGT_32      =   ( 0b10 << 2U ),   /*!<  AVGT Nr. internal average 32                 */
        RES_CONF_AVGT_64      =   ( 0b11 << 2U )    /*!<  AVGT Nr. internal average 64     [ Default ] */
    } LPS25HB_res_conf_avgt_t;


    /* AVGP <1:0>
     *    NOTE: Pressure internal average configuration.
     */
    typedef enum {
        RES_CONF_AVGP_MASK    =   ( 0b11 << 0U ),   /*!<  AVGP mask                                    */
        RES_CONF_AVGP_8       =   ( 0b00 << 0U ),   /*!<  AVGP Nr. internal average   8                */
        RES_CONF_AVGP_32      =   ( 0b01 << 0U ),   /*!<  AVGP Nr. internal average  32                */
        RES_CONF_AVGP_128     =   ( 0b10 << 0U ),   /*!<  AVGP Nr. internal average 128                */
        RES_CONF_AVGP_512     =   ( 0b11 << 0U )    /*!<  AVGP Nr. internal average 512    [ Default ] */
    } LPS25HB_res_conf_avgp_t;


    /**
      * @brief   CTRL_REG1 REGISTER.
      */
    /* PD <7>
     *    NOTE: Power-down control.
     */
    typedef enum {
        CTRL_REG1_PD_MASK             =   ( 1U << 7U ),     /*!<  PD mask                                       */
        CTRL_REG1_PD_POWER_DOWN_MODE  =   ( 0U << 7U ),     /*!<  Power-down mode                   [ Default ] */
        CTRL_REG1_PD_ACTIVE_MODE      =   ( 1U << 7U )      /*!<  Active mode                                   */
    } LPS25HB_ctrl_reg1_pd_t;


    /* ODR <6:4>
     *    NOTE: Output data rate selection.
     */
    typedef enum {
        CTRL_REG1_ODR_MASK            =   ( 0b111 << 4U ),  /*!<  ODR mask                                      */
        CTRL_REG1_ODR_ONE_SHOT_MODE   =   ( 0b000 << 4U ),  /*!<  One- shot mode enabled            [ Default ] */
        CTRL_REG1_ODR_1_HZ            =   ( 0b001 << 4U ),  /*!<  ODR:  1 HZ                                    */
        CTRL_REG1_ODR_7_HZ            =   ( 0b010 << 4U ),  /*!<  ODR:  7 HZ                                    */
        CTRL_REG1_ODR_12_5_HZ         =   ( 0b011 << 4U ),  /*!<  ODR: 12.5 HZ                                  */
        CTRL_REG1_ODR_25_HZ           =   ( 0b100 << 4U )   /*!<  ODR: 25 HZ                                    */
    } LPS25HB_ctrl_reg1_odr_t;


    /* DIFF_EN <3>
     *    NOTE: Interrupt generation enable.
     */
    typedef enum {
        CTRL_REG1_DIFF_EN_MASK        =   ( 1U << 3U ),     /*!<  DIFF_EN mask                                  */
        CTRL_REG1_DIFF_EN_DISABLED    =   ( 0U << 3U ),     /*!<  Interrupt generation disabled     [ Default ] */
        CTRL_REG1_DIFF_EN_ENABLED     =   ( 1U << 3U )      /*!<  Interrupt generation enabled                  */
    } LPS25HB_ctrl_reg1_diff_en_t;


    /* BDU <2>
     *    NOTE: Block data update.
     */
    typedef enum {
        CTRL_REG1_BDU_MASK            =   ( 1U << 2U ),     /*!<  BDU mask                                      */
        CTRL_REG1_BDU_0               =   ( 0U << 2U ),     /*!<  Continuous update                 [ Default ] */
        CTRL_REG1_BDU_1               =   ( 1U << 2U )      /*!<  Not updated until MSB and LSB have been read  */
    } LPS25HB_ctrl_reg1_bdu_t;


    /* RESET_AZ <1>
     *    NOTE: Reset Autozero function.
     */
    typedef enum {
        CTRL_REG1_RESET_AZ_MASK                     =   ( 1U << 1U ), /*!<  RESET_AZ mask                                 */
        CTRL_REG1_RESET_AZ_NORMAL_MODE              =   ( 0U << 1U ), /*!<  Normal mode                       [ Default ] */
        CTRL_REG1_RESET_AZ_RESET_AUTOZERO_FUNCTION  =   ( 1U << 1U )  /*!<  Reset Autozero function                       */
    } LPS25HB_ctrl_reg1_reset_az_t;


    /* SIM <0>
     *    NOTE: SPI Serial Interface Mode selection.
     */
    typedef enum {
        CTRL_REG1_SIM_MASK              =   ( 1U << 0U ),   /*!<  SIM mask                                      */
        CTRL_REG1_SIM_4_WIRE_INTERFACE  =   ( 0U << 0U ),   /*!<  4-wire interface                  [ Default ] */
        CTRL_REG1_SIM_3_WIRE_INTERFACE  =   ( 1U << 0U )    /*!<  3-wire interface                              */
    } LPS25HB_ctrl_reg1_sim_t;


    /**
      * @brief   CTRL_REG2 REGISTER.
      */
    /* BOOT <7>
     *    NOTE: Reboot memory content. The bit is self-cleared when the BOOT is completed
     */
    typedef enum {
        CTRL_REG2_BOOT_MASK           =   ( 1U << 7U ),     /*!<  BOOT mask                                     */
        CTRL_REG2_BOOT_NORMAL_MODE    =   ( 0U << 7U ),     /*!<  Normal mode                       [ Default ] */
        CTRL_REG2_BOOT_REBOOT_MODE    =   ( 1U << 7U )      /*!<  Reboot memory content                         */
    } LPS25HB_ctrl_reg2_boot_t;


    /* FIFO_EN <6>
     *    NOTE: FIFO enable
     */
    typedef enum {
        CTRL_REG2_FIFO_EN_MASK        =   ( 1U << 6U ),     /*!<  FIFO_EN mask                                  */
        CTRL_REG2_FIFO_EN_DISABLED    =   ( 0U << 6U ),     /*!<  Disable                           [ Default ] */
        CTRL_REG2_FIFO_EN_ENABLED     =   ( 1U << 6U )      /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg2_fifo_en_t;


    /* STOP_ON_FTH <5>
     *    NOTE: Enable the FTH_FIFO bit in FIFO_STATUS (2Fh) for monitoring of FIFO level.
     */
    typedef enum {
        CTRL_REG2_STOP_ON_FTH_MASK      =   ( 1U << 5U ),   /*!<  STOP_ON_FTH mask                              */
        CTRL_REG2_STOP_ON_FTH_DISABLED  =   ( 0U << 5U ),   /*!<  Disable                           [ Default ] */
        CTRL_REG2_STOP_ON_FTH_ENABLED   =   ( 1U << 5U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg2_stop_on_fth_t;


    /* FIFO_MEAN_DEC <4>
     *    NOTE: Enable to decimate the output pressure to 1Hz with FIFO Mean mode.
     */
    typedef enum {
        CTRL_REG2_FIFO_MEAN_DEC_MASK      =   ( 1U << 4U ),   /*!<  FIFO_MEAN_DEC mask                            */
        CTRL_REG2_FIFO_MEAN_DEC_DISABLED  =   ( 0U << 4U ),   /*!<  Disable                           [ Default ] */
        CTRL_REG2_FIFO_MEAN_DEC_ENABLED   =   ( 1U << 4U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg2_fifo_mean_dec_t;


    /* I2C_DIS <3>
     *    NOTE: I2C interface enabled
     */
    typedef enum {
        CTRL_REG2_I2C_DIS_MASK        =   ( 1U << 3U ),     /*!<  I2C_DIS mask                                  */
        CTRL_REG2_I2C_DIS_ENABLED     =   ( 0U << 3U ),     /*!<  Enabled                           [ Default ] */
        CTRL_REG2_I2C_DIS_DISABLED    =   ( 1U << 3U )      /*!<  Disabled                                      */
    } LPS25HB_ctrl_reg2_i2c_dis_t;


    /* SWRESET <2>
     *    NOTE: Software reset. The bit is self-cleared when the reset is completed.
     */
    typedef enum {
        CTRL_REG2_SWRESET_MASK        =   ( 1U << 2U ),     /*!<  SWRESET mask                                  */
        CTRL_REG2_SWRESET_NORMAL_MODE =   ( 0U << 2U ),     /*!<  Normal mode                       [ Default ] */
        CTRL_REG2_SWRESET_SW_RESET    =   ( 1U << 2U )      /*!<  Software reset                                */
    } LPS25HB_ctrl_reg2_swreset_t;


    /* AUTOZERO <1>
     *    NOTE: Autozero enable.
     */
    typedef enum {
        CTRL_REG2_AUTOZERO_MASK             =   ( 1U << 1U ),     /*!<  AUTOZERO mask                                 */
        CTRL_REG2_AUTOZERO_NORMAL_MODE      =   ( 0U << 1U ),     /*!<  Normal mode                       [ Default ] */
        CTRL_REG2_AUTOZERO_AUTOZERO_ENABLED =   ( 1U << 1U )      /*!<  Autozero enabled                              */
    } LPS25HB_ctrl_reg2_autozero_t;


    /* ONE_SHOT <0>
     *    NOTE: One shot mode enable.
     */
    typedef enum {
        CTRL_REG2_ONE_SHOT_MASK         =   ( 1U << 0U ),   /*!<  ONE_SHOT mask                                 */
        CTRL_REG2_ONE_SHOTL_IDLE_MODE   =   ( 0U << 0U ),   /*!<  Idle mode                         [ Default ] */
        CTRL_REG2_ONE_SHOT_NEW_DATASET  =   ( 1U << 0U )    /*!<  A new dataset is acquired                     */
    } LPS25HB_ctrl_reg2_one_shot_t;


    /**
      * @brief   CTRL_REG3 REGISTER
      */
    /* INT_H_L <7>
     *    NOTE: Interrupt active high
     */
    typedef enum {
        CTRL_REG3_INT_H_L_MASK          =   ( 1U << 7U ),   /*!<  INT_H_L mask                                 */
        CTRL_REG3_INT_H_L_ACTIVE_HIGH   =   ( 0U << 7U ),   /*!<  active high                      [ Default ] */
        CTRL_REG3_INT_H_L_ACTIVE_LOW    =   ( 1U << 7U )    /*!<  active low                                   */
    } LPS25HB_ctrl_reg3_int_h_l_t;


    /* PP_OD <6>
     *    NOTE: Push-pull/open drain selection on interrupt pads.
     */
    typedef enum {
        CTRL_REG3_PP_OD_MASK                  =   ( 1U << 6U ), /*!<  PP_OD mask                                  */
        CTRL_REG3_PP_OD_PUSH_PULL             =   ( 0U << 6U ), /*!<  push-pull                       [ Default ] */
        CTRL_REG3_PP_OD_OPEN_DRAIN            =   ( 1U << 6U )  /*!<  open drain                                  */
    } LPS25HB_ctrl_reg3_pp_od_t;


    /* INT_S2 <1:0>
     *    NOTE: Data signal on INT_DRDY pin control bits.
     */
    typedef enum {
        CTRL_REG3_INT_S2_MASK                 =   ( 0b11 << 0U ), /*!<  INT_S2 mask                                 */
        CTRL_REG3_INT_S2_DATA_SIGNAL          =   ( 0b00 << 0U ), /*!<  Data signal                     [ Default ] */
        CTRL_REG3_INT_S2_PRESSURE_HIGH        =   ( 0b01 << 0U ), /*!<  Pressure high (P_high)                      */
        CTRL_REG3_INT_S2_PRESSURE_LOW         =   ( 0b10 << 0U ), /*!<  Pressure low (P_low)                        */
        CTRL_REG3_INT_S2_PRESSURE_LOW_OR_HIGH =   ( 0b11 << 0U )  /*!<  Pressure low OR high                        */
    } LPS25HB_ctrl_reg3_int_s2_t;


    /**
      * @brief   CTRL_REG4 REGISTER
      */
    /* F_EMPTY <3>
     *    NOTE: FIFO empty flag on INT_DRDY pin
     */
    typedef enum {
        CTRL_REG4_F_EMPTY_MASK          =   ( 1U << 3U ),   /*!<  F_EMPTY mask                                  */
        CTRL_REG4_F_EMPTY_DISABLED      =   ( 0U << 3U ),   /*!<  Disabled                          [ Default ] */
        CTRL_REG4_F_EMPTY_ENABLED       =   ( 1U << 3U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg4_f_empty_t;


    /* F_FTH <2>
     *    NOTE: FIFO threshold (watermark) status on INT_DRDY pin to indicate that FIFO is filled up to the threshold level
     */
    typedef enum {
        CTRL_REG4_F_FTH_MASK            =   ( 1U << 2U ),   /*!<  F_FTH mask                                    */
        CTRL_REG4_F_FTH_DISABLED        =   ( 0U << 2U ),   /*!<  Disabled                          [ Default ] */
        CTRL_REG4_F_FTH_ENABLED         =   ( 1U << 2U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg4_f_fth_t;


    /* F_OVR <1>
     *    NOTE: FIFO overrun interrupt on INT_DRDY pin to indicate that FIFO is full in FIFO mode or that an overrun occurred in Stream mode
     */
    typedef enum {
        CTRL_REG4_F_OVR_MASK            =   ( 1U << 1U ),   /*!<  F_OVR mask                                    */
        CTRL_REG4_F_OVR_DISABLED        =   ( 0U << 1U ),   /*!<  Disabled                          [ Default ] */
        CTRL_REG4_F_OVR_ENABLED         =   ( 1U << 1U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg4_f_ovr_t;


    /* DRDY <0>
     *    NOTE: Data-ready signal on INT_DRDY pin
     */
    typedef enum {
        CTRL_REG4_DRDY_MASK             =   ( 1U << 0U ),   /*!<  DRDY mask                                     */
        CTRL_REG4_DRDY_DISABLED         =   ( 0U << 0U ),   /*!<  Disabled                          [ Default ] */
        CTRL_REG4_DRDY_ENABLED          =   ( 1U << 0U )    /*!<  Enabled                                       */
    } LPS25HB_ctrl_reg4_drdy_t;


    /**
      * @brief   INTERRUPT_CFG REGISTER
      */
    /* LIR <2>
     *    NOTE: Latch interrupt request to the INT_SOURCE (25h) register
     */
    typedef enum {
        INTERRUPT_CFG_LIR_MASK          =   ( 1U << 2U ),   /*!<  LIR mask                                      */
        INTERRUPT_CFG_LIR_NOT_LATCHED   =   ( 0U << 2U ),   /*!<  Interrupt request not latched     [ Default ] */
        INTERRUPT_CFG_LIR_LATCHED       =   ( 1U << 2U )    /*!<  Interrupt request latched                     */
    } LPS25HB_interrupt_cfg_lir_t;


    /* PL_E <1>
     *    NOTE: Enable interrupt generation on differential pressure low event
     */
    typedef enum {
        INTERRUPT_CFG_PL_E_MASK         =   ( 1U << 1U ),   /*!<  PL_E mask                                     */
        INTERRUPT_CFG_PL_E_DISABLED     =   ( 0U << 1U ),   /*!<  Disable interrupt request         [ Default ] */
        INTERRUPT_CFG_PL_E_ENABLED      =   ( 1U << 1U )    /*!<  Enable interrupt request on measured differential pressure value lower than preset threshold      */
    } LPS25HB_interrupt_cfg_pl_e_t;


    /* PH_E <0>
     *    NOTE: Enable interrupt generation on differential pressure high event
     */
    typedef enum {
        INTERRUPT_CFG_PH_E_MASK         =   ( 1U << 0U ),   /*!<  PH_E mask                                     */
        INTERRUPT_CFG_PH_E_DISABLED     =   ( 0U << 0U ),   /*!<  Disable interrupt request         [ Default ] */
        INTERRUPT_CFG_PH_E_ENABLED      =   ( 1U << 0U )    /*!<  enable interrupt request on measured differential pressure value higher than preset threshold      */
    } LPS25HB_interrupt_cfg_ph_e_t;


    /**
      * @brief   INT_SOURCE REGISTER ( INT_SOURCE register is cleared by reading it )
      */
    /* IA <2>
     *    NOTE: Interrupt active
     */
    typedef enum {
        INT_SOURCE_IA_MASK                    =   ( 1U << 2U ),   /*!<  IA mask                                       */
        INT_SOURCE_IA_NO_INTERRUPT_GENERATED  =   ( 0U << 2U ),   /*!<  No interrupt has been generated               */
        INT_SOURCE_IA_INTERRUPT_GENERATED     =   ( 1U << 2U )    /*!<  One/more interrupt events have been generated */
    } LPS25HB_int_source_ia_t;


    /* PL <1>
     *    NOTE: Differential pressure Low
     */
    typedef enum {
        INT_SOURCE_PL_MASK                    =   ( 1U << 1U ),   /*!<  PL mask                                       */
        INT_SOURCE_PL_NO_INTERRUPT_GENERATED  =   ( 0U << 1U ),   /*!<  No interrupt has been generated               */
        INT_SOURCE_PL_EVENT_OCCURRED          =   ( 1U << 1U )    /*!<  Low differential pressure event has occurred  */
    } LPS25HB_int_source_pl_t;


    /* PH <0>
     *    NOTE: Differential pressure High
     */
    typedef enum {
        INT_SOURCE_PH_MASK                    =   ( 1U << 0U ),   /*!<  PH mask                                       */
        INT_SOURCE_PH_NO_INTERRUPT_GENERATED  =   ( 0U << 0U ),   /*!<  No interrupt has been generated               */
        INT_SOURCE_PH_EVENT_OCCURRED          =   ( 1U << 0U )    /*!<  High differential pressure event has occurred */
    } LPS25HB_int_source_ph_t;


    /**
      * @brief   STATUS_REG REGISTER
      */
    /* P_OR <5>
     *    NOTE: Pressure data overrun
     */
    typedef enum {
        STATUS_REG_P_OR_MASK            =   ( 1U << 5U ),   /*!<  P_OR mask                                               */
        STATUS_REG_P_OR_NO_DATA         =   ( 0U << 5U ),   /*!<  no overrun has occurred                                 */
        STATUS_REG_P_OR_NEW_DATA        =   ( 1U << 5U )    /*!<  new data for pressure has overwritten the previous one  */
    } LPS25HB_status_reg_p_or_t;


    /* T_OR <4>
     *    NOTE: Temperature data overrun
     */
    typedef enum {
        STATUS_REG_T_OR_MASK            =   ( 1U << 4U ),   /*!<  T_OR mask                                                 */
        STATUS_REG_T_OR_NO_DATA         =   ( 0U << 4U ),   /*!<  no overrun has occurred                                   */
        STATUS_REG_T_OR_NEW_DATA        =   ( 1U << 4U )    /*!<  new data for temperature has overwritten the previous one */
    } LPS25HB_status_reg_t_or_t;


    /* P_DA <1>
     *    NOTE: Pressure data available
     */
    typedef enum {
        STATUS_REG_P_DA_MASK            =   ( 1U << 1U ),   /*!<  P_DA mask                                               */
        STATUS_REG_P_DA_NO_AVAILABLE    =   ( 0U << 1U ),   /*!<  new data for pressure is not yet available              */
        STATUS_REG_P_DA_NEW_DATA        =   ( 1U << 1U )    /*!<  new data for pressure is available                      */
    } LPS25HB_status_reg_p_da_t;


    /* T_DA <0>
     *    NOTE: Temperature data available
     */
    typedef enum {
        STATUS_REG_T_DA_MASK            =   ( 1U << 0U ),   /*!<  T_DA mask                                               */
        STATUS_REG_T_DA_NO_AVAILABLE    =   ( 0U << 0U ),   /*!<  new data for temperature is not yet available           */
        STATUS_REG_T_DA_NEW_DATA        =   ( 1U << 0U )    /*!<  new data for temperature is available                   */
    } LPS25HB_status_reg_t_da_t;


    /**
      * @brief   FIFO_CTRL REGISTER
      */
    /* F_MODE <7:5>
     *    NOTE: FIFO mode selection
     */
    typedef enum {
        FIFO_CTRL_F_MODE_MASK                   =   ( 0b111 << 5U ),  /*!<  F_MODE mask                                           */
        FIFO_CTRL_F_MODE_BYPASS_MODE            =   ( 0b000 << 5U ),  /*!<  Bypass mode                               [ Default ] */
        FIFO_CTRL_F_MODE_FIFO_MODE              =   ( 0b001 << 5U ),  /*!<  FIDO mode                                             */
        FIFO_CTRL_F_MODE_STREAM_MOD             =   ( 0b010 << 5U ),  /*!<  Stream mode                                           */
        FIFO_CTRL_F_MODE_STREAM_TO_FIFO_MODE    =   ( 0b011 << 5U ),  /*!<  Stream-to-FIDO mode                                   */
        FIFO_CTRL_F_MODE_BYPASS_TO_STREAM_MODE  =   ( 0b100 << 5U ),  /*!<  Bypass-to-Stream mode                                 */
        FIFO_CTRL_F_MODE_FIFO_MEAN_MODE         =   ( 0b110 << 5U ),  /*!<  FIDO mean mode                                        */
        FIFO_CTRL_F_MODE_BYPASS_TO_FIFO_MODE    =   ( 0b111 << 5U )   /*!<  Bypass-to-FIDO mode                                   */
    } LPS25HB_fifo_ctrl_f_mode_t;


    /* WTM_POINT <4:0>
     *    NOTE: FIFO threshold (watermark) level selection
     */
    typedef enum {
        FIFO_CTRL_WTM_POINT_MASK                =   ( 0b11111 << 0U ),  /*!<  WTM_POINT mask                                        */
        FIFO_CTRL_WTM_POINT_2_SAMPLE_MOV_AVG    =   ( 0b00001 << 0U ),  /*!<  2-sample moving average                   [ Default ] */
        FIFO_CTRL_WTM_POINT_4_SAMPLE_MOV_AVG    =   ( 0b00011 << 0U ),  /*!<  4-sample moving average                               */
        FIFO_CTRL_WTM_POINT_8_SAMPLE_MOV_AVG    =   ( 0b00111 << 0U ),  /*!<  8-sample moving average                               */
        FIFO_CTRL_WTM_POINT_16_SAMPLE_MOV_AVG   =   ( 0b01111 << 0U ),  /*!<  16-sample moving average                              */
        FIFO_CTRL_WTM_POINT_32_SAMPLE_MOV_AVG   =   ( 0b11111 << 0U )   /*!<  32-sample moving average                              */
    } LPS25HB_fifo_ctrl_wtm_point_t;


    /**
      * @brief   FIFO_STATUS REGISTER
      */
    /* FTH_FIFO <7>
     *    NOTE: FIFO threshold status
     */
    typedef enum {
        FIFO_STATUS_FTH_FIFO_MASK       =   ( 1U << 7U ),   /*!<  FTH_FIFO mask                                           */
        FIFO_STATUS_FTH_FIFO_0          =   ( 0U << 7U ),   /*!<  FIFO filling is lower than FTH level                    */
        FIFO_STATUS_FTH_FIFO_1          =   ( 1U << 7U )    /*!<  FIFO filling is equal or higher than FTH level          */
    } LPS25HB_fifo_status_fth_fifo_t;


    /* OVR <6>
     *    NOTE: Overrun bit status
     */
    typedef enum {
        FIFO_STATUS_OVR_MASK            =   ( 1U << 6U ),   /*!<  OVR mask                                                */
        FIFO_STATUS_OVR_FIFO_NOT_FULL   =   ( 0U << 6U ),   /*!<  FIFO not full                                           */
        FIFO_STATUS_OVR_FIFO_FULL       =   ( 1U << 6U )    /*!<  FIFO is full and at least one sample in the FIFO has been overwritten */
    } LPS25HB_fifo_status_ovr_t;


    /* EMPTY_FIFO <5>
     *    NOTE: Empty FIFO bit status
     */
    typedef enum {
        FIFO_STATUS_EMPTY_FIFO_MASK       =   ( 1U << 5U ),   /*!<  EMPTY_FIFO mask                                       */
        FIFO_STATUS_EMPTY_FIFO_NOT_EMPTY  =   ( 0U << 5U ),   /*!<  FIFO not empty                                        */
        FIFO_STATUS_EMPTY_FIFO_EMPTY      =   ( 1U << 5U )    /*!<  FIFO empty                                            */
    } LPS25HB_fifo_status_empty_fifo_t;


    /* FSS <4:0>
     *    NOTE: FIFO stored data level
     */
    typedef enum {
        FIFO_STATUS_FSS_MASK              =   ( 0b11111 << 0U ) /*!<  FSS mask                                              */
    } LPS25HB_fifo_status_fss_t;




#ifndef LPS25HB_VECTOR_STRUCT_H
#define LPS25HB_VECTOR_STRUCT_H
    typedef struct {
        /* Output registers  */
        int32_t   rawReferencePressure;         /*!<  Raw reference pressure        */
        int32_t   rawPressure;                  /*!<  Raw pressure                  */
        int16_t   rawTemperature;               /*!<  Raw temperature               */

        float     pressure;                     /*!<  Pressure in mbar              */
        float     temperature;                  /*!<  Temperature in Celsius degree */

        /* Resolution  */
        LPS25HB_res_conf_avgt_t avgt;           /*!<  Temperature resolution        */
        LPS25HB_res_conf_avgp_t avgp;           /*!<  Pressure resolution           */

        /* Configuration  */
        LPS25HB_ctrl_reg1_odr_t      odr;       /*!<  Output data rate selection    */
        LPS25HB_ctrl_reg1_reset_az_t reset_az;  /*!<  Reset autozero function       */
        LPS25HB_ctrl_reg2_boot_t     boot;      /*!<  Reboot memory content         */
        LPS25HB_ctrl_reg2_fifo_en_t  fifo_en;   /*!<  FIFO enable                   */
        LPS25HB_ctrl_reg2_swreset_t  swreset;   /*!<  Software reset                */
        LPS25HB_ctrl_reg2_autozero_t autozero;  /*!<  Autozero enable               */
        LPS25HB_ctrl_reg2_one_shot_t one_shot;  /*!<  One-shot                      */

        /* INT_DRDY behaviour   */
        LPS25HB_ctrl_reg4_f_empty_t  f_empty;   /*!<  FIFO empty flag on INT_DRDY pin                   */
        LPS25HB_ctrl_reg4_f_fth_t    f_fth;     /*!<  FIFO threshold (watermark) status on INT_DRDY pin */
        LPS25HB_ctrl_reg4_f_ovr_t    f_ovr;     /*!<  FIFO overrun interrupt on INT_DRDY pin            */
        LPS25HB_ctrl_reg4_drdy_t     drdy;      /*!<  Data-ready signal on INT_DRDY pin                 */

        /* Interrupt configuration   */
        LPS25HB_interrupt_cfg_lir_t  lir;       /*!<  Latch interrupt request                                         */
        LPS25HB_interrupt_cfg_pl_e_t pl_e;      /*!<  Enable interrupt generation on differential pressure low event  */
        LPS25HB_interrupt_cfg_ph_e_t ph_e;      /*!<  Enable interrupt generation on differential pressure high event */

        /* Interrupt source  */
        uint8_t                      int_source;  /*!<  Interrupt source            */

        /* Status register  */
        uint8_t                      status_reg;  /*!<  Status register             */

        /* FIFO control  */
        LPS25HB_fifo_ctrl_f_mode_t    f_mode;     /*!<  FIFO mode selection                         */
        LPS25HB_fifo_ctrl_wtm_point_t wtm_point;  /*!<  FIFO threshold (watermark) level selection  */

        /* FIFO status   */
        uint8_t                       FIFOstatus; /*!<  FIFO status register        */

        /* Pressure threshold  */
        uint16_t                      ths_p;      /*!<  Threshold value for pressure interrupt generation */

        /* Pressure offset  */
        uint16_t                      rpds;       /*!<  Pressure offset             */

        /* Device identification   */
        uint8_t                       deviceID;   /*!<  Device ID                   */
    } LPS25HB_data_t;
#endif


    /**
      * @brief   INTERNAL CONSTANTS
      */
    typedef enum {
        LPS25HB_SUCCESS    =      0U,
        LPS25HB_FAILURE    =      1U,
        I2C_SUCCESS       =       0U               /*!<   I2C communication was fine        */
    } LPS25HB_status_t;




    /** Create an LPS25HB object connected to the specified I2C pins.
      *
      * @param sda     I2C data pin
      * @param scl     I2C clock pin
      * @param addr    I2C slave address
      * @param freq    I2C frequency
      */
    LPS25HB ( PinName sda, PinName scl, uint32_t addr, uint32_t freq );

    /** Delete LPS25HB object.
     */
    ~LPS25HB();

    /** It gets raw reference pressure.
    */
    LPS25HB_status_t LPS25HB_GetReferencePressure     ( LPS25HB_data_t* myREFL                );

    /** It sets raw reference pressure.
      */
    LPS25HB_status_t LPS25HB_SetReferencePressure     ( LPS25HB_data_t myREFL                 );

    /** It gets the device ID.
      */
    LPS25HB_status_t LPS25HB_GetDeviceID              ( LPS25HB_data_t* myID                  );

    /** It sets temperature resolution.
      */
    LPS25HB_status_t LPS25HB_SetTemperatureResolution ( LPS25HB_data_t myAVGT                 );

    /** It gets temperature resolution.
      */
    LPS25HB_status_t LPS25HB_GetTemperatureResolution ( LPS25HB_data_t* myAVGT                );

    /** It sets pressure resolution.
      */
    LPS25HB_status_t LPS25HB_SetPressureResolution    ( LPS25HB_data_t myAVGP                 );

    /** It gets pressure resolution.
      */
    LPS25HB_status_t LPS25HB_GetPressureResolution    ( LPS25HB_data_t* myAVGP                );

    /** It sets the power mode.
      */
    LPS25HB_status_t LPS25HB_SetPowerMode             ( LPS25HB_ctrl_reg1_pd_t myPD           );

    /** It sets the output data rate.
      */
    LPS25HB_status_t LPS25HB_SetOutputDataRate        ( LPS25HB_data_t myODR                  );

    /** It gets the output data rate.
      */
    LPS25HB_status_t LPS25HB_GetOutputDataRate        ( LPS25HB_data_t* myODR                 );

    /** It sets the interrupt generation enable.
      */
    LPS25HB_status_t LPS25HB_SetInterruptGeneration   ( LPS25HB_ctrl_reg1_diff_en_t myDIFF_EN );

    /** It sets the block data update.
      */
    LPS25HB_status_t LPS25HB_SetBlockDataUpdate       ( LPS25HB_ctrl_reg1_bdu_t myBDU         );

    /** It sets the reset autozero function.
      */
    LPS25HB_status_t LPS25HB_SetResetAutozero         ( void                                  );

    /** It gets the reset autozero function.
      */
    LPS25HB_status_t LPS25HB_GetResetAutozero         ( LPS25HB_data_t* myRESET_AZ            );

    /** It sets the reboot memory content.
      */
    LPS25HB_status_t LPS25HB_SetRebootMemoryContent   ( void                                  );

    /** It gets the reboot memory content.
      */
    LPS25HB_status_t LPS25HB_GetRebootMemoryContent   ( LPS25HB_data_t* myBOOT                );

    /** It sets the FIFO enable/disable.
      */
    LPS25HB_status_t LPS25HB_SetFIFOEnable            ( LPS25HB_data_t myFIFO_EN              );

    /** It gets the FIFO enable/disable.
      */
    LPS25HB_status_t LPS25HB_GetFIFOEnable            ( LPS25HB_data_t* myFIFO_EN             );

    /** It enables/disables the decimate the output pressure to 1Hz with FIFO Mean mode.
      */
    LPS25HB_status_t LPS25HB_SetFIFOMeanDec           ( LPS25HB_ctrl_reg2_fifo_mean_dec_t myFIFO_MEAN_DEC );

    /** It sets the software reset.
      */
    LPS25HB_status_t LPS25HB_SetSoftwareReset         ( void                                  );

    /** It gets the software reset flag value.
      */
    LPS25HB_status_t LPS25HB_GetSoftwareReset         ( LPS25HB_data_t* mySWRESET             );

    /** It sets the autozero enable.
      */
    LPS25HB_status_t LPS25HB_SetAutozero              ( LPS25HB_data_t myAUTOZERO             );

    /** It gets the autozero enable value.
      */
    LPS25HB_status_t LPS25HB_GetAutozero              ( LPS25HB_data_t* myAUTOZERO            );

    /** It sets the one-shot mode.
      */
    LPS25HB_status_t LPS25HB_SetOneShot               ( void                                  );

    /** It gets the one-shot mode flag.
      */
    LPS25HB_status_t LPS25HB_GetOneShot               ( LPS25HB_data_t* myONE_SHOT            );

    /** It sets the interrupt active mode.
      */
    LPS25HB_status_t LPS25HB_SetInterruptActiveMode   ( LPS25HB_ctrl_reg3_int_h_l_t myINT_H_L );

    /** It sets the Push-pull/open drain selection on interrupt pads.
      */
    LPS25HB_status_t LPS25HB_SetDrainSelectionMode    ( LPS25HB_ctrl_reg3_pp_od_t myPP_OD     );

    /** It sets the Data signal on INT_DRDY pin control bits.
      */
    LPS25HB_status_t LPS25HB_SetDataSignalOnPin       ( LPS25HB_ctrl_reg3_int_s2_t myINT_S    );

    /** It sets the INT_DRDY behaviour.
      */
    LPS25HB_status_t LPS25HB_SetINT_DRDY_Behaviour    ( LPS25HB_data_t myIntConfig            );

    /** It gets the INT_DRDY behaviour.
      */
    LPS25HB_status_t LPS25HB_GetINT_DRDY_Behaviour    ( LPS25HB_data_t* myIntConfig           );

    /** It sets the interrupt configuration register.
      */
    LPS25HB_status_t LPS25HB_SetInterruptConfiguration ( LPS25HB_data_t myIntConfig           );

    /** It gets the interrupt configuration register.
      */
    LPS25HB_status_t LPS25HB_GetInterruptConfiguration ( LPS25HB_data_t* myIntConfig          );

    /** It reads the interrupt source register.
      */
    LPS25HB_status_t LPS25HB_GetInterruptSource       ( LPS25HB_data_t* myIntSource           );

    /** It reads the status register.
      */
    LPS25HB_status_t LPS25HB_GetStatusRegister        ( LPS25HB_data_t* myStatusRegister      );

    /** It gets the raw pressure value.
      */
    LPS25HB_status_t LPS25HB_GetRawPressure           ( LPS25HB_data_t* myRawPressure         );

    /** It gets the raw temperature value.
      */
    LPS25HB_status_t LPS25HB_GetRawTemperature        ( LPS25HB_data_t* myRawTemperature      );

    /** It gets the FIFO mode selection.
      */
    LPS25HB_status_t LPS25HB_GetFIFO_Mode             ( LPS25HB_data_t* myFIFOmode            );

    /** It sets the FIFO mode selection.
      */
    LPS25HB_status_t LPS25HB_SetFIFO_Mode             ( LPS25HB_data_t myFIFOmode             );

    /** It gets the FIFO threshold (watermark) level selection.
      */
    LPS25HB_status_t LPS25HB_GetFIFO_Threshold        ( LPS25HB_data_t* myFIFOthreshold       );

    /** It sets the FIFO threshold (watermark) level selection.
      */
    LPS25HB_status_t LPS25HB_SetFIFO_Threshold        ( LPS25HB_data_t myFIFOthreshold        );

    /** It reads the FIFO status register.
      */
    LPS25HB_status_t LPS25HB_GetFIFO_Status           ( LPS25HB_data_t* myFIFOstatus          );

    /** It sets the FIFO threshold value.
      */
    LPS25HB_status_t LPS25HB_SetFIFO_ThresholdValue   ( LPS25HB_data_t myFIFOthresholdValue   );

    /** It gets the FIFO threshold value.
      */
    LPS25HB_status_t LPS25HB_GetFIFO_ThresholdValue   ( LPS25HB_data_t* myFIFOthresholdValue  );

    /** It sets the Pressure offset value.
      */
    LPS25HB_status_t LPS25HB_SetPressureOffset        ( LPS25HB_data_t myPressureOffset       );

    /** It gets the Pressure offset value.
      */
    LPS25HB_status_t LPS25HB_GetPressureOffset        ( LPS25HB_data_t* myPressureOffset      );

    /** It gets the current pressure in mbar.
      */
    LPS25HB_status_t LPS25HB_GetPressure              ( LPS25HB_data_t* myPressure            );

    /** It gets the current temperature in Celsius degrees.
      */
    LPS25HB_status_t LPS25HB_GetTemperature           ( LPS25HB_data_t* myTemperature         );



private:
    I2C      _i2c;
    uint32_t _LPS25HB_Addr;
};

#endif
