/**
 * @brief       LEM_HAIS.h
 * @details     Current transducer. For the electronic measurement of currents:
 *              DC, AC, pulsed..., with galvanic separation between the primary
 *              circuit and the secondary circuit.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     19/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef LEM_HAIS_H
#define LEM_HAIS_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "LEM_HAIS.h"


LEM_HAIS  myCurrentTransducer      ( p20 );
Serial pc                          ( USBTX, USBRX );

Ticker newReading;
DigitalOut myled1            ( LED1 );
DigitalOut myled2            ( LED2 );
AnalogIn   myVref            ( p19 );
AnalogIn   myINPUT           ( p20 );


LEM_HAIS::LEM_HAIS_parameters_t          myParameters;
LEM_HAIS::LEM_HAIS_voltage_t             myVoltages;


void readDATA ( void )
{
    LEM_HAIS::LEM_HAIS_current_t      myCurrent;

    myled2    =  1;

    myVoltages          =    myCurrentTransducer.LEM_HAIS_GetVoltage       ();
    myCurrent           =    myCurrentTransducer.LEM_HAIS_CalculateCurrent ( myVoltages, myParameters, LEM_HAIS::FILTER_DISABLED );

    pc.printf( "IP: %0.5f A\r\n", myCurrent.Current );

    myled2    =  0;
}


int main()
{
    uint32_t    i               =   0;

    // CONFIGURATION. The parameters of the system
    // myParameters.lem_hais_reference_voltage  =   2.5;                        // ( uncomment ) Mandatory if calibration is NOT used!
    myParameters.voltage_divider             =   2.0;                           // Resistor ( both with the same value ) divider at the Sensor ( LEM-HAIS ) Vout
    myParameters.adc_reference_voltage       =   3.3;                           // ADC microcontroller ~ 3.3V
    myParameters.lem_hais_ipm                =   150.0;                         // HAIS 150-P

    pc.baud ( 115200 );


    // [ OPTIONAL ] CALIBRATION. It calculates the offset to calibrate the future measurements.
    //              It reads the Vref from the device.
    myled1    =  1;
    for ( i = 0; i < 10; i++ ) {
        myParameters.lem_hais_reference_voltage +=   myVref.read();
        wait ( 0.25 );
    }

    myParameters.lem_hais_reference_voltage  /=  10.0;
    myParameters.lem_hais_reference_voltage  *=  myParameters.adc_reference_voltage;

    // It reads OUPUT from the device. NOTE: This MUST be done at 0A current!!!
    myParameters.lem_hais_offset_voltage    =   myCurrentTransducer.LEM_HAIS_SetAutoOffset ( myParameters );
    myled1    =  0;

    pc.printf( "Vref: %0.5f V Voff: %0.5f V\r\n", myParameters.lem_hais_reference_voltage, myParameters.lem_hais_offset_voltage );
    // CALIBRATION ends here


    newReading.attach( &readDATA, 1.5 );                                        // the address of the function to be attached ( readDATA ) and the interval ( 1.5s )


    // Let the callbacks take care of everything
    while(1) {
        sleep();
    }
}

*/


/*!
 Library for the LEM_HAIS Current transducer.
*/
class LEM_HAIS
{
public:
#define CALIBRATION_AVERAGE  10                                                 // Change it if you wish to calculate the offset with less measurements
#define SAMPLE_DATA          1000                                               // 1 sample every 1ms -> 1000 total samples in total ( = 1s )

    typedef enum {
        FILTER_ENABLED   =   1,
        FILTER_DISABLED  =   0
    } LEM_HAIS_filter_status_t;




#ifndef LEM_HAIS_DATA_STRUCT_H
#define LEM_HAIS_DATA_STRUCT_H
    typedef struct {
        float Current;
    } LEM_HAIS_current_t;

    typedef struct {
        float OUTPUT_Voltage[SAMPLE_DATA];
    } LEM_HAIS_voltage_t;

    typedef struct {
        float lem_hais_offset_voltage;
        float lem_hais_reference_voltage;
        float voltage_divider;
        float adc_reference_voltage;
        float lem_hais_ipm;
    } LEM_HAIS_parameters_t;
#endif




    /** Create an LEM_HAIS object connected to the specified pins.
      *
      * @param OUTPUT   Vout from the device
      */
    LEM_HAIS ( PinName OUTPUT );

    /** Delete LEM_HAIS object.
     */
    ~LEM_HAIS();

    /** It gets the voltage.
     */
    LEM_HAIS_voltage_t  LEM_HAIS_GetVoltage          ( void );

    /** It calculates the offset automatically ( at 0A current ).
     */
    float               LEM_HAIS_SetAutoOffset       ( LEM_HAIS_parameters_t myParameters );

    /** It calculates the current.
     */
    LEM_HAIS_current_t  LEM_HAIS_CalculateCurrent    ( LEM_HAIS_voltage_t myVoltages, LEM_HAIS_parameters_t myParameters, LEM_HAIS_filter_status_t myFilter = FILTER_DISABLED );



private:
    AnalogIn     _OUTPUT;
};

#endif
