/**
 * @brief       LEM_HAIS.h
 * @details     Current transducer. For the electronic measurement of currents: 
 *              DC, AC, pulsed..., with galvanic separation between the primary
 *              circuit and the secondary circuit.
 *              Function file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     19/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */

#include "LEM_HAIS.h"

LEM_HAIS::LEM_HAIS ( PinName OUTPUT )
    : _OUTPUT               ( OUTPUT )
{
}

LEM_HAIS::~LEM_HAIS()
{
}



/**
 * @brief       LEM_HAIS_GetVoltage   ( void )
 *
 * @details     It performs a new voltage measurement.
 *
 * @param[in]    NaN.
 *
 * @param[out]   myAuxVoltage:   All the samples.
 *
 *
 * @return       The actual voltage.
 *
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     27/September/2017   RMS calculations included.
 *              19/September/2017   The ORIGIN
 * @pre         If SAMPLE_DATA = 1000, this funcion will last for 1000 * 1ms 
 *              = 1 second.
 * @warning     NaN.
 */
LEM_HAIS::LEM_HAIS_voltage_t  LEM_HAIS::LEM_HAIS_GetVoltage ( void )
{
    uint32_t     i   =   0;
    
    LEM_HAIS_voltage_t myAuxVoltage;
    

    for ( i = 0; i < SAMPLE_DATA; i++ )
    {       
        myAuxVoltage.OUTPUT_Voltage[i]     =   _OUTPUT.read();
        
        wait_ms ( 1 );                                                        
    }
    

    return   myAuxVoltage;
}



/**
 * @brief       LEM_HAIS_CalculateCurrent   ( Vector_LEM_HAIS_voltage_t , LEM_HAIS_parameters_t , LEM_HAIS_filter_status_t )
 *
 * @details     It calculates the actual current.
 *
 * @param[in]    myVoltages:    Both voltages, OUTPUT and Vref voltages.
 * @param[in]    myParameters:  Parameters are necessary to calculate the
 *                              result.
 * @param[in]    myFilter:      If a low pass filter is enabled/disabled.
 *
 * @param[out]   NaN.
 *
 *
 * @return       The calculated current.
 *
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     10/October/2017     New parameters are sent to this function to
 *                                  process the data in the right way.
 *              19/September/2017   The ORIGIN
 * @pre         LEM_HAIS_GetVoltage function MUST be called first.
 * @warning     NaN.
 */
LEM_HAIS::LEM_HAIS_current_t  LEM_HAIS::LEM_HAIS_CalculateCurrent ( LEM_HAIS_voltage_t myVoltages, LEM_HAIS_parameters_t myParameters, LEM_HAIS_filter_status_t myFilter )
{
    LEM_HAIS_current_t myAuxCurrent;
    
    float    myAuxSQI               =  0;
    float    myAuxVol               =  0;
    float    myI_filtered           =  0;
    float    myI_Previousfiltered   =  0;
    uint32_t i                      =  0;

    
    
    // Check if we want to use a low pass filter
    if ( myFilter    ==  FILTER_ENABLED )
    { 
        myI_Previousfiltered =   myParameters.voltage_divider * myParameters.adc_reference_voltage * myVoltages.OUTPUT_Voltage[0];
        i                    =   1;
    }
    else
        i                    =   0;
        
    
    // Calculate the RMS current 
    for ( ; i < SAMPLE_DATA; i++ )
    {                              
        myAuxVol     =   myParameters.voltage_divider * myParameters.adc_reference_voltage * myVoltages.OUTPUT_Voltage[i];
        
        if ( myFilter    ==  FILTER_ENABLED )
        {
            myI_filtered    =   ( ( 1 - 0.5 ) * myAuxVol ) + ( 0.5 * myI_Previousfiltered );
            
            myI_Previousfiltered = myI_filtered;
            
            myAuxSQI     =   ( 8.0 / 5.0 ) * ( ( myI_filtered + myParameters.lem_hais_offset_voltage ) - myParameters.lem_hais_reference_voltage ) * myParameters.lem_hais_ipm;
        }
        else  
            myAuxSQI     =   ( 8.0 / 5.0 ) * ( ( myAuxVol + myParameters.lem_hais_offset_voltage ) - myParameters.lem_hais_reference_voltage ) * myParameters.lem_hais_ipm;  
        
        
        myAuxSQI    *=   myAuxSQI;
        
        
        myAuxCurrent.Current      +=   myAuxSQI;                                                        
    }
    
    
    if ( myFilter    ==  FILTER_ENABLED )
        i    =   1;
    else
        i    =   0;
        
    
    myAuxCurrent.Current    /=   ( float )( SAMPLE_DATA - i );
    myAuxCurrent.Current     =   sqrt( myAuxCurrent.Current );
    
    

    return   myAuxCurrent;
}



/**
 * @brief       LEM_HAIS_SetAutoOffset ( LEM_HAIS_parameters_t )
 *
 * @details     It calculates the offset automatically ( at 0A current ).
 *
 * @param[in]    myParameters:  voltage_divider, adc_reference_voltage. These 
 *                              parameters are necessary to calculate the
 *                              result. 
 *
 * @param[out]   NaN.
 *
 *
 * @return       The actual offset volatge.
 *
 *
 * @author      Manuel Caballero
 * @date        19/September/2017
 * @version     10/October/2017     New parameters are sent to this function to
 *                                  process the data in the right way.
 *              19/September/2017   The ORIGIN
 * @pre         If CALIBRATION_AVERAGE = 10, this function will last for 
 *              10 * 0.25 = 2.5 seconds.
 * @warning     This test has to be perfomed at 0A ( no current through
 *              the sensor at all ).
 */
float  LEM_HAIS::LEM_HAIS_SetAutoOffset ( LEM_HAIS_parameters_t myParameters )
{
    uint32_t i          =   0;
    float    myVoltage  =   0;
    
    
    // Calculate the average, get a new measurement every 0.25s
    for ( i = 0; i < CALIBRATION_AVERAGE; i++ ){
        myVoltage   +=   _OUTPUT.read();
        wait ( 0.25 );
    }
    
    myVoltage   /=   ( float )CALIBRATION_AVERAGE;
    
    
    // Store the offset
    myParameters.lem_hais_offset_voltage   =   myParameters.lem_hais_reference_voltage - ( myParameters.voltage_divider * myVoltage * myParameters.adc_reference_voltage );



    return   myParameters.lem_hais_offset_voltage;
}
