/**
 * @brief       HX711.h
 * @details     24-Bit Analog-to-Digital Converter (ADC) for Weigh Scales.
 *              Header file.
 *
 *
 * @return      NA
 *
 * @author      Manuel Caballero
 * @date        11/September/2017
 * @version     11/September/2017    The ORIGIN
 * @pre         NaN.
 * @warning     NaN
 * @pre         This code belongs to Nimbus Centre ( http://www.nimbus.cit.ie ).
 */
#ifndef HX711_H
#define HX711_H

#include "mbed.h"


/**
    Example:

#include "mbed.h"
#include "HX711.h"

HX711  myWeightSensor        ( p5, p6 );
Serial pc                    ( USBTX, USBRX );

Ticker newReading;
DigitalOut myled1            ( LED1 );
DigitalOut myled2            ( LED2 );
DigitalOut myled3            ( LED3 );
DigitalOut myled4            ( LED4 );

HX711::HX711_status_t        aux;
HX711::Vector_count_t        myData;
HX711::Vector_mass_t         myCalculatedMass;
HX711::Vector_voltage_t      myCalculatedVoltage;


void readDATA ( void )
{
    myled4    =  1;

    aux                 =    myWeightSensor.HX711_ReadRawData       ( HX711::CHANNEL_A_GAIN_128, &myData, 4 );
    myCalculatedMass    =    myWeightSensor.HX711_CalculateMass     ( &myData, 1.0, HX711::HX711_SCALE_kg );
    myCalculatedVoltage =    myWeightSensor.HX711_CalculateVoltage  ( &myData, 5.0 );

    pc.printf( "Raw Data: %ld  Mass: %0.5f kg Voltage: %0.5f mV\r\n", (uint32_t)myData.myRawValue, myCalculatedMass.myMass, 1000*myCalculatedVoltage.myVoltage );

    myled4    =  0;
}


int main()
{
    pc.baud ( 115200 );


    // Reset and wake the device up
    aux = myWeightSensor.HX711_PowerDown    ();
    aux = myWeightSensor.HX711_Reset        ();
    wait(1);


    // CALIBRATION time start!
    // 1. REMOVE THE MASS ON THE LOAD CELL ( ALL LEDs OFF ). Read data without any mass on the load cell
    aux = myWeightSensor.HX711_ReadData_WithoutMass ( HX711::CHANNEL_A_GAIN_128, &myData, 4 );

    myled1   =   1;
    wait(3);


    // 2. PUT A KNOWN MASS ON THE LOAD CELL ( JUST LED1 ON ). Read data with an user-specified calibration mass
    aux = myWeightSensor.HX711_ReadData_WithCalibratedMass ( HX711::CHANNEL_A_GAIN_128, &myData, 4 );
    // CALIBRATION time end!


    // [ OPTIONAL ] REMOVE THE MASS ON THE LOAD CELL ( JUST LED2 ON ). Read the device without any mass to calculate the tare weight for 5 seconds
    myled1   =   0;
    myled2   =   1;
    wait(3);
    myled2   =   0;

    // Calculating the tare weight ( JUST LED3 ON )
    myled3   =   1;
    aux = myWeightSensor.HX711_SetAutoTare ( HX711::CHANNEL_A_GAIN_128, 1.0, HX711::HX711_SCALE_kg, &myData, 5 );
    myled3   =   0;


    newReading.attach( &readDATA, 0.5 );                                        // the address of the function to be attached ( readDATA ) and the interval ( 0.5s ) ( JUST LED4 BLINKING )

    // Let the callbacks take care of everything
    while(1)
    {
        sleep();
    }
}

*/


/*!
 Library for the HX711 24-Bit Analog-to-Digital Converter (ADC) for Weigh Scales.
*/
class HX711
{
public:
    /**
      * @brief   CHANNELS & GAIN
      */
    typedef enum {
        CHANNEL_A_GAIN_128      =   0,              /*!<  Channel A 128 Gain.                                               */
        CHANNEL_B_GAIN_32       =   1,              /*!<  Channel B 32 Gain.                                                */
        CHANNEL_A_GAIN_64       =   2               /*!<  Channel A 64 Gain.                                                */
    } HX711_channel_gain_t;



    /**
      * @brief   READY/BUSY DATA
      */
    typedef enum {
        HX711_DATA_BUSY         =   0,              /*!<  HX711 data is NOT ready to be read.                                */
        HX711_DATA_READY        =   1               /*!<  HX711 data is ready to be read.                                    */
    } HX711_data_output_status_t;



    /**
      * @brief   SCALE
      */
    typedef enum {
        HX711_SCALE_kg          =   0,              /*!<  HX711 Scale in kg.                                                 */
        HX711_SCALE_g           =   1,              /*!<  HX711 Scale in  g.                                                 */
        HX711_SCALE_mg          =   2,              /*!<  HX711 Scale in mg.                                                 */
        HX711_SCALE_ug          =   3               /*!<  HX711 Scale in ug.                                                 */
    } HX711_scale_t;




#ifndef VECTOR_STRUCT_H
#define VECTOR_STRUCT_H
    typedef struct {
        float myRawValue_WithCalibratedMass;
        float myRawValue_WithoutCalibratedMass;
        float myRawValue_TareWeight;
        uint32_t myRawValue;
    } Vector_count_t;

    typedef struct {
        float myMass;
    } Vector_mass_t;

    typedef struct {
        float myVoltage;
    } Vector_voltage_t;
#endif



    /**
      * @brief   INTERNAL CONSTANTS
      */
#define HX711_PIN_HIGH           0x01               /*!<   Pin 'HIGH'                                                       */
#define HX711_PIN_LOW            0x00               /*!<   Pin 'LOW'                                                        */

    typedef enum {
        HX711_SUCCESS     =       0,
        HX711_FAILURE     =       1,
    } HX711_status_t;




    /** Create an HX711 object connected to the specified pins.
      *
      * @param pd_sck           HX711 Power down control (high active) and serial clock input
      * @param dout             HX711 Serial data output
      */
    HX711 ( PinName PD_SCK, PinName DOUT );

    /** Delete HX711 object.
     */
    ~HX711();

    /** It performs an internal reset.
     */
    HX711_status_t  HX711_Reset                         ( void );

    /** It puts the device into power-down mode.
     */
    HX711_status_t  HX711_PowerDown                     ( void );

    /** It sets both the channel and the gain for the next measurement.
     */
    HX711_status_t  HX711_SetChannelAndGain             ( HX711_channel_gain_t myChannel_Gain );

    /** It gets both the channel and the gain for the current measurement.
     */
    HX711_channel_gain_t  HX711_GetChannelAndGain       ( void );

    /** It reads raw data from the device.
     */
    HX711_status_t  HX711_ReadRawData                   ( HX711_channel_gain_t myChannel_Gain, Vector_count_t* myNewRawData, uint32_t myAverage );

    /** It reads raw data with an user-specified calibrated mass.
     */
    HX711_status_t  HX711_ReadData_WithCalibratedMass   ( HX711_channel_gain_t myChannel_Gain, Vector_count_t* myNewRawData, uint32_t myAverage );

    /** It reads raw data without any mass.
     */
    HX711_status_t  HX711_ReadData_WithoutMass          ( HX711_channel_gain_t myChannel_Gain, Vector_count_t* myNewRawData, uint32_t myAverage );

    /** It reads raw data without any mass after the system is calibrated.
     */
    HX711_status_t  HX711_SetAutoTare                   ( HX711_channel_gain_t myChannel_Gain, float myCalibratedMass, HX711_scale_t myScaleCalibratedMass, Vector_count_t* myNewRawData, float myTime );

    /** It sets a tare weight manually.
     */
    Vector_count_t  HX711_SetManualTare                 ( float myTareWeight );

    /** It calculates scaled data.
     */
    Vector_mass_t  HX711_CalculateMass                  ( Vector_count_t* myNewRawData, float myCalibratedMass, HX711_scale_t myScaleCalibratedMass );

    /** It calculates voltage data.
     */
    Vector_voltage_t  HX711_CalculateVoltage            ( Vector_count_t* myNewRawData, float myVoltageReference );




private:
    DigitalOut              _PD_SCK;
    DigitalIn               _DOUT;
    HX711_channel_gain_t    _HX711_CHANNEL_GAIN;
    HX711_scale_t           _HX711_SCALE;
    float                   _HX711_USER_CALIBATED_MASS;
};

#endif
